/**
 * Copyright (c) Facebook, Inc. and its affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 *
 * @format
 */
"use strict";

const HermesParser = require("hermes-parser");

const nullthrows = require("nullthrows");

const { parseSync, transformFromAstSync } = require("@babel/core");

const { generateFunctionMap } = require("metro-source-map");

function transform({ filename, options, plugins, src }) {
  const OLD_BABEL_ENV = process.env.BABEL_ENV;
  process.env.BABEL_ENV = options.dev
    ? "development"
    : process.env.BABEL_ENV || "production";

  try {
    const babelConfig = {
      caller: {
        name: "metro",
        bundler: "metro",
        platform: options.platform
      },
      ast: true,
      babelrc: options.enableBabelRCLookup,
      code: false,
      highlightCode: true,
      filename,
      plugins,
      sourceType: "module"
    };
    const sourceAst = options.hermesParser
      ? HermesParser.parse(src, {
          babel: true,
          sourceType: babelConfig.sourceType
        })
      : parseSync(src, babelConfig);
    const { ast } = transformFromAstSync(sourceAst, src, babelConfig);
    const functionMap = generateFunctionMap(sourceAst, {
      filename
    });
    return {
      ast: nullthrows(ast),
      functionMap
    };
  } finally {
    if (OLD_BABEL_ENV) {
      process.env.BABEL_ENV = OLD_BABEL_ENV;
    }
  }
}

module.exports = {
  transform
};
