"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Handles async tasks.
 */
class AsyncTaskManager {
    constructor() {
        this.runningTasks = {};
        this.runningTimers = [];
        this.callbacks = [];
    }
    /**
     * Returns a promise that is fulfilled when async tasks are complete.
     * This method is not part of the HTML standard.
     *
     * @returns Promise.
     */
    whenComplete() {
        return new Promise((resolve) => {
            this.callbacks.push(resolve);
            this.endTask(null);
        });
    }
    /**
     * Ends all tasks.
     *
     * @param [error] Error.
     */
    cancelAll() {
        this.endAll(true);
    }
    /**
     * Starts a timer.
     *
     * @param timerID Timer ID.
     */
    startTimer(timerID) {
        this.runningTimers.push(timerID);
    }
    /**
     * Ends a timer.
     *
     * @param timerID Timer ID.
     */
    endTimer(timerID) {
        const index = this.runningTimers.indexOf(timerID);
        if (index !== -1) {
            this.runningTimers.splice(index, 1);
        }
        if (!Object.keys(this.runningTasks).length && !this.runningTimers.length) {
            this.endAll();
        }
    }
    /**
     * Starts an async task.
     *
     * @param abortHandler Abort handler.
     * @returns Task ID.
     */
    startTask(abortHandler) {
        const taskID = this.newTaskID();
        this.runningTasks[taskID] = abortHandler ? abortHandler : () => { };
        return taskID;
    }
    /**
     * Ends an async task.
     *
     * @param taskID Task ID.
     */
    endTask(taskID) {
        if (this.runningTasks[taskID]) {
            delete this.runningTasks[taskID];
        }
        if (!Object.keys(this.runningTasks).length && !this.runningTimers.length) {
            this.endAll();
        }
    }
    /**
     * Returns the amount of running tasks.
     *
     * @returns Count.
     */
    getTaskCount() {
        return Object.keys(this.runningTasks).length;
    }
    /**
     * Returns a new task ID.
     *
     * @returns Task ID.
     */
    newTaskID() {
        this.constructor.taskID++;
        return this.constructor.taskID;
    }
    /**
     * Ends all tasks.
     *
     * @param [canceled] Canceled.
     */
    endAll(canceled) {
        const runningTimers = this.runningTimers;
        const runningTasks = this.runningTasks;
        this.runningTasks = {};
        this.runningTimers = [];
        for (const timer of runningTimers) {
            global.clearTimeout(timer);
        }
        for (const key of Object.keys(runningTasks)) {
            runningTasks[key]();
        }
        if (this.callbacks.length) {
            if (canceled) {
                const callbacks = this.callbacks;
                this.callbacks = [];
                for (const callback of callbacks) {
                    callback();
                }
            }
            else {
                const timerID = global.setTimeout(() => {
                    if (!Object.keys(this.runningTasks).length && this.runningTimers.length === 1) {
                        const callbacks = this.callbacks;
                        this.callbacks = [];
                        this.runningTimers = [];
                        for (const callback of callbacks) {
                            callback();
                        }
                    }
                    else {
                        this.endTimer(timerID);
                    }
                }, 10);
                this.startTimer(timerID);
            }
        }
    }
}
AsyncTaskManager.taskID = 0;
exports.default = AsyncTaskManager;
//# sourceMappingURL=AsyncTaskManager.cjs.map