import { middleware } from "../events/Middleware";
import { Entity, PrimaryKeyOf } from "../types";
import { BlinkKey } from "./createDB";
import { Table } from "./createTable";
import { internalInsertMany } from "./insertMany";
import { internalUpdateMany } from "./updateMany";

/**
 * For each given entity of `entities`, either updates the entity if it exists,
 * or inserts it into the table if it doesn't.
 *
 * @returns the primary key of the inserted/updated entities,
 * in the same order as the items.
 *
 * @example
 * const db = createDB();
 * const userTable = createTable<User>(db, "users")();
 * const aliceId = await insert(userTable, { id: uuid(), name: 'Alice', age: 15 });
 * // Increase the age of Alice & insert a new user
 * await upsertMany(userTable, [
 *   { id: aliceId, age: 16 },
 *   { id: uuid(), age: 45 }
 * ]);
 */
export function upsertMany<T extends Entity<T>, P extends PrimaryKeyOf<T>>(
  table: Table<T, P>,
  entities: T[]
): Promise<T[P][]> {
  return Promise.resolve(
    middleware<T, P, "upsertMany">(
      table,
      { action: "upsertMany", params: [table, entities] },
      (table, entities) => internalUpsertMany(table, entities)
    )
  );
}

export async function internalUpsertMany<T extends Entity<T>, P extends PrimaryKeyOf<T>>(
  table: Table<T, P>,
  entities: T[]
): Promise<T[P][]> {
  // Split entities into items to create & items to update
  // reduces the number of outgoing events to 2
  const items: { entity: T; method: "insert" | "update" }[] = [];
  const primaryKeyProperty = table[BlinkKey].options.primary;
  for (const entity of entities) {
    const primaryKey = entity[primaryKeyProperty];
    const primaryKeyExists = table[BlinkKey].storage.primary.has(primaryKey);
    items.push({ entity, method: primaryKeyExists ? "update" : "insert" });
  }

  // Insert all items that need to be inserted, update all that need to be updated
  const itemsToInsert = items.filter((i) => i.method === "insert");
  const itemsToUpdate = items.filter((i) => i.method === "update");
  const insertIds = await internalInsertMany<T, P>(
    table,
    itemsToInsert.map((i) => i.entity)
  );
  const updateIds = await internalUpdateMany<T, P>(
    table,
    itemsToUpdate.map((i) => i.entity)
  );

  // Return the indexes in the correct order
  const ids: T[P][] = [];
  let insertIdsIndex = 0;
  let updateIdsIndex = 0;
  for (const item of items) {
    switch (item.method) {
      case "insert":
        ids.push(insertIds[insertIdsIndex]);
        insertIdsIndex++;
        break;
      case "update":
        ids.push(updateIds[updateIdsIndex]);
        updateIdsIndex++;
    }
  }

  return ids;
}
