/**
 * Valid types of entity properties.
 *
 * BlinkDB doesn't allow functions or Symbols as valid properties
 * in entities.
 */
export declare type EntityPropTypes = string | boolean | number | null | undefined | BigInt | Date | unknown[] | object;
/**
 * Types that can be ordered with >, >=, <, and <= operations.
 */
export declare type Ordinal = string | number | null | undefined | BigInt | Date;
export declare function isOrdinal(val: unknown): val is Ordinal;
/**
 * Types comparable by equality with `deepEqual(a, b)`.
 */
export declare type Eq = EntityPropTypes;
/**
 * Types comparable by simple equality with `a === b`.
 */
export declare type SimpleEq = string | boolean | number | null | undefined | Date;
/**
 * Types valid for a primary key.
 */
declare type PrimaryKey = string | number;
/**
 * Select only properties valid for a primary key of a given object.
 */
export declare type PrimaryKeyOf<T> = keyof T & {
    [Key in keyof T]: T[Key] extends PrimaryKey ? Key : never;
}[keyof T];
/**
 * An object with a primary key.
 */
export declare type Entity<T> = {
    [key in PrimaryKeyOf<T>]: T[key];
};
/**
 * Returns type T if T only contains valid properties.
 */
export declare type ValidEntity<T> = T extends ValidProperties<T> ? T : never;
export declare type ValidProperties<T> = T extends Function | Symbol ? never : T extends Date ? T : T extends BigInt ? T : T extends object ? {
    [K in keyof T]: ValidEntity<T[K]>;
} : T;
/**
 * Returns true if the given object is a valid entity.
 *
 * @example
 * assert(isValidEntity({ a: "" }) === true);
 * assert(isValidEntity({ b: () => {} }) === false);
 */
export declare const isValidEntity: <T>(obj: T) => obj is ValidEntity<T>;
export {};
