"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.insertIntoSortedList = exports.sortWithSortQuery = exports.sortItems = void 0;
const core_1 = require("../../core");
const compare_1 = require("../compare");
/**
 * @returns all items from `items` sorted according to the given `sort` object.
 */
function sortItems(table, items, sort, selectResult) {
    // In the event we want to sort by id and the select engine scanned only the id column, it's already sorted
    const primaryKey = table[core_1.BlinkKey].options.primary;
    if (sort.key === primaryKey) {
        if (selectResult && !selectResult.fullTableScan) {
            const rowsScanned = selectResult.rowsScanned;
            if (rowsScanned && rowsScanned.length === 1 && rowsScanned[0] === primaryKey) {
                switch (sort.order) {
                    case "asc":
                        return items;
                    case "desc":
                        items.reverse();
                        return items;
                }
            }
        }
    }
    items.sort(sortWithSortQuery(sort));
    return items;
}
exports.sortItems = sortItems;
/**
 * Returns a comparison function that can be used to sort a list according to a sort query.
 */
function sortWithSortQuery(sort) {
    return (a, b) => {
        const aKey = a[sort.key];
        const bKey = b[sort.key];
        return sort.order === "asc" ? (0, compare_1.compare)(aKey, bKey) : (0, compare_1.compare)(bKey, aKey);
    };
}
exports.sortWithSortQuery = sortWithSortQuery;
/**
 * Inserts `item` into `arr` which is assumed to be sorted with `sort`.
 */
function insertIntoSortedList(arr, item, sort) {
    function insertInIndex(index) {
        arr.splice(index, 0, item);
    }
    function getSortedIndex() {
        let low = 0;
        let high = arr.length;
        while (low < high) {
            const mid = (low + high) >>> 1;
            if (sortWithSortQuery(sort)(arr[mid], item) < 0)
                low = mid + 1;
            else
                high = mid;
        }
        return low;
    }
    insertInIndex(getSortedIndex());
}
exports.insertIntoSortedList = insertIntoSortedList;
