"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.selectForWhere = void 0;
const core_1 = require("../../core");
const matchers_1 = require("../analyze/matchers");
const matchers_2 = require("./matchers");
/**
 * Select all items for `filter`.
 *
 * @returns the selected items from the database, or `null` in case a full table scan is required.
 */
function selectForWhere(table, filter, cb, from) {
    // No matchers in filter? We can return early
    if (Object.keys(filter).length === 0)
        return { fullTableScan: false };
    const primaryKeyProperty = table[core_1.BlinkKey].options.primary;
    // Check for primary key index
    if (primaryKeyProperty in filter) {
        const btree = table[core_1.BlinkKey].storage.primary;
        const matcher = filter[primaryKeyProperty];
        (0, matchers_2.selectForMatcher)(btree, matcher, cb, from);
        return { rowsScanned: [primaryKeyProperty], fullTableScan: false };
    }
    let minComplexity = Number.MAX_SAFE_INTEGER;
    let bestFilter = undefined;
    // Analyze performance of other indexes and use the most performant one
    for (const property in table[core_1.BlinkKey].storage.indexes) {
        const btree = table[core_1.BlinkKey].storage.indexes[property];
        if (property in filter) {
            const matcher = filter[property];
            const complexity = (0, matchers_1.analyzeMatcher)(btree, matcher);
            if (complexity < minComplexity) {
                minComplexity = complexity;
                bestFilter = { btree, matcher, property };
            }
        }
    }
    // If we have at least one filter, use it
    if (bestFilter) {
        (0, matchers_2.selectForMatcher)(bestFilter.btree, bestFilter.matcher, (items) => {
            for (const item of items) {
                cb(item);
            }
        });
        return {
            rowsScanned: [bestFilter.property],
            fullTableScan: false,
        };
    }
    // Otherwise, we need a full table scan
    table[core_1.BlinkKey].storage.primary.valuesArray().forEach((v) => cb(v));
    return { fullTableScan: true };
}
exports.selectForWhere = selectForWhere;
