"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.analyzeMatcher = void 0;
const compare_1 = require("../compare");
/**
 * Returns the theoretical complexity of a given matcher,
 * e.g. how many items will need to be evaluated approximately.
 */
function analyzeMatcher(btree, matcher, from) {
    if (typeof matcher !== "object" || matcher === null) {
        return 1; // Simple Equals matcher
    }
    else {
        if ("eq" in matcher) {
            return 1; // Complex Equals matcher
        }
        else if ("in" in matcher) {
            return matcher.in.length;
        }
        else if ("gt" in matcher) {
            const key = matcher.gt;
            return analyzeGtMatcher(key, btree, from);
        }
        else if ("gte" in matcher) {
            const key = matcher.gte;
            return analyzeGtMatcher(key, btree, from);
        }
        else if ("lt" in matcher) {
            const key = matcher.lt;
            return analyzeLtMatcher(key, btree, from);
        }
        else if ("lte" in matcher) {
            const key = matcher.lte;
            return analyzeLtMatcher(key, btree, from);
        }
        else if ("between" in matcher) {
            const keys = matcher.between;
            return analyzeBetweenMatcher(keys[0], keys[1], btree, from);
        }
    }
    return Number.MAX_SAFE_INTEGER;
}
exports.analyzeMatcher = analyzeMatcher;
/**
 * returns the theoretical complexity of a gt/gte matcher.
 */
function analyzeGtMatcher(key, btree, from) {
    key = from ? (0, compare_1.getBiggerKey)(key, from) : key;
    let num = 1;
    const rootKeys = btree["_root"].keys;
    if (rootKeys.length === 0)
        return 0;
    for (let i = rootKeys.length - 1; i >= 0; i--) {
        if ((0, compare_1.compare)(rootKeys[i], key) >= 0) {
            num++;
        }
        else {
            break;
        }
    }
    return (Math.floor((num * 10) / rootKeys.length) / 10) * btree.size;
}
/**
 * returns the theoretical complexity of a lt/lte matcher.
 */
function analyzeLtMatcher(key, btree, from) {
    if (from)
        return analyzeBetweenMatcher(from, key, btree);
    let num = 1;
    const rootKeys = btree["_root"].keys;
    if (rootKeys.length === 0)
        return 0;
    for (let i = 0; i < rootKeys.length; i++) {
        if ((0, compare_1.compare)(rootKeys[i], key) <= 0) {
            num++;
        }
        else {
            break;
        }
    }
    return (Math.floor((num * 10) / rootKeys.length) / 10) * btree.size;
}
/**
 * returns the theoretical complexity of a between matcher.
 */
function analyzeBetweenMatcher(min, max, btree, from) {
    min = from ? (0, compare_1.getBiggerKey)(min, from) : min;
    let num = 1;
    const rootKeys = btree["_root"].keys;
    if (rootKeys.length === 0)
        return 0;
    for (let i = 0; i < rootKeys.length; i++) {
        const c = compare_1.compare;
        const key = rootKeys[i];
        if (c(key, min) >= 0 && c(key, max) <= 0) {
            num++;
        }
        if (c(key, max) >= 0) {
            break;
        }
    }
    return (Math.floor((num * 10) / rootKeys.length) / 10) * btree.size;
}
