"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.watch = void 0;
const Middleware_1 = require("../events/Middleware");
const filter_1 = require("../query/filter");
const limit_1 = require("../query/limit");
const sort_1 = require("../query/sort");
const createDB_1 = require("./createDB");
const many_1 = require("./many");
const table_utils_1 = require("./table.utils");
function watch(table, queryOrCallback, callback) {
    return __awaiter(this, void 0, void 0, function* () {
        let query;
        let cb;
        if (typeof queryOrCallback === "object") {
            query = queryOrCallback;
            cb = callback;
        }
        else {
            query = undefined;
            cb = queryOrCallback;
        }
        const primaryKeyProperty = table[createDB_1.BlinkKey].options.primary;
        function sanitize(items) {
            return table_utils_1.TableUtils.cloneIfNecessary(table, items);
        }
        function sort(items) {
            var _a;
            if (!((_a = query === null || query === void 0 ? void 0 : query.limit) === null || _a === void 0 ? void 0 : _a.from))
                return items;
            return items.filter((i) => i[primaryKeyProperty] >= query.limit.from);
        }
        function limit(items) {
            if (!(query === null || query === void 0 ? void 0 : query.limit))
                return items;
            return (0, limit_1.limitItems)(table, items, query.limit, true);
        }
        function insertIntoEntityList(item) {
            var _a;
            (0, sort_1.insertIntoSortedList)(entities, item, (_a = query === null || query === void 0 ? void 0 : query.sort) !== null && _a !== void 0 ? _a : {
                key: primaryKeyProperty,
                order: "asc",
            });
        }
        // Retrieve initial entities
        let entities = yield (0, Middleware_1.middleware)(table, {
            action: "watch",
            params: [
                table,
                {
                    where: query === null || query === void 0 ? void 0 : query.where,
                    sort: query === null || query === void 0 ? void 0 : query.sort,
                    limit: (query === null || query === void 0 ? void 0 : query.limit)
                        ? Object.assign(Object.assign({}, query === null || query === void 0 ? void 0 : query.limit), { take: undefined, skip: undefined }) : undefined,
                },
            ],
        }, (table, query) => (0, many_1.internalMany)(table, query));
        cb(sanitize(limit(sort(entities))));
        const removeOnInsertCb = table[createDB_1.BlinkKey].events.onInsert.register((changes) => {
            let entitiesHaveChanged = false;
            for (const change of changes) {
                const entity = change.entity;
                if ((query === null || query === void 0 ? void 0 : query.where) && !(0, filter_1.matches)(entity, query.where)) {
                    continue;
                }
                insertIntoEntityList(entity);
                entitiesHaveChanged = true;
            }
            if (entitiesHaveChanged) {
                cb(sanitize(limit(sort(entities))));
            }
        });
        const removeOnUpdateCb = table[createDB_1.BlinkKey].events.onUpdate.register((changes) => {
            let entitiesHaveChanged = false;
            for (const change of changes) {
                const oldEntity = change.oldEntity;
                const newEntity = change.newEntity;
                const matchesOldEntity = !(query === null || query === void 0 ? void 0 : query.where) || (0, filter_1.matches)(oldEntity, query.where);
                const matchesNewEntity = !(query === null || query === void 0 ? void 0 : query.where) || (0, filter_1.matches)(newEntity, query.where);
                if (!matchesOldEntity && !matchesNewEntity) {
                    continue;
                }
                else if (matchesOldEntity && !matchesNewEntity) {
                    const primaryKey = oldEntity[primaryKeyProperty];
                    entities = entities.filter((e) => e[primaryKeyProperty] !== primaryKey);
                }
                else if (!matchesOldEntity && matchesNewEntity) {
                    insertIntoEntityList(newEntity);
                }
                else if (matchesOldEntity && matchesNewEntity) {
                    const primaryKey = newEntity[primaryKeyProperty];
                    const entityInArray = entities.find((e) => e[primaryKeyProperty] === primaryKey);
                    if (entityInArray) {
                        for (const prop in entityInArray) {
                            entityInArray[prop] = newEntity[prop];
                        }
                    }
                }
                entitiesHaveChanged = true;
            }
            if (entitiesHaveChanged) {
                cb(sanitize(limit(sort(entities))));
            }
        });
        const removeOnRemoveCb = table[createDB_1.BlinkKey].events.onRemove.register((changes) => {
            let entitiesHaveChanged = false;
            for (const change of changes) {
                const entity = change.entity;
                const primaryKey = entity[primaryKeyProperty];
                const index = entities.findIndex((e) => e[primaryKeyProperty] === primaryKey);
                if (index !== -1) {
                    entities.splice(index, 1);
                    entitiesHaveChanged = true;
                }
            }
            if (entitiesHaveChanged) {
                cb(sanitize(limit(sort(entities))));
            }
        });
        const removeOnClearCb = table[createDB_1.BlinkKey].events.onClear.register(() => {
            entities = [];
            cb([]);
        });
        return () => {
            removeOnInsertCb();
            removeOnUpdateCb();
            removeOnRemoveCb();
            removeOnClearCb();
        };
    });
}
exports.watch = watch;
