import { Query } from "../query/types";
import { Entity, PrimaryKeyOf } from "../types";
import { Table } from "./createTable";
/**
 * Watches all changes in `table` and calls `callback` whenever entities are inserted, updated, or removed.
 *
 * The callback is called once immediately after registering with all current entities in the `table`.
 *
 * @returns a function to be called if you want to stop watching changes.
 *
 * @example
 * const db = createDB();
 * const userTable = createTable<User>(db, "users")();
 * // This will output 'List of all users: []'
 * await watch(userTable, (users) => {
 *   console.log('List of all users: ', users);
 * });
 * // This will output 'List of all users: [{ id: "some-uuid", name: "Alice" }]'
 * const userId = await insert(userTable, { id: uuid(), name: 'Alice' });
 * // This will output 'List of all users: [{ id: "some-uuid", name: "Bob" }]'
 * await update(userTable, { id: userId, name: 'Bob' });
 * // This will output 'List of all users: []'
 * await remove(userTable, { id: userId });
 */
export declare function watch<T extends Entity<T>, P extends PrimaryKeyOf<T>>(table: Table<T, P>, callback: (entities: T[]) => Promise<void> | void): Promise<() => void>;
/**
 * Watches all changes in `table` and calls `callback` whenever entities
 * that match the given `filter` are inserted, updated, or removed.
 *
 * The callback is called once immediately after registering with all
 * current entities in the `table` that match the given `filter`.
 *
 * @returns a function to be called if you want to stop watching changes.
 *
 * @example
 * const db = createDB();
 * const userTable = createTable<User>(db, "users")();
 * // This will output 'List of all babies: []'
 * await watch(userTable, { age: { lt: 3 } }, (users) => {
 *   console.log('List of all babies: ', users);
 * });
 * // This will output 'List of all babies: [{ id: "some-uuid", name: "Alice" }]'
 * const userId = await insert(userTable, { id: uuid(), name: 'Alice', age: 2 });
 * // This will output 'List of all babies: []'
 * await update(userTable, { id: userId, age: 40 });
 * // This won't output anything
 * await remove(userTable, { id: userId });
 */
export declare function watch<T extends Entity<T>, P extends PrimaryKeyOf<T>>(table: Table<T, P>, query: Query<T, P>, callback: (entities: T[]) => Promise<void> | void): Promise<() => void>;
