"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.internalUpsertMany = exports.upsertMany = void 0;
const Middleware_1 = require("../events/Middleware");
const createDB_1 = require("./createDB");
const insertMany_1 = require("./insertMany");
const updateMany_1 = require("./updateMany");
/**
 * For each given entity of `entities`, either updates the entity if it exists,
 * or inserts it into the table if it doesn't.
 *
 * @returns the primary key of the inserted/updated entities,
 * in the same order as the items.
 *
 * @example
 * const db = createDB();
 * const userTable = createTable<User>(db, "users")();
 * const aliceId = await insert(userTable, { id: uuid(), name: 'Alice', age: 15 });
 * // Increase the age of Alice & insert a new user
 * await upsertMany(userTable, [
 *   { id: aliceId, age: 16 },
 *   { id: uuid(), age: 45 }
 * ]);
 */
function upsertMany(table, entities) {
    return Promise.resolve((0, Middleware_1.middleware)(table, { action: "upsertMany", params: [table, entities] }, (table, entities) => internalUpsertMany(table, entities)));
}
exports.upsertMany = upsertMany;
function internalUpsertMany(table, entities) {
    return __awaiter(this, void 0, void 0, function* () {
        // Split entities into items to create & items to update
        // reduces the number of outgoing events to 2
        const items = [];
        const primaryKeyProperty = table[createDB_1.BlinkKey].options.primary;
        for (const entity of entities) {
            const primaryKey = entity[primaryKeyProperty];
            const primaryKeyExists = table[createDB_1.BlinkKey].storage.primary.has(primaryKey);
            items.push({ entity, method: primaryKeyExists ? "update" : "insert" });
        }
        // Insert all items that need to be inserted, update all that need to be updated
        const itemsToInsert = items.filter((i) => i.method === "insert");
        const itemsToUpdate = items.filter((i) => i.method === "update");
        const insertIds = yield (0, insertMany_1.internalInsertMany)(table, itemsToInsert.map((i) => i.entity));
        const updateIds = yield (0, updateMany_1.internalUpdateMany)(table, itemsToUpdate.map((i) => i.entity));
        // Return the indexes in the correct order
        const ids = [];
        let insertIdsIndex = 0;
        let updateIdsIndex = 0;
        for (const item of items) {
            switch (item.method) {
                case "insert":
                    ids.push(insertIds[insertIdsIndex]);
                    insertIdsIndex++;
                    break;
                case "update":
                    ids.push(updateIds[updateIdsIndex]);
                    updateIdsIndex++;
            }
        }
        return ids;
    });
}
exports.internalUpsertMany = internalUpsertMany;
