"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.internalUpsert = exports.upsert = void 0;
const Middleware_1 = require("../events/Middleware");
const upsertMany_1 = require("./upsertMany");
/**
 * Inserts an `entity` into `table`, or updates it if the given primary key already exists.
 *
 * @returns the primary key of the entity.
 *
 * @example
 * const db = createDB();
 * const userTable = createTable<User>(db, "users")();
 * const aliceId = await insert(userTable, { id: uuid(), name: "Alice", age: 23 });
 *
 * // This will update alice
 * await upsert(userTable, { id: aliceId, name: "Alice", age: 24 });
 * // This will create a new entity
 * await upsert(userTable, { id: uuid(), name: "Bob", age: 49 });
 */
function upsert(table, entity) {
    return Promise.resolve((0, Middleware_1.middleware)(table, { action: "upsert", params: [table, entity] }, (table, entity) => internalUpsert(table, entity)));
}
exports.upsert = upsert;
function internalUpsert(table, entity) {
    return (0, upsertMany_1.internalUpsertMany)(table, [entity]).then((ids) => ids[0]);
}
exports.internalUpsert = internalUpsert;
