"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.internalUpdateMany = exports.updateMany = void 0;
const Middleware_1 = require("../events/Middleware");
const types_1 = require("../types");
const clone_1 = require("./clone");
const createDB_1 = require("./createDB");
const errors_1 = require("./errors");
/**
 * Saves updates of the given entities in the `table`.
 *
 * @throws if one of the entities has not been inserted into the table before,
 * e.g. if the primary key of the entity was not found.
 *
 * @returns the primary key of the updated entities,
 * in the same order as the items.
 *
 * @example
 * const db = createDB();
 * const userTable = createTable<User>(db, "users")();
 * const aliceId = await insert(userTable, { id: uuid(), name: 'Alice', age: 15 });
 * const bobId = await insert(userTable, { id: uuid(), name: "Bob", age: 45 });
 * // Increase the age of Alice
 * await updateMany(userTable, [
 *   { id: aliceId, age: 16 },
 *   { id: bobId, age: 45 }
 * ]);
 */
function updateMany(table, diffs) {
    return Promise.resolve((0, Middleware_1.middleware)(table, { action: "updateMany", params: [table, diffs] }, (table, diffs) => internalUpdateMany(table, diffs)));
}
exports.updateMany = updateMany;
function internalUpdateMany(table, diffs) {
    const primaryKeys = [];
    const events = [];
    for (const diff of diffs) {
        const primaryKeyProperty = table[createDB_1.BlinkKey].options.primary;
        const primaryKey = diff[primaryKeyProperty];
        if (primaryKey === undefined || primaryKey === null) {
            return Promise.reject(new errors_1.InvalidPrimaryKeyError(primaryKey));
        }
        const item = table[createDB_1.BlinkKey].storage.primary.get(primaryKey);
        if (item === undefined || item === null) {
            return Promise.reject(new errors_1.ItemNotFoundError(primaryKey));
        }
        const oldItem = (0, clone_1.clone)(item);
        let key;
        for (key in diff) {
            if (key !== primaryKeyProperty) {
                item[key] = diff[key];
                const oldVal = oldItem[key];
                const newVal = item[key];
                if (oldVal !== item[key]) {
                    key = key;
                    const btree = table[createDB_1.BlinkKey].storage.indexes[key];
                    if (btree !== undefined && (0, types_1.isOrdinal)(oldVal) && (0, types_1.isOrdinal)(newVal)) {
                        let oldIndexItems = btree.get(oldVal);
                        const arrayIndex = oldIndexItems.indexOf(item);
                        // This condition is only false if clone is disabled and the user changed the indexed property without calling update
                        if (arrayIndex !== -1) {
                            oldIndexItems.splice(arrayIndex, 1);
                        }
                        const newIndexItems = btree.get(newVal);
                        if (newIndexItems !== undefined) {
                            newIndexItems.push(item);
                        }
                        else {
                            btree.set(newVal, [item]);
                        }
                    }
                }
            }
        }
        primaryKeys.push(primaryKey);
        events.push({ oldEntity: oldItem, newEntity: item });
    }
    void table[createDB_1.BlinkKey].events.onUpdate.dispatch(events);
    return Promise.resolve(primaryKeys);
}
exports.internalUpdateMany = internalUpdateMany;
