"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.internalRemoveMany = exports.removeMany = void 0;
const Middleware_1 = require("../events/Middleware");
const types_1 = require("../types");
const createDB_1 = require("./createDB");
/**
 * Removes the given `entities` from the `table`.
 *
 * @returns the number of removed entities.
 *
 * @example
 * const db = createDB();
 * const userTable = createTable<User>(db, "users")();
 * const userId = await insert(userTable, { id: uuid(), name: 'Alice', age: 15 });
 * // Remove Alice from the table
 * await remove(userTable, { id: userId });
 */
function removeMany(table, entities) {
    return Promise.resolve((0, Middleware_1.middleware)(table, { action: "removeMany", params: [table, entities] }, (table, entities) => internalRemoveMany(table, entities)));
}
exports.removeMany = removeMany;
function internalRemoveMany(table, entities) {
    const events = [];
    let numEntitiesRemoved = 0;
    for (const entity of entities) {
        const primaryKeyProperty = table[createDB_1.BlinkKey].options.primary;
        const primaryKey = entity[primaryKeyProperty];
        const indexes = table[createDB_1.BlinkKey].storage.indexes;
        if (Object.keys(indexes).length > 0) {
            const item = table[createDB_1.BlinkKey].storage.primary.get(primaryKey);
            if (!item)
                continue;
            for (const property in indexes) {
                const btree = indexes[property];
                const key = item[property];
                if (!(0, types_1.isOrdinal)(key))
                    continue;
                const items = btree.get(key);
                const deleteIndex = items.indexOf(item);
                if (deleteIndex !== -1) {
                    if (items.length === 1) {
                        btree.delete(key);
                    }
                    else {
                        items.splice(deleteIndex, 1);
                    }
                    btree.totalItemSize--;
                }
            }
        }
        events.push({ entity: entity });
        const hasDeleted = table[createDB_1.BlinkKey].storage.primary.delete(primaryKey);
        if (hasDeleted) {
            table[createDB_1.BlinkKey].storage.primary.totalItemSize--;
            numEntitiesRemoved++;
        }
    }
    void table[createDB_1.BlinkKey].events.onRemove.dispatch(events);
    return Promise.resolve(numEntitiesRemoved);
}
exports.internalRemoveMany = internalRemoveMany;
