import { Entity, PrimaryKeyOf } from "../types";
import { Table } from "./createTable";
import { Diff } from "./update";
/**
 * Returns the name of the property used as the primary key.
 *
 * @example
 * const db = createDB();
 * const userTable = createTable<{ uuid: string; }>(db, "users")({
 *   primary: "uuid"
 * });
 * const pk = key(userTable); // => "uuid"
 */
export declare function key<T extends Entity<T>, P extends PrimaryKeyOf<T>>(table: Table<T, P>): P;
/**
 * Returns the primary key of an entity.
 *
 * @example
 * const pk = key(userTable, { id: "random-uuid", name: "Alice", age: 23 });
 */
export declare function key<T extends Entity<T>, P extends PrimaryKeyOf<T>>(table: Table<T, P>, item: Diff<T, P>): T[P];
/**
 * Returns the primary key of entities.
 *
 * @example
 * const [pk1, pk2] = key(userTable, [
 *   { id: "random-uuid-1", name: "Alice", age: 23 },
 *   { id: "random-uuid-2", name: "Bob", age: 49 }
 * ]);
 */
export declare function key<T extends Entity<T>, P extends PrimaryKeyOf<T>>(table: Table<T, P>, item: Diff<T, P>[]): T[P][];
/**
 * Returns the primary key of an entity.
 *
 * @example
 * await insert(userTable, { id: uuid(), name: "Alice", age: 23 });
 * const pk = await key(userTable, first(userTable, {
 *   where: {
 *     name: "Alice"
 *   }
 * }));
 */
export declare function key<T extends Entity<T>, P extends PrimaryKeyOf<T>>(table: Table<T, P>, item: Promise<Diff<T, P>>): Promise<T[P]>;
/**
 * Returns the primary key of an entity.
 *
 * @example
 * await insert(userTable, { id: uuid(), name: "Alice", age: 23 });
 * await insert(userTable, { id: uuid(), name: "Bob", age: 23 });
 * const pk = await key(userTable, many(userTable, {
 *   where: {
 *     age: 23
 *   }
 * }));
 */
export declare function key<T extends Entity<T>, P extends PrimaryKeyOf<T>>(table: Table<T, P>, item: Promise<Diff<T, P>[]>): Promise<T[P][]>;
