"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.internalInsertMany = exports.insertMany = void 0;
const table_utils_1 = require("../core/table.utils");
const Middleware_1 = require("../events/Middleware");
const types_1 = require("../types");
const createDB_1 = require("./createDB");
const errors_1 = require("./errors");
/**
 * Inserts new entities into `table`.
 *
 * @returns the primary key of the inserted entities,
 * in the same order as the items.
 *
 * @example
 * const db = createDB();
 * const userTable = createTable<User>(db, "users")();
 * const [aliceId, bobId, charlieId] = await insertMany(userTable, [
 *   { id: uuid(), name: "Alice", age: 23 },
 *   { id: uuid(), name: "Bob", age: 45 },
 *   { id: uuid(), name: "Charlie", age: 34 }
 * ]);
 */
function insertMany(table, entities) {
    return Promise.resolve((0, Middleware_1.middleware)(table, { action: "insertMany", params: [table, entities] }, (table, entities) => internalInsertMany(table, entities)));
}
exports.insertMany = insertMany;
function internalInsertMany(table, entities) {
    const primaryKeys = [];
    const events = [];
    const blinkTable = table[createDB_1.BlinkKey];
    const primaryKeyProperty = blinkTable.options.primary;
    const blinkTableStorage = blinkTable.storage;
    const primaryBtree = blinkTableStorage.primary;
    const indexBtrees = blinkTableStorage.indexes;
    // Allocating all objects in the `events` array is slow. Only do this if it's actually necessary.
    const hasEventListeners = blinkTable.events.onInsert.hasListeners();
    for (const entity of entities) {
        const primaryKey = entity[primaryKeyProperty];
        const storageEntity = table_utils_1.TableUtils.cloneIfNecessary(table, entity);
        const inserted = primaryBtree.set(primaryKey, storageEntity, false);
        if (!inserted) {
            return Promise.reject(new errors_1.PrimaryKeyAlreadyInUseError(primaryKey));
        }
        primaryBtree.totalItemSize++;
        for (const property in indexBtrees) {
            const btree = indexBtrees[property];
            const key = entity[property];
            if (!(0, types_1.isOrdinal)(key))
                continue;
            const items = btree.get(key);
            if (items !== undefined) {
                items.push(storageEntity);
            }
            else {
                btree.set(key, [storageEntity]);
            }
            btree.totalItemSize++;
        }
        primaryKeys.push(primaryKey);
        if (hasEventListeners) {
            events.push({ entity: storageEntity });
        }
    }
    if (hasEventListeners) {
        void blinkTable.events.onInsert.dispatch(events);
    }
    return Promise.resolve(primaryKeys);
}
exports.internalInsertMany = internalInsertMany;
