import BTree from "sorted-btree";
import { Dispatcher } from "../events/Dispatcher";
import { Hook } from "../events/types";
import { Entity, Ordinal, PrimaryKeyOf, ValidEntity } from "../types";
import { BlinkKey, Database } from "./createDB";
/**
 * Creates a new table where entities can be inserted/updated/deleted/retrieved.
 *
 * The default primary key of a table is `id`. If your interface does not have
 * a `id` property, or you'd like to change it to something else, use this snippet:
 *
 * ```
 * interface User {
 *   uuid: string;
 *   name: string;
 * }
 *
 * const db = createDB();
 * const userTable = createTable<User>(db, "users")({
 *   primary: "uuid" // whatever you want your primary key to be
 * });
 * ```
 *
 * Other options can be supplied with the `options` parameter.
 *
 * @example
 * interface User {
 *   id: string;
 *   name: string;
 * }
 *
 * const db = createDB();
 * const userTable = createTable<User>(db, "users")();
 */
export declare function createTable<T extends {
    id: string | number;
} & Entity<T>>(db: Database, tableName: string): <P extends PrimaryKeyOf<T> & PrimaryKeyOf<ValidEntity<T>> = "id" & PrimaryKeyOf<T> & PrimaryKeyOf<ValidEntity<T>>>(options?: TableOptions<T, P>) => Table<ValidEntity<T>, P>;
/**
 * Creates a new table where entities can be inserted/updated/deleted/retrieved.
 *
 * The primary key of the table and other options are set
 * with the `options` parameter.
 *
 * @example
 * interface User {
 *   uuid: string;
 *   name: string;
 * }
 *
 * const db = createDB();
 * const userTable = createTable<User>(db, "users")({
 *   primary: "uuid"
 * });
 */
export declare function createTable<T extends Entity<T>>(db: Database, tableName: string): <P extends PrimaryKeyOf<T> & PrimaryKeyOf<ValidEntity<T>>>(options: TableOptions<T, P>) => Table<ValidEntity<T>, P>;
export interface TableOptions<T extends Entity<T>, P extends PrimaryKeyOf<T>> {
    /**
     * The primary key of the entity.
     *
     * Defaults to `id` if your entity has that property.
     */
    primary: P;
    /**
     * Properties on which BlinkDB should create an index.
     *
     * Indexes drastically increase query performance when you specify properties often used in filters,
     * but decrease write performance a bit.
     */
    indexes?: Exclude<keyof T, P>[];
}
export interface Table<T extends Entity<T>, P extends PrimaryKeyOf<T>> {
    [BlinkKey]: {
        db: Database;
        tableName: string;
        storage: {
            primary: BTree<T[P], T>;
            indexes: IndexStorage<T>;
        };
        events: {
            onInsert: Dispatcher<{
                entity: T;
            }[]>;
            onUpdate: Dispatcher<{
                oldEntity: T;
                newEntity: T;
            }[]>;
            onRemove: Dispatcher<{
                entity: T;
            }[]>;
            onClear: Dispatcher;
        };
        hooks: Hook<T, P>[];
        options: Required<TableOptions<T, P>>;
    };
}
declare type IndexStorage<T> = {
    [Key in keyof T]?: BTree<T[Key] & Ordinal, T[]>;
};
export {};
