import { Filter } from "../query/types";
import { Entity, PrimaryKeyOf } from "../types";
import { Table } from "./createTable";
/**
 * Returns the total number of items in the given `table`.
 *
 * @example
 * const db = createDB();
 * const userTable = createTable<User>(db, "users")();
 * // Count how many entities exist in userTable
 * const count = await count(userTable);
 */
export declare function count<T extends Entity<T>, P extends PrimaryKeyOf<T>>(table: Table<T, P>): Promise<number>;
/**
 * Returns the total number of items which match the `filter`.
 *
 * The performance of count depends on the filter used. If you
 * have a large table and prefer performance over accuracy,
 * use `{ exact: false }`.
 *
 * @example
 * const db = createDB();
 * const userTable = createTable<User>(db, "users")();
 * // Count how many 22-year-olds users exist
 * const count = await count(userTable, {
 *   where: {
 *     age: 22
 *   }
 * });
 */
export declare function count<T extends Entity<T>, P extends PrimaryKeyOf<T>>(table: Table<T, P>, filter: Filter<T>, options?: {
    exact: boolean;
}): Promise<number>;
export declare function internalCount<T extends Entity<T>, P extends PrimaryKeyOf<T>>(table: Table<T, P>, filter?: Filter<T>, options?: {
    exact: boolean;
}): Promise<number>;
