package com.opennotes.feature_node.presentation.add_edit_note.components.markdown

sealed interface MarkdownElement {
    fun render(builder: StringBuilder)
}

data class Heading(val level: Int, val text: String) : MarkdownElement {
    override fun render(builder: StringBuilder) {
        builder.append("#".repeat(level)).append(" $text\n\n")
    }
}

data class CheckboxItem(val text: String, var checked: Boolean = false, var index: Int) : MarkdownElement {
    override fun render(builder: StringBuilder) {
        builder.append("[${if (checked) "X" else " "}] $text\n")
    }
}

data class Quote(val level: Int, val text: String) : MarkdownElement {
    override fun render(builder: StringBuilder) {
        builder.append("> ${text}\n")
    }
}

data class ListItem(
    val text: String,
    val isNumbered: Boolean = false,
    val number: Int? = null
) : MarkdownElement {
    override fun render(builder: StringBuilder) {
        if (isNumbered && number != null) {
            builder.append("$number. $text\n")
        } else {
            builder.append("- $text\n")
        }
    }
}

data class CodeBlock(
    val code: String,
    val isEnded: Boolean = false,
    val firstLine: String,
    val language: String? = null
) : MarkdownElement {
    override fun render(builder: StringBuilder) {
        builder.append("```")
        language?.let { builder.append(it) }
        builder.append("\n$code")
        if (isEnded) {
            builder.append("```")
        }
        builder.append("\n")
    }
}

data class NormalText(val text: String) : MarkdownElement {
    override fun render(builder: StringBuilder) {
        builder.append("$text\n\n")
    }
}

data class ImageInsertion(val photoUri: String) : MarkdownElement {
    override fun render(builder: StringBuilder) {
        builder.append("!($photoUri)\n\n")
    }
}

data class Link(val fullText: String, val urlRanges: List<Pair<String, IntRange>>) : MarkdownElement {
    override fun render(builder: StringBuilder) {
        builder.append("$fullText\n\n")
    }
}

data class HorizontalRule(val fullText: String) : MarkdownElement {
    override fun render(builder: StringBuilder) {
        builder.append("---\n\n")
    }
}