package com.opennotes.feature_node.presentation.add_edit_note.components.markdown

interface MarkdownLineProcessor {
    fun canProcessLine(line: String): Boolean
    fun processLine(line: String, builder: MarkdownBuilder)
}

class CodeBlockProcessor : MarkdownLineProcessor {
    override fun canProcessLine(line: String): Boolean {
        return line.startsWith("```")
    }

    override fun processLine(line: String, builder: MarkdownBuilder) {
        val language = line.removePrefix("```").trim().takeIf { it.isNotEmpty() }
        val codeBlock = StringBuilder()
        var index = builder.lineIndex + 1
        var isEnded = false

        while (index < builder.lines.size) {
            val nextLine = builder.lines[index]
            if (nextLine == "```") {
                builder.lineIndex = index
                isEnded = true
                break
            }
            codeBlock.appendLine(nextLine)
            index++
        }

        builder.add(CodeBlock(codeBlock.toString(), isEnded, line, language))
    }
}

class CheckboxProcessor : MarkdownLineProcessor {
    override fun canProcessLine(line: String): Boolean = line.matches(Regex("^\\[[ xX]]( .*)?"))

    override fun processLine(line: String, builder: MarkdownBuilder) {
        val checked = line.contains(Regex("^\\[[Xx]]"))
        val text = line.replace(Regex("^\\[[ xX]] ?"), "").trim()
        builder.add(CheckboxItem(text, checked, builder.lineIndex))
    }
}


class HeadingProcessor : MarkdownLineProcessor {
    override fun canProcessLine(line: String): Boolean = line.startsWith("#")

    override fun processLine(line: String, builder: MarkdownBuilder) {
        val level = line.takeWhile { it == '#' }.length
        val text = line.drop(level).trim()
        builder.add(Heading(level, text))
    }
}

class QuoteProcessor : MarkdownLineProcessor {
    override fun canProcessLine(line: String): Boolean = line.trim().startsWith(">")

    override fun processLine(line: String, builder: MarkdownBuilder) {
        val level = line.takeWhile { it == '>' }.length
        val text = line.drop(level).trim()
        builder.add(Quote(level, text))
    }
}

class ListItemProcessor : MarkdownLineProcessor {
    override fun canProcessLine(line: String): Boolean {
        val trimmed = line.trim()
        return trimmed.startsWith("- ") ||
                trimmed.startsWith("+ ") ||
                trimmed.startsWith("* ") ||
                trimmed.matches(Regex("^\\d+\\. .*"))
    }

    override fun processLine(line: String, builder: MarkdownBuilder) {
        val trimmed = line.trim()
        val text = when {
            trimmed.startsWith("- ") -> trimmed.removePrefix("- ").trim()
            trimmed.startsWith("+ ") -> trimmed.removePrefix("+ ").trim()
            trimmed.startsWith("* ") -> trimmed.removePrefix("* ").trim()
            trimmed.matches(Regex("^\\d+\\. .*")) -> {
                trimmed.substringAfter(". ").trim()
            }

            else -> trimmed
        }

        val isNumbered = trimmed.matches(Regex("^\\d+\\. .*"))
        val number = if (isNumbered) {
            trimmed.substringBefore(".").toIntOrNull() ?: 1
        } else null

        builder.add(ListItem(text, isNumbered, number))
    }
}

class ImageInsertionProcessor : MarkdownLineProcessor {
    override fun canProcessLine(line: String): Boolean {
        return line.trim().startsWith("!(") && line.trim().endsWith(")")
    }

    override fun processLine(line: String, builder: MarkdownBuilder) {
        val photoUri = line.substringAfter("!(", "").substringBefore(")")
        builder.add(ImageInsertion(photoUri))
    }
}

class LinkProcessor : MarkdownLineProcessor {
    override fun canProcessLine(line: String): Boolean {
        // Simple, fast check for URLs without regex
        return line.contains("http://") || line.contains("https://")
    }

    override fun processLine(line: String, builder: MarkdownBuilder) {
        try {
            // Simple URL detection without regex to prevent freezing
            val urlRanges = mutableListOf<Pair<String, IntRange>>()
            var searchIndex = 0
            var urlCount = 0
            val maxUrls = 3 // Limit URLs per line to prevent performance issues

            while (urlCount < maxUrls && searchIndex < line.length) {
                val httpIndex = line.indexOf("http://", searchIndex)
                val httpsIndex = line.indexOf("https://", searchIndex)

                val startIndex = when {
                    httpIndex != -1 && httpsIndex != -1 -> minOf(httpIndex, httpsIndex)
                    httpIndex != -1 -> httpIndex
                    httpsIndex != -1 -> httpsIndex
                    else -> -1
                }

                if (startIndex == -1) break

                // Find end of URL - stop at first space, newline, or end of string
                var endIndex = startIndex
                while (endIndex < line.length &&
                    line[endIndex] != ' ' &&
                    line[endIndex] != '\n' &&
                    line[endIndex] != '\t' &&
                    line[endIndex] != ')' &&
                    line[endIndex] != ']'
                ) {
                    endIndex++
                }

                // Extract URL - limit length to prevent issues
                val url = line.substring(startIndex, minOf(endIndex, startIndex + 200))
                if (url.length > 10) { // Only include reasonable URLs
                    urlRanges.add(Pair(url, startIndex until endIndex))
                    urlCount++
                }

                searchIndex = endIndex
            }

            if (urlRanges.isNotEmpty()) {
                builder.add(Link(line, urlRanges))
            } else {
                builder.add(NormalText(line))
            }
        } catch (e: Exception) {
            // Always fallback to normal text if there's any issue
            builder.add(NormalText(line))
        }
    }
}

class HorizontalRuleProcessor : MarkdownLineProcessor {
    override fun canProcessLine(line: String): Boolean {
        return line.trim() == "---"
    }

    override fun processLine(line: String, builder: MarkdownBuilder) {
        builder.add(HorizontalRule(line))
    }
}