import { Capacitor, registerPlugin } from "@capacitor/core";
import "@capacitor/dialog";
import "@capacitor/share";
class Nip07Signer {
  // Static cache to work across all instances
  static cachedPublicKey = null;
  static publicKeyPromise = null;
  // Cache for encryption operations to avoid repeated prompts
  static encryptionCache = /* @__PURE__ */ new Map();
  static decryptionCache = /* @__PURE__ */ new Map();
  // Rate limiting to prevent overwhelming user
  static operationQueue = Promise.resolve();
  static operationCount = 0;
  static lastOperationTime = 0;
  async getPublicKey() {
    this.checkExtension();
    console.log("[NIP-07] getPublicKey() called, operation count:", ++Nip07Signer.operationCount);
    if (Nip07Signer.cachedPublicKey) {
      console.log("[NIP-07] getPublicKey() using cached value");
      return Nip07Signer.cachedPublicKey;
    }
    if (Nip07Signer.publicKeyPromise) {
      console.log("[NIP-07] getPublicKey() using in-progress promise");
      return Nip07Signer.publicKeyPromise;
    }
    console.log("[NIP-07] getPublicKey() making new call to extension");
    Nip07Signer.publicKeyPromise = this.queueOperation(async () => {
      return window.nostr.getPublicKey();
    });
    try {
      const pubkey = await Nip07Signer.publicKeyPromise;
      Nip07Signer.cachedPublicKey = pubkey;
      console.log("[NIP-07] getPublicKey() completed, cached:", pubkey);
      return pubkey;
    } finally {
      Nip07Signer.publicKeyPromise = null;
    }
  }
  async signEvent(event) {
    this.checkExtension();
    console.log("[NIP-07] signEvent() called for kind:", event.kind, "operation count:", ++Nip07Signer.operationCount);
    return this.queueOperation(async () => {
      return window.nostr.signEvent(event);
    });
  }
  async encrypt(recipient, message) {
    this.checkExtension();
    if (!window.nostr.nip44) {
      throw new Error("Extension does not support NIP-44");
    }
    console.log("[NIP-07] encrypt() called for recipient:", recipient.substring(0, 8) + "...", "message length:", message.length, "operation count:", ++Nip07Signer.operationCount);
    const cacheKey = `${recipient}:${message}`;
    if (Nip07Signer.encryptionCache.has(cacheKey)) {
      console.log("[NIP-07] encrypt() using cached promise");
      return Nip07Signer.encryptionCache.get(cacheKey);
    }
    console.log("[NIP-07] encrypt() making new call to extension");
    const encryptPromise = this.queueOperation(async () => {
      return window.nostr.nip44.encrypt(recipient, message);
    }, 0);
    Nip07Signer.encryptionCache.set(cacheKey, encryptPromise);
    try {
      const result = await encryptPromise;
      console.log("[NIP-07] encrypt() completed");
      return result;
    } finally {
      setTimeout(() => {
        Nip07Signer.encryptionCache.delete(cacheKey);
      }, 1e3);
    }
  }
  async decrypt(sender, ciphertext) {
    this.checkExtension();
    if (!window.nostr.nip44) {
      throw new Error("Extension does not support NIP-44");
    }
    console.log("[NIP-07] decrypt() called for sender:", sender.substring(0, 8) + "...", "ciphertext length:", ciphertext.length, "operation count:", ++Nip07Signer.operationCount);
    const cacheKey = `${sender}:${ciphertext}`;
    if (Nip07Signer.decryptionCache.has(cacheKey)) {
      console.log("[NIP-07] decrypt() using cached promise");
      return Nip07Signer.decryptionCache.get(cacheKey);
    }
    console.log("[NIP-07] decrypt() making new call to extension");
    const decryptPromise = this.queueOperation(async () => {
      return window.nostr.nip44.decrypt(sender, ciphertext);
    }, 0);
    Nip07Signer.decryptionCache.set(cacheKey, decryptPromise);
    try {
      const result = await decryptPromise;
      console.log("[NIP-07] decrypt() completed");
      return result;
    } finally {
      setTimeout(() => {
        Nip07Signer.decryptionCache.delete(cacheKey);
      }, 1e3);
    }
  }
  async requestNip44Permissions() {
    try {
      const pubkey = await this.getPublicKey();
      console.log("[NIP-07] Triggering NIP-44 permissions check...");
      const ciphertext = await this.encrypt(pubkey, "NIP-44 Permission Check");
      await this.decrypt(pubkey, ciphertext);
      console.log("[NIP-07] NIP-44 permissions confirmed");
    } catch (e) {
      console.warn("[NIP-07] Failed to acquire NIP-44 permissions:", e);
    }
  }
  async queueOperation(operation, minDelay = 200) {
    Nip07Signer.operationQueue = Nip07Signer.operationQueue.then(async () => {
      const now = Date.now();
      const timeSinceLastOp = now - Nip07Signer.lastOperationTime;
      if (timeSinceLastOp < minDelay) {
        const delay = minDelay - timeSinceLastOp;
        if (delay > 10) {
          console.log(`[NIP-07] Delaying operation for ${delay}ms to prevent overwhelming user`);
        }
        await new Promise((resolve) => setTimeout(resolve, delay));
      }
      Nip07Signer.lastOperationTime = Date.now();
      return operation();
    });
    return Nip07Signer.operationQueue;
  }
  checkExtension() {
    if (!window.nostr) {
      throw new Error("Nostr extension not found");
    }
  }
  // Static method to clear cache (useful for logout)
  static clearCache() {
    Nip07Signer.cachedPublicKey = null;
    Nip07Signer.publicKeyPromise = null;
    Nip07Signer.encryptionCache.clear();
    Nip07Signer.decryptionCache.clear();
    Nip07Signer.operationQueue = Promise.resolve();
    Nip07Signer.operationCount = 0;
    Nip07Signer.lastOperationTime = 0;
  }
}
Capacitor.getPlatform() === "android" ? registerPlugin("AndroidNip55Signer") : null;
(() => {
  if (Capacitor.getPlatform() !== "android") {
    return null;
  }
  return registerPlugin("AndroidLocalSecretKey");
})();
