import "clsx";
import { a as attr_class, b as attr, c as clsx, s as store_get, u as unsubscribe_stores, d as attr_style, f as stringify, e as ensure_array_like, h as head } from "../../../../chunks/index2.js";
import { d as db, e as extractKind0Username, b as cacheAndroidProfileIdentity, f as extractKind0Picture, g as connectionManager, i as isAndroidNative, h as signer, a as currentUser, r as reactionRepo, j as reactionsStore, k as initRelaySendStatus, l as retryQueue, m as registerRelaySuccess, n as isMobileWeb, o as lastRelaySendStatus, p as clearRelayStatus, q as nativeDialogService } from "../../../../chunks/AndroidTapSound.js";
import Dexie from "dexie";
import { nip19, getEventHash, generateSecretKey, getPublicKey, nip44, finalizeEvent, Relay } from "nostr-tools";
import { g as get, w as writable } from "../../../../chunks/index.js";
import { D as DEFAULT_RUNTIME_CONFIG } from "../../../../chunks/defaults.js";
import { LocalNotifications } from "@capacitor/local-notifications";
import { B as Button, A as Avatar } from "../../../../chunks/Avatar.js";
import { W as ssr_context, V as escape_html } from "../../../../chunks/context.js";
import { $ as $format } from "../../../../chunks/index3.js";
import { h as html } from "../../../../chunks/CircularProgress.svelte_svelte_type_style_lang.js";
import { Haptics, ImpactStyle } from "@capacitor/haptics";
import { i as isAndroidCapacitorShell } from "../../../../chunks/platform.js";
import { Capacitor, registerPlugin } from "@capacitor/core";
import "@capacitor/dialog";
import "@capacitor/share";
import "@sveltejs/kit/internal";
import "../../../../chunks/exports.js";
import "../../../../chunks/utils.js";
import "@sveltejs/kit/internal/server";
import "../../../../chunks/state.svelte.js";
import { p as page } from "../../../../chunks/index4.js";
function onDestroy(fn) {
  /** @type {SSRContext} */
  ssr_context.r.on_destroy(fn);
}
class ProfileRepository {
  ttl = 24 * 60 * 60 * 1e3;
  // 24 hours
  async getProfile(npub) {
    const profile = await db.profiles.get(npub);
    if (profile) {
      if (Date.now() > profile.expiresAt) {
        return void 0;
      }
      return profile;
    }
    return void 0;
  }
  async getProfileIgnoreTTL(npub) {
    return await db.profiles.get(npub);
  }
  async cacheProfile(npub, metadata, options, nip05Info) {
    const now = Date.now();
    const existing = await db.profiles.get(npub);
    const messagingRelays = options.messagingRelays ?? existing?.messagingRelays ?? [];
    const mediaServers = options.mediaServers ?? existing?.mediaServers ?? [];
    const profile = {
      npub,
      metadata: metadata ?? existing?.metadata,
      messagingRelays,
      mediaServers,
      cachedAt: now,
      expiresAt: now + this.ttl
    };
    if (nip05Info) {
      profile.nip05Status = nip05Info.status;
      profile.nip05LastChecked = nip05Info.lastChecked;
      profile.nip05Pubkey = nip05Info.pubkey;
      profile.nip05Error = nip05Info.error;
    } else if (existing) {
      const oldNip05 = existing.metadata?.nip05 ?? null;
      const newNip05 = profile.metadata?.nip05 ?? null;
      if (oldNip05 === newNip05) {
        profile.nip05Status = existing.nip05Status;
        profile.nip05LastChecked = existing.nip05LastChecked;
        profile.nip05Pubkey = existing.nip05Pubkey;
        profile.nip05Error = existing.nip05Error;
      }
    }
    await db.profiles.put(profile);
  }
}
const profileRepo = new ProfileRepository();
const runtimeConfig = writable(DEFAULT_RUNTIME_CONFIG);
function getRuntimeConfigSnapshot() {
  return get(runtimeConfig);
}
function getDiscoveryRelays() {
  return getRuntimeConfigSnapshot().discoveryRelays;
}
function getBlasterRelayUrl() {
  return getRuntimeConfigSnapshot().blasterRelayUrl;
}
function getDefaultBlossomServers() {
  return getRuntimeConfigSnapshot().defaultBlossomServers;
}
function getRobohashBaseUrl() {
  return getRuntimeConfigSnapshot().robohashBaseUrl;
}
const MEMORY_CACHE_TTL_MS = 15 * 60 * 1e3;
const memoryCache = /* @__PURE__ */ new Map();
function buildCacheKey(nip05, pubkeyHex) {
  return `${nip05.toLowerCase()}|${pubkeyHex.toLowerCase()}`;
}
async function verifyNip05(nip05, pubkeyHex) {
  const trimmed = (nip05 || "").trim();
  const now = Date.now();
  if (!trimmed) {
    return { status: "invalid", nip05: "", checkedAt: now, error: "empty-nip05" };
  }
  if (typeof window === "undefined") {
    return { status: "unknown", nip05: trimmed, checkedAt: now, error: "no-window" };
  }
  const atIndex = trimmed.indexOf("@");
  if (atIndex <= 0 || atIndex === trimmed.length - 1) {
    return { status: "invalid", nip05: trimmed, checkedAt: now, error: "invalid-format" };
  }
  const localPart = trimmed.slice(0, atIndex).toLowerCase();
  const domain = trimmed.slice(atIndex + 1);
  const cacheKey = buildCacheKey(trimmed, pubkeyHex);
  const cached = memoryCache.get(cacheKey);
  if (cached && now - cached.cachedAt < MEMORY_CACHE_TTL_MS) {
    return cached.result;
  }
  const url = `https://${domain}/.well-known/nostr.json?name=${encodeURIComponent(localPart)}`;
  let status = "unknown";
  let matchedPubkey;
  let error;
  try {
    const resp = await fetch(url);
    if (!resp.ok) {
      error = `http-${resp.status}`;
    } else {
      const json = await resp.json();
      const names = json.names;
      if (!names || typeof names !== "object") {
        status = "invalid";
        error = "missing-names";
      } else {
        const mapped = names[localPart];
        if (!mapped) {
          status = "invalid";
          error = "name-not-found";
        } else {
          matchedPubkey = mapped;
          if (mapped.toLowerCase() === pubkeyHex.toLowerCase()) {
            status = "valid";
          } else {
            status = "invalid";
            error = "pubkey-mismatch";
          }
        }
      }
    }
  } catch (e) {
    status = "unknown";
    error = e?.message || "network-error";
  }
  const result = {
    status,
    nip05: trimmed,
    checkedAt: now,
    matchedPubkey,
    error
  };
  memoryCache.set(cacheKey, { result, cachedAt: now });
  return result;
}
function normalizeBlossomServerUrl(input) {
  const trimmed = input.trim();
  if (!trimmed) return null;
  const withScheme = trimmed.includes("://") ? trimmed : `https://${trimmed}`;
  try {
    const url = new URL(withScheme);
    if (url.protocol !== "http:" && url.protocol !== "https:") {
      return null;
    }
    return url.origin;
  } catch {
    return null;
  }
}
function parseBlossomServerListEvent(event) {
  const servers = [];
  const seen = /* @__PURE__ */ new Set();
  for (const tag of event.tags) {
    if (tag.length < 2 || tag[0] !== "server") continue;
    const normalized = normalizeBlossomServerUrl(tag[1]);
    if (!normalized || seen.has(normalized)) continue;
    seen.add(normalized);
    servers.push(normalized);
  }
  return servers;
}
class ProfileResolver {
  async resolveProfile(npub, forceRefresh = false) {
    const cached = await profileRepo.getProfile(npub);
    if (cached && !forceRefresh) {
      try {
        const decoded = nip19.decode(npub);
        const pubkey2 = typeof decoded.data === "string" ? decoded.data : null;
        const username = extractKind0Username(cached.metadata);
        if (pubkey2 && username) {
          await cacheAndroidProfileIdentity({
            pubkeyHex: pubkey2,
            username,
            picture: extractKind0Picture(cached.metadata) ?? void 0,
            updatedAt: Date.now()
          });
        }
      } catch (e) {
        console.warn("Failed to cache Android profile identity from cached profile:", e);
      }
      console.log(`Using cached profile for ${npub}`);
      return;
    }
    const { data } = nip19.decode(npub);
    const pubkey = data;
    console.log(`Fetching profile for ${npub}...`);
    const filters = [
      {
        kinds: [0, 10050, 10002, 10063],
        authors: [pubkey],
        limit: 10
        // Get multiple to ensure we catch both kinds if they exist
      }
    ];
    const buildNip05Info = async (currentMetadata) => {
      if (!currentMetadata || !currentMetadata.nip05) {
        return void 0;
      }
      const existing = await profileRepo.getProfileIgnoreTTL(npub);
      const now = Date.now();
      const NIP05_TTL_MS = 24 * 60 * 60 * 1e3;
      const sameNip05 = existing?.metadata?.nip05 === currentMetadata.nip05;
      const lastChecked = existing?.nip05LastChecked ?? 0;
      const fresh = sameNip05 && lastChecked > 0 && now - lastChecked < NIP05_TTL_MS;
      if (fresh && existing?.nip05Status) {
        return {
          status: existing.nip05Status,
          lastChecked,
          pubkey: existing.nip05Pubkey,
          error: existing.nip05Error
        };
      }
      const result = await verifyNip05(currentMetadata.nip05, pubkey);
      return {
        status: result.status,
        lastChecked: result.checkedAt,
        pubkey: result.matchedPubkey,
        error: result.error
      };
    };
    return new Promise((resolve) => {
      let metadata = null;
      let messagingRelays = [];
      let mediaServers = [];
      let foundProfile = false;
      let foundMessagingRelays = false;
      let foundMediaServers = false;
      let foundNip65Relays = false;
      const finalize = async () => {
        const nip05Info = await buildNip05Info(metadata);
        await profileRepo.cacheProfile(
          npub,
          metadata,
          {
            messagingRelays,
            mediaServers: foundMediaServers ? mediaServers : void 0
          },
          nip05Info
        );
        const username = extractKind0Username(metadata);
        if (username) {
          await cacheAndroidProfileIdentity({
            pubkeyHex: pubkey,
            username,
            picture: extractKind0Picture(metadata) ?? void 0,
            updatedAt: Date.now()
          });
        }
        resolve();
      };
      const timeout = setTimeout(() => {
        cleanup();
        finalize();
      }, 3e3);
      const cleanup = connectionManager.subscribe(filters, (event) => {
        if (event.kind === 0 && !foundProfile) {
          try {
            metadata = JSON.parse(event.content);
            foundProfile = true;
          } catch (e) {
            console.error("Failed to parse profile metadata", e);
          }
        } else if (event.kind === 10050 && !foundMessagingRelays) {
          const parsed = this.parseMessagingRelayList(event);
          messagingRelays = parsed;
          foundMessagingRelays = true;
        } else if (event.kind === 10063 && !foundMediaServers) {
          mediaServers = parseBlossomServerListEvent(event);
          foundMediaServers = true;
        } else if (event.kind === 10002 && !foundNip65Relays && !foundMessagingRelays) {
          const parsed = this.parseNIP65RelayList(event);
          messagingRelays = Array.from(/* @__PURE__ */ new Set([...parsed.read || [], ...parsed.write || []]));
          foundNip65Relays = true;
        }
        if (foundProfile && (foundMessagingRelays || foundNip65Relays)) {
          clearTimeout(timeout);
          cleanup();
          void finalize();
        }
      });
    });
  }
  parseMessagingRelayList(event) {
    const urls = [];
    const seen = /* @__PURE__ */ new Set();
    for (const tag of event.tags) {
      if (tag.length < 2 || tag[0] !== "relay") continue;
      const url = tag[1];
      if (!url || seen.has(url)) continue;
      seen.add(url);
      urls.push(url);
    }
    return urls;
  }
  parseNIP65RelayList(event) {
    const read = [];
    const write = [];
    const seenRead = /* @__PURE__ */ new Set();
    const seenWrite = /* @__PURE__ */ new Set();
    for (const tag of event.tags) {
      if (tag.length < 2 || tag[0] !== "r") continue;
      const url = tag[1];
      if (!url) continue;
      const marker = tag.length >= 3 ? tag[2] : "";
      if (marker === "read") {
        if (!seenRead.has(url)) {
          read.push(url);
          seenRead.add(url);
        }
      } else if (marker === "write") {
        if (!seenWrite.has(url)) {
          write.push(url);
          seenWrite.add(url);
        }
      } else {
        if (!seenRead.has(url)) {
          read.push(url);
          seenRead.add(url);
        }
        if (!seenWrite.has(url)) {
          write.push(url);
          seenWrite.add(url);
        }
      }
    }
    return { read, write };
  }
}
const profileResolver = new ProfileResolver();
async function discoverUserRelays(npub, isCurrentUser = false) {
  console.log(`Starting relay discovery for ${npub} (isCurrentUser: ${isCurrentUser})`);
  if (isCurrentUser) {
    connectionManager.clearAllRelays();
  }
  for (const url of getDiscoveryRelays()) {
    connectionManager.addTemporaryRelay(url);
  }
  await new Promise((r) => setTimeout(r, 1e3));
  await profileResolver.resolveProfile(npub, true);
  const profile = await profileRepo.getProfile(npub);
  if (profile) {
    if (profile.messagingRelays && profile.messagingRelays.length > 0) {
      for (const url of profile.messagingRelays) {
        if (isCurrentUser) {
          connectionManager.addPersistentRelay(url);
        }
      }
    }
  } else {
    console.log("No profile found after discovery");
  }
  connectionManager.cleanupTemporaryConnections();
}
class MessageRepository {
  async getMessages(recipientNpub, limit = 50, beforeTimestamp) {
    let collection;
    if (recipientNpub === "ALL") {
      if (beforeTimestamp) {
        collection = db.messages.where("sentAt").below(beforeTimestamp).reverse();
      } else {
        collection = db.messages.orderBy("sentAt").reverse();
      }
    } else {
      collection = db.messages.where("[recipientNpub+sentAt]").between(
        [recipientNpub, Dexie.minKey],
        [recipientNpub, beforeTimestamp || Dexie.maxKey],
        true,
        // include lower
        false
        // exclude upper
      ).reverse();
    }
    const items = await collection.limit(limit).toArray();
    return items.reverse();
  }
  async getConversationPage(recipientNpub, pageSize = 50, beforeTimestamp) {
    return this.getMessages(recipientNpub, pageSize, beforeTimestamp);
  }
  async getMessageByEventId(eventId) {
    const message = await db.messages.where("eventId").equals(eventId).first();
    return message || void 0;
  }
  emitMessageSaved(msg) {
    if (typeof window !== "undefined" && typeof window.dispatchEvent === "function") {
      window.dispatchEvent(new CustomEvent("nospeak:new-message", {
        detail: {
          recipientNpub: msg.recipientNpub,
          direction: msg.direction,
          eventId: msg.eventId
        }
      }));
    }
  }
  async saveMessage(msg) {
    try {
      await db.messages.put(msg);
      this.emitMessageSaved(msg);
    } catch (e) {
      if (e.name === "ConstraintError") {
        return;
      }
      throw e;
    }
  }
  async saveMessages(messages) {
    try {
      await db.messages.bulkPut(messages);
      for (const msg of messages) {
        this.emitMessageSaved(msg);
      }
    } catch (e) {
      if (e.name === "ConstraintError") {
        for (const msg of messages) {
          try {
            await db.messages.put(msg);
            this.emitMessageSaved(msg);
          } catch (individualError) {
            if (individualError.name !== "ConstraintError") {
              throw individualError;
            }
          }
        }
      } else {
        throw e;
      }
    }
  }
  async hasMessage(eventId) {
    const count = await db.messages.where("eventId").equals(eventId).count();
    return count > 0;
  }
  async hasMessages(eventIds) {
    const messages = await db.messages.where("eventId").anyOf(eventIds).toArray();
    return new Set(messages.map((msg) => msg.eventId));
  }
  async getLastMessageRecipient() {
    const lastMessage = await db.messages.orderBy("sentAt").reverse().first();
    return lastMessage?.recipientNpub || null;
  }
  async countMessages(recipientNpub) {
    if (recipientNpub === "ALL") {
      return await db.messages.count();
    }
    return await db.messages.where("[recipientNpub+sentAt]").between(
      [recipientNpub, Dexie.minKey],
      [recipientNpub, Dexie.maxKey]
    ).count();
  }
  async getAllMessagesFor(recipientNpub) {
    if (recipientNpub === "ALL") {
      const items2 = await db.messages.orderBy("sentAt").toArray();
      return items2.sort((a, b) => a.sentAt - b.sentAt);
    }
    const items = await db.messages.where("[recipientNpub+sentAt]").between(
      [recipientNpub, Dexie.minKey],
      [recipientNpub, Dexie.maxKey],
      true,
      true
    ).toArray();
    const filtered = items.filter((m) => m.recipientNpub === recipientNpub);
    if (filtered.length !== items.length) {
      console.warn("getAllMessagesFor: filtered out mismatched recipient messages", {
        requested: recipientNpub,
        total: items.length,
        kept: filtered.length
      });
    }
    return filtered.sort((a, b) => a.sentAt - b.sentAt);
  }
}
const messageRepo = new MessageRepository();
function sleep(ms) {
  return new Promise((resolve) => setTimeout(resolve, ms));
}
async function promiseWithTimeout(promise, timeoutMs) {
  if (timeoutMs <= 0) {
    throw new Error("Timeout");
  }
  return await Promise.race([
    promise,
    sleep(timeoutMs).then(() => {
      throw new Error("Timeout");
    })
  ]);
}
async function waitForRelayConnected(connectionManager2, url, deadlineAt, pollIntervalMs) {
  while (Date.now() < deadlineAt) {
    const health = connectionManager2.getRelayHealth(url);
    if (health?.isConnected && health.relay) {
      return health;
    }
    await sleep(pollIntervalMs);
  }
  return null;
}
async function publishWithDeadline(options) {
  const {
    connectionManager: connectionManager2,
    event,
    relayUrls,
    deadlineMs,
    pollIntervalMs = 50,
    onRelaySuccess
  } = options;
  const deadlineAt = Date.now() + deadlineMs;
  const attemptPublish = async (url) => {
    const remainingForConnect = deadlineAt - Date.now();
    if (remainingForConnect <= 0) {
      return { url, status: "timeout" };
    }
    const health = await waitForRelayConnected(connectionManager2, url, deadlineAt, pollIntervalMs);
    if (!health?.relay || !health.isConnected) {
      return { url, status: "timeout" };
    }
    const remainingForPublish = deadlineAt - Date.now();
    if (remainingForPublish <= 0) {
      return { url, status: "timeout" };
    }
    const relay = health.relay;
    if (!relay) {
      return { url, status: "timeout" };
    }
    const publishOnce = async () => {
      const remaining = deadlineAt - Date.now();
      if (remaining <= 0) {
        throw new Error("Timeout");
      }
      const publishPromise = relay.publish(event);
      publishPromise.catch(() => void 0);
      await promiseWithTimeout(publishPromise, remaining);
    };
    let retriedAfterAuth = false;
    try {
      await publishOnce();
      onRelaySuccess?.(url);
      return { url, status: "success" };
    } catch (e) {
      const message = e?.message || String(e);
      if (message === "Timeout") {
        return { url, status: "timeout" };
      }
      if (!retriedAfterAuth && message.startsWith("auth-required")) {
        retriedAfterAuth = true;
        connectionManager2.markRelayAuthRequired?.(url);
        const remainingForAuth = deadlineAt - Date.now();
        if (remainingForAuth <= 0) {
          return { url, status: "timeout" };
        }
        try {
          const authPromise = connectionManager2.authenticateRelay ? connectionManager2.authenticateRelay(url) : relay.auth && relay.onauth ? relay.auth(relay.onauth) : false;
          const authenticated = await promiseWithTimeout(Promise.resolve(authPromise), remainingForAuth);
          if (authenticated === false) {
            return { url, status: "failure" };
          }
          await publishOnce();
          onRelaySuccess?.(url);
          return { url, status: "success" };
        } catch (authError) {
          const authMessage = authError?.message || String(authError);
          if (authMessage === "Timeout") {
            return { url, status: "timeout" };
          }
          return { url, status: "failure" };
        }
      }
      return { url, status: "failure" };
    }
  };
  const settled = await Promise.allSettled(relayUrls.map((url) => attemptPublish(url)));
  const successfulRelays = [];
  const failedRelays = [];
  const timedOutRelays = [];
  for (const item of settled) {
    if (item.status !== "fulfilled") {
      continue;
    }
    if (item.value.status === "success") {
      successfulRelays.push(item.value.url);
    } else if (item.value.status === "timeout") {
      timedOutRelays.push(item.value.url);
    } else {
      failedRelays.push(item.value.url);
    }
  }
  return { successfulRelays, failedRelays, timedOutRelays };
}
const DEFAULT_NOTIFICATION_ICON = "/nospeak.svg";
const ANDROID_MESSAGE_CHANNEL_ID = "messages";
const ROBOHASH_URL_SUFFIX = ".png?set=set1&bgset=bg2";
const ANDROID_NOTIFICATION_ICON_TIMEOUT_MS = 1e3;
function isNonEmptyString(value) {
  return typeof value === "string" && value.trim().length > 0;
}
function getRobohashAvatarUrl(npub) {
  return `${getRobohashBaseUrl()}${npub.slice(-10)}${ROBOHASH_URL_SUFFIX}`;
}
function getNotificationAvatarUrl(npub, picture) {
  return isNonEmptyString(picture) ? picture : getRobohashAvatarUrl(npub);
}
function getFilesystemPlugin() {
  if (typeof window === "undefined") {
    return void 0;
  }
  const filesystem = window.Capacitor?.Plugins?.Filesystem;
  if (!filesystem || typeof filesystem.writeFile !== "function") {
    return void 0;
  }
  return filesystem;
}
function bytesToBase64(bytes) {
  const nodeBuffer = globalThis.Buffer;
  if (nodeBuffer) {
    return nodeBuffer.from(bytes).toString("base64");
  }
  let binary = "";
  const chunkSize = 32768;
  for (let offset = 0; offset < bytes.length; offset += chunkSize) {
    binary += String.fromCharCode(...bytes.slice(offset, offset + chunkSize));
  }
  return btoa(binary);
}
async function blobToBase64(blob) {
  if (typeof FileReader === "undefined") {
    const buffer = await blob.arrayBuffer();
    return bytesToBase64(new Uint8Array(buffer));
  }
  return await new Promise((resolve, reject) => {
    const reader = new FileReader();
    reader.onloadend = () => {
      const result = reader.result;
      if (typeof result !== "string") {
        reject(new Error("Failed to convert blob to base64"));
        return;
      }
      const commaIndex = result.indexOf(",");
      resolve(commaIndex >= 0 ? result.slice(commaIndex + 1) : result);
    };
    reader.onerror = () => {
      reject(reader.error ?? new Error("FileReader error"));
    };
    reader.readAsDataURL(blob);
  });
}
class NotificationService {
  settings = {
    notificationsEnabled: true
  };
  isAndroidNativeEnv;
  androidChannelInitialized = false;
  constructor() {
    this.isAndroidNativeEnv = isAndroidNative();
    this.loadSettings();
    if (this.isAndroidNativeEnv) {
      this.initializeAndroidListeners();
    }
  }
  loadSettings() {
    try {
      if (typeof window !== "undefined" && window.localStorage) {
        const saved = window.localStorage.getItem("nospeak-settings");
        if (saved) {
          const settings = JSON.parse(saved);
          this.settings = { ...this.settings, ...settings };
        }
      }
    } catch (e) {
      console.error("Failed to load notification settings:", e);
    }
  }
  async ensureAndroidChannel() {
    if (!this.isAndroidNativeEnv || this.androidChannelInitialized) {
      return;
    }
    try {
      await LocalNotifications.createChannel({
        id: ANDROID_MESSAGE_CHANNEL_ID,
        name: "Messages",
        description: "nospeak message notifications",
        importance: 4
      });
      this.androidChannelInitialized = true;
    } catch (e) {
      console.warn("Failed to create Android notification channel:", e);
    }
  }
  initializeAndroidListeners() {
    try {
      LocalNotifications.addListener("localNotificationActionPerformed", (event) => {
        try {
          const extra = event.notification?.extra;
          const url = extra && typeof extra.url === "string" ? extra.url : void 0;
          if (typeof window !== "undefined" && url) {
            window.location.href = url;
          }
        } catch (e) {
          console.error("Failed to handle Android notification tap:", e);
        }
      });
    } catch (e) {
      console.warn("Failed to register Android local notification listener:", e);
    }
  }
  async showNewMessageNotification(senderNpub, message) {
    this.loadSettings();
    if (!this.settings.notificationsEnabled) {
      return;
    }
    if (this.isAndroidNativeEnv) {
      return;
    }
    if (typeof document !== "undefined") {
      const hasFocus = typeof document.hasFocus === "function" ? document.hasFocus() : true;
      const isVisible = document.visibilityState === "visible";
      let isSameConversation = false;
      if (typeof window !== "undefined") {
        const path = window.location.pathname;
        if (path.startsWith("/chat/")) {
          const currentNpub = decodeURIComponent(path.slice("/chat/".length).replace(/\/$/, ""));
          isSameConversation = currentNpub === senderNpub;
        }
      }
      const shouldSuppress = isVisible && hasFocus && isSameConversation;
      if (shouldSuppress) {
        return;
      }
    }
    let senderName = senderNpub.slice(0, 10) + "...";
    let senderPicture;
    try {
      const profile = await profileRepo.getProfileIgnoreTTL(senderNpub);
      if (profile && profile.metadata) {
        senderName = profile.metadata.name || profile.metadata.display_name || profile.metadata.displayName || senderName;
        senderPicture = profile.metadata.picture;
      }
    } catch (e) {
      console.error("Failed to load sender profile for notification:", e);
    }
    if (this.isAndroidNativeEnv) {
      await this.showAndroidNotification(senderNpub, senderName, senderPicture, message);
      return;
    }
    await this.showWebNotification(senderNpub, senderName, senderPicture, message);
  }
  async resolveAndroidNotificationLargeIcon(senderNpub, senderPicture) {
    const filesystem = getFilesystemPlugin();
    if (!filesystem) {
      return void 0;
    }
    try {
      const avatarUrl = getNotificationAvatarUrl(senderNpub, senderPicture);
      const controller = typeof AbortController !== "undefined" ? new AbortController() : void 0;
      const timeoutId = controller ? setTimeout(() => controller.abort(), ANDROID_NOTIFICATION_ICON_TIMEOUT_MS) : void 0;
      let response;
      try {
        response = await fetch(avatarUrl, controller ? { signal: controller.signal } : void 0);
      } finally {
        if (timeoutId) {
          clearTimeout(timeoutId);
        }
      }
      if (!response.ok) {
        return void 0;
      }
      const blob = await response.blob();
      const base64 = await blobToBase64(blob);
      const seed = senderNpub.slice(-10);
      const path = `nospeak-notification-icons/${seed}.png`;
      const result = await filesystem.writeFile({
        path,
        data: base64,
        directory: "CACHE",
        recursive: true
      });
      return result.uri ?? result.path;
    } catch {
      return void 0;
    }
  }
  async showAndroidNotification(senderNpub, senderName, senderPicture, message) {
    try {
      await this.ensureAndroidChannel();
      const permissions = await LocalNotifications.checkPermissions().catch(() => ({ display: "denied" }));
      if (permissions.display !== "granted") {
        return;
      }
      const id = Math.floor(Date.now() % 2147483647);
      const largeIcon = await this.resolveAndroidNotificationLargeIcon(senderNpub, senderPicture);
      const notification = {
        id,
        title: `New message from ${senderName}`,
        body: message,
        channelId: ANDROID_MESSAGE_CHANNEL_ID,
        smallIcon: "ic_stat_nospeak",
        extra: {
          url: `/chat/${senderNpub}`
        }
      };
      if (isNonEmptyString(largeIcon)) {
        notification.largeIcon = largeIcon;
      }
      await LocalNotifications.schedule({
        notifications: [notification]
      });
    } catch (e) {
      console.error("Failed to show Android local notification:", e);
    }
  }
  async showWebNotification(senderNpub, senderName, senderPicture, message) {
    if (typeof Notification === "undefined" || Notification.permission !== "granted") {
      return;
    }
    try {
      let swRegistration;
      if (typeof navigator !== "undefined" && "serviceWorker" in navigator) {
        try {
          swRegistration = await Promise.race([
            navigator.serviceWorker.ready,
            new Promise((resolve) => setTimeout(() => resolve(void 0), 1e3))
          ]);
        } catch (e) {
          console.warn("Service Worker check failed:", e);
        }
      }
      if (swRegistration) {
        await swRegistration.showNotification(`New message from ${senderName}`, {
          body: message,
          icon: getNotificationAvatarUrl(senderNpub, senderPicture),
          badge: DEFAULT_NOTIFICATION_ICON,
          tag: `message-${senderNpub}`,
          // Group notifications by sender
          requireInteraction: false,
          silent: false,
          data: {
            url: `/chat/${senderNpub}`
          }
        });
      } else if (typeof Notification !== "undefined") {
        const notification = new Notification(`New message from ${senderName}`, {
          body: message,
          icon: getNotificationAvatarUrl(senderNpub, senderPicture),
          badge: DEFAULT_NOTIFICATION_ICON,
          tag: `message-${senderNpub}`,
          // Group notifications by sender
          requireInteraction: false,
          silent: false
        });
        notification.onclick = () => {
          if (typeof window !== "undefined") {
            window.focus();
            window.location.href = `/chat/${senderNpub}`;
          }
          notification.close();
        };
        setTimeout(() => {
          notification.close();
        }, 5e3);
      }
    } catch (e) {
      console.error("Failed to show notification:", e);
    }
  }
  async requestPermission() {
    if (this.isAndroidNativeEnv) {
      try {
        const result = await LocalNotifications.requestPermissions();
        return result.display === "granted";
      } catch (e) {
        console.error("Failed to request Android notification permission:", e);
        return false;
      }
    }
    if (typeof window !== "undefined" && "Notification" in window) {
      const permission = await Notification.requestPermission();
      return permission === "granted";
    }
    return false;
  }
  async showReactionNotification(senderNpub, emoji) {
    this.loadSettings();
    if (!this.settings.notificationsEnabled) {
      return;
    }
    if (this.isAndroidNativeEnv) {
      return;
    }
    if (typeof document !== "undefined") {
      const hasFocus = typeof document.hasFocus === "function" ? document.hasFocus() : true;
      const isVisible = document.visibilityState === "visible";
      let isSameConversation = false;
      if (typeof window !== "undefined") {
        const path = window.location.pathname;
        if (path.startsWith("/chat/")) {
          const currentNpub = decodeURIComponent(path.slice("/chat/".length).replace(/\/$/, ""));
          isSameConversation = currentNpub === senderNpub;
        }
      }
      const shouldSuppress = isVisible && hasFocus && isSameConversation;
      if (shouldSuppress) {
        return;
      }
    }
    let senderName = senderNpub.slice(0, 10) + "...";
    let senderPicture;
    try {
      const profile = await profileRepo.getProfileIgnoreTTL(senderNpub);
      if (profile && profile.metadata) {
        senderName = profile.metadata.name || profile.metadata.display_name || profile.metadata.displayName || senderName;
        senderPicture = profile.metadata.picture;
      }
    } catch (e) {
      console.error("Failed to load sender profile for reaction notification:", e);
    }
    const message = `reacted ${emoji} to your message`;
    if (this.isAndroidNativeEnv) {
      await this.showAndroidReactionNotification(senderNpub, senderName, senderPicture, message);
      return;
    }
    await this.showWebReactionNotification(senderNpub, senderName, senderPicture, message);
  }
  async showAndroidReactionNotification(senderNpub, senderName, senderPicture, message) {
    try {
      await this.ensureAndroidChannel();
      const permissions = await LocalNotifications.checkPermissions().catch(() => ({ display: "denied" }));
      if (permissions.display !== "granted") {
        return;
      }
      const id = Math.floor(Date.now() % 2147483647);
      const largeIcon = await this.resolveAndroidNotificationLargeIcon(senderNpub, senderPicture);
      const notification = {
        id,
        title: `New reaction from ${senderName}`,
        body: message,
        channelId: ANDROID_MESSAGE_CHANNEL_ID,
        smallIcon: "ic_stat_nospeak",
        extra: {
          url: `/chat/${senderNpub}`
        }
      };
      if (isNonEmptyString(largeIcon)) {
        notification.largeIcon = largeIcon;
      }
      await LocalNotifications.schedule({
        notifications: [notification]
      });
    } catch (e) {
      console.error("Failed to show Android reaction notification:", e);
    }
  }
  async showWebReactionNotification(senderNpub, senderName, senderPicture, message) {
    if (typeof Notification === "undefined" || Notification.permission !== "granted") {
      return;
    }
    try {
      let swRegistration;
      if (typeof navigator !== "undefined" && "serviceWorker" in navigator) {
        try {
          swRegistration = await Promise.race([
            navigator.serviceWorker.ready,
            new Promise((resolve) => setTimeout(() => resolve(void 0), 1e3))
          ]);
        } catch (e) {
          console.warn("Service Worker check failed:", e);
        }
      }
      const title = `New reaction from ${senderName}`;
      if (swRegistration) {
        await swRegistration.showNotification(title, {
          body: message,
          icon: getNotificationAvatarUrl(senderNpub, senderPicture),
          badge: DEFAULT_NOTIFICATION_ICON,
          tag: `message-${senderNpub}`,
          requireInteraction: false,
          silent: false,
          data: {
            url: `/chat/${senderNpub}`
          }
        });
      } else if (typeof Notification !== "undefined") {
        const notification = new Notification(title, {
          body: message,
          icon: getNotificationAvatarUrl(senderNpub, senderPicture),
          badge: DEFAULT_NOTIFICATION_ICON,
          tag: `message-${senderNpub}`,
          requireInteraction: false,
          silent: false
        });
        notification.onclick = () => {
          if (typeof window !== "undefined") {
            window.focus();
            window.location.href = `/chat/${senderNpub}`;
          }
          notification.close();
        };
        setTimeout(() => {
          notification.close();
        }, 5e3);
      }
    } catch (e) {
      console.error("Failed to show reaction notification:", e);
    }
  }
  isSupported() {
    if (this.isAndroidNativeEnv) {
      return true;
    }
    return typeof window !== "undefined" && "Notification" in window;
  }
  isEnabled() {
    this.loadSettings();
    if (!this.settings.notificationsEnabled) {
      return false;
    }
    if (this.isAndroidNativeEnv) {
      return true;
    }
    return typeof window !== "undefined" && "Notification" in window && Notification.permission === "granted";
  }
}
let notificationServiceInstance = null;
const notificationService = {
  showNewMessageNotification: (senderNpub, message) => {
    if (!notificationServiceInstance) {
      notificationServiceInstance = new NotificationService();
    }
    return notificationServiceInstance.showNewMessageNotification(senderNpub, message);
  },
  showReactionNotification: (senderNpub, emoji) => {
    if (!notificationServiceInstance) {
      notificationServiceInstance = new NotificationService();
    }
    return notificationServiceInstance.showReactionNotification(senderNpub, emoji);
  },
  requestPermission: () => {
    if (!notificationServiceInstance) {
      notificationServiceInstance = new NotificationService();
    }
    return notificationServiceInstance.requestPermission();
  },
  isSupported: () => {
    if (!notificationServiceInstance) {
      notificationServiceInstance = new NotificationService();
    }
    return notificationServiceInstance.isSupported();
  },
  isEnabled: () => {
    if (!notificationServiceInstance) {
      notificationServiceInstance = new NotificationService();
    }
    return notificationServiceInstance.isEnabled();
  }
};
class ContactRepository {
  async addContact(npub, lastReadAt, lastActivityAt) {
    await db.contacts.put({
      npub,
      createdAt: Date.now(),
      lastReadAt,
      lastActivityAt
    });
  }
  async markAsRead(npub) {
    await db.contacts.update(npub, {
      lastReadAt: Date.now()
    });
  }
  async markActivity(npub, timestamp) {
    await db.contacts.update(npub, {
      lastActivityAt: timestamp ?? Date.now()
    });
  }
  async removeContact(npub) {
    await db.contacts.delete(npub);
  }
  async getContacts() {
    return await db.contacts.toArray();
  }
}
const contactRepo = new ContactRepository();
const STORAGE_VERSION = 1;
const STORAGE_PREFIX = "nospeak:unread:";
function isBrowser() {
  if (typeof window === "undefined") {
    return false;
  }
  const localStorage2 = window.localStorage;
  return !!localStorage2 && typeof localStorage2.getItem === "function" && typeof localStorage2.setItem === "function";
}
function getStorageKey(currentUserNpub) {
  return `${STORAGE_PREFIX}${currentUserNpub}`;
}
function createEmptyState() {
  return {
    version: STORAGE_VERSION,
    byChat: {}
  };
}
function normalizeChatState(state) {
  return {
    messages: Array.isArray(state?.messages) ? state?.messages.filter((v) => typeof v === "string") : [],
    reactions: Array.isArray(state?.reactions) ? state?.reactions.filter((v) => typeof v === "string") : []
  };
}
function isActivelyViewingConversation(partnerNpub) {
  if (!isBrowser()) {
    return false;
  }
  {
    return false;
  }
}
function readState(currentUserNpub) {
  if (!isBrowser()) {
    return createEmptyState();
  }
  const key = getStorageKey(currentUserNpub);
  const raw = window.localStorage.getItem(key);
  if (!raw) {
    return createEmptyState();
  }
  try {
    const parsed = JSON.parse(raw);
    if (parsed.version !== STORAGE_VERSION || typeof parsed.byChat !== "object" || parsed.byChat === null) {
      return createEmptyState();
    }
    const byChat = {};
    for (const [npub, chat] of Object.entries(parsed.byChat)) {
      byChat[npub] = normalizeChatState(chat);
    }
    return {
      version: STORAGE_VERSION,
      byChat
    };
  } catch {
    return createEmptyState();
  }
}
function writeState(currentUserNpub, state) {
  if (!isBrowser()) {
    return;
  }
  const key = getStorageKey(currentUserNpub);
  try {
    window.localStorage.setItem(key, JSON.stringify(state));
  } catch {
  }
}
function addUnreadMessage(currentUserNpub, partnerNpub, eventId) {
  if (!isBrowser()) {
    return;
  }
  if (!eventId || eventId.trim().length === 0) {
    return;
  }
  const state = readState(currentUserNpub);
  const chat = normalizeChatState(state.byChat[partnerNpub]);
  if (!chat.messages.includes(eventId)) {
    chat.messages = [...chat.messages, eventId];
  }
  state.byChat[partnerNpub] = chat;
  writeState(currentUserNpub, state);
  syncAppBadge(currentUserNpub);
}
function addUnreadReaction(currentUserNpub, partnerNpub, reactionEventId) {
  if (!isBrowser()) {
    return;
  }
  if (!reactionEventId || reactionEventId.trim().length === 0) {
    return;
  }
  const state = readState(currentUserNpub);
  const chat = normalizeChatState(state.byChat[partnerNpub]);
  if (!chat.reactions.includes(reactionEventId)) {
    chat.reactions = [...chat.reactions, reactionEventId];
  }
  state.byChat[partnerNpub] = chat;
  writeState(currentUserNpub, state);
  syncAppBadge(currentUserNpub);
}
function clearChatUnread(currentUserNpub, partnerNpub) {
  if (!isBrowser()) {
    return;
  }
  const state = readState(currentUserNpub);
  if (state.byChat[partnerNpub]) {
    delete state.byChat[partnerNpub];
    writeState(currentUserNpub, state);
    syncAppBadge(currentUserNpub);
  }
}
function getTotalUnreadCount(currentUserNpub) {
  const state = readState(currentUserNpub);
  let total = 0;
  for (const chat of Object.values(state.byChat)) {
    total += (chat.messages?.length ?? 0) + (chat.reactions?.length ?? 0);
  }
  return total;
}
async function syncAppBadge(currentUserNpub) {
  if (typeof navigator === "undefined") {
    return;
  }
  try {
    const total = getTotalUnreadCount(currentUserNpub);
    if (total === 0) {
      if (typeof navigator.clearAppBadge === "function") {
        await navigator.clearAppBadge();
        return;
      }
      if (typeof navigator.setAppBadge === "function") {
        await navigator.setAppBadge(0);
      }
      return;
    }
    if (typeof navigator.setAppBadge === "function") {
      await navigator.setAppBadge(total);
    }
  } catch {
  }
}
function createInitialSteps() {
  return [
    {
      id: "connect-discovery-relays",
      labelKey: "sync.steps.connectDiscoveryRelays",
      status: "pending"
    },
    {
      id: "fetch-messaging-relays",
      labelKey: "sync.steps.fetchMessagingRelays",
      status: "pending"
    },
    {
      id: "connect-read-relays",
      labelKey: "sync.steps.connectReadRelays",
      status: "pending"
    },
    {
      id: "fetch-history",
      labelKey: "sync.steps.fetchHistory",
      status: "pending"
    },
    {
      id: "fetch-contact-profiles",
      labelKey: "sync.steps.fetchContactProfiles",
      status: "pending"
    },
    {
      id: "fetch-user-profile",
      labelKey: "sync.steps.fetchUserProfile",
      status: "pending"
    }
  ];
}
const initialState = {
  isSyncing: false,
  progress: 0,
  isFirstSync: false,
  flowActive: false,
  steps: createInitialSteps(),
  currentStepId: null
};
const syncState = writable(initialState);
function startSync(isFirstSync) {
  syncState.update((state) => ({
    ...state,
    isSyncing: true,
    progress: 0,
    isFirstSync
  }));
}
function updateSyncProgress(progress) {
  syncState.update((state) => ({
    ...state,
    progress
  }));
}
function endSync() {
  syncState.update((state) => ({
    ...state,
    isSyncing: false,
    progress: 0,
    isFirstSync: false
  }));
}
function getSubtle$1() {
  if (typeof window !== "undefined" && window.crypto && window.crypto.subtle) {
    return window.crypto.subtle;
  }
  throw new Error("WebCrypto SubtleCrypto is not available in this environment");
}
function toHex$1(bytes) {
  return Array.from(bytes).map((b) => b.toString(16).padStart(2, "0")).join("");
}
async function sha256Hex(data) {
  const subtle = getSubtle$1();
  const hashBuffer = await subtle.digest("SHA-256", data.buffer);
  return toHex$1(new Uint8Array(hashBuffer));
}
async function encryptFileWithAesGcm(file) {
  const subtle = getSubtle$1();
  const fileBuffer = new Uint8Array(await file.arrayBuffer());
  const key = await subtle.generateKey(
    { name: "AES-GCM", length: 256 },
    true,
    ["encrypt", "decrypt"]
  );
  const rawKeyBuffer = await subtle.exportKey("raw", key);
  const rawKey = new Uint8Array(rawKeyBuffer);
  const nonceBytes = new Uint8Array(16);
  if (typeof window !== "undefined" && window.crypto && window.crypto.getRandomValues) {
    window.crypto.getRandomValues(nonceBytes);
  } else if (typeof crypto !== "undefined" && crypto.getRandomValues) {
    crypto.getRandomValues(nonceBytes);
  } else {
    throw new Error("No secure random source available for AES-GCM nonce");
  }
  const encryptedBuffer = await subtle.encrypt(
    {
      name: "AES-GCM",
      iv: nonceBytes
    },
    key,
    fileBuffer.buffer
  );
  const ciphertextBuffer = new Uint8Array(encryptedBuffer);
  const hashEncrypted = await sha256Hex(ciphertextBuffer);
  const hashPlain = await sha256Hex(fileBuffer);
  return {
    ciphertext: ciphertextBuffer,
    key: toHex$1(rawKey),
    nonce: toHex$1(nonceBytes),
    size: ciphertextBuffer.byteLength,
    hashEncrypted,
    hashPlain
  };
}
const BLOSSOM_AUTH_KIND = 24242;
const DEFAULT_EXPIRATION_SECONDS = 5 * 60;
function toBase64(input) {
  if (typeof btoa === "function") {
    return btoa(input);
  }
  return Buffer.from(input, "utf8").toString("base64");
}
async function buildBlossomUploadAuthHeader(params) {
  const s = get(signer);
  if (!s) {
    return null;
  }
  const now = Math.floor(Date.now() / 1e3);
  const expiration = now + (params.expirationSeconds ?? DEFAULT_EXPIRATION_SECONDS);
  const event = {
    kind: BLOSSOM_AUTH_KIND,
    created_at: now,
    tags: [
      ["t", "upload"],
      ["x", params.sha256],
      ["expiration", String(expiration)]
    ],
    content: params.content ?? "Upload blob"
  };
  const signed = await s.signEvent(event);
  const json = JSON.stringify(signed);
  const token = toBase64(json);
  return `Nostr ${token}`;
}
class BlossomHttpError extends Error {
  status;
  reason;
  constructor(message, params) {
    super(message);
    this.name = "BlossomHttpError";
    this.status = params.status;
    this.reason = params.reason;
  }
}
function getSubtle() {
  if (typeof window !== "undefined" && window.crypto && window.crypto.subtle) {
    return window.crypto.subtle;
  }
  throw new Error("WebCrypto SubtleCrypto is not available in this environment");
}
function toHex(bytes) {
  return Array.from(bytes).map((b) => b.toString(16).padStart(2, "0")).join("");
}
async function sha256HexFromBlob(blob) {
  const subtle = getSubtle();
  const buffer = await blob.arrayBuffer();
  const hashBuffer = await subtle.digest("SHA-256", buffer);
  return toHex(new Uint8Array(hashBuffer));
}
async function putUpload(params) {
  const authHeader = await buildBlossomUploadAuthHeader({
    sha256: params.sha256,
    content: "Upload blob"
  });
  if (!authHeader) {
    throw new Error("You must be logged in to upload media");
  }
  return await new Promise((resolve, reject) => {
    const xhr = new XMLHttpRequest();
    xhr.upload.addEventListener("progress", (e) => {
      if (!params.onProgress) return;
      if (e.lengthComputable) {
        params.onProgress(Math.round(e.loaded / e.total * 100));
      }
    });
    const uploadTimeout = setTimeout(() => {
      xhr.abort();
      reject(new Error("Upload timeout - please try again"));
    }, 3e4);
    xhr.onload = () => {
      clearTimeout(uploadTimeout);
      if (xhr.status >= 200 && xhr.status < 300) {
        try {
          const descriptor = JSON.parse(xhr.responseText);
          if (!descriptor?.url) {
            reject(new Error("Invalid response from Blossom server"));
            return;
          }
          resolve(descriptor);
        } catch {
          reject(new Error("Invalid response from Blossom server"));
        }
        return;
      }
      const reason = xhr.getResponseHeader("X-Reason");
      if (reason) {
        reject(new Error(reason));
        return;
      }
      reject(new Error(`Upload failed with status ${xhr.status}`));
    };
    xhr.onerror = () => {
      clearTimeout(uploadTimeout);
      reject(new Error("Network error during upload"));
    };
    xhr.onabort = () => {
      clearTimeout(uploadTimeout);
      reject(new Error("Upload was cancelled"));
    };
    xhr.open("PUT", `${params.server}/upload`);
    xhr.setRequestHeader("Authorization", authHeader);
    xhr.setRequestHeader("Content-Type", params.mimeType);
    xhr.send(params.body);
  });
}
async function putMirror(params) {
  const authHeader = await buildBlossomUploadAuthHeader({
    sha256: params.sha256,
    content: "Mirror blob"
  });
  if (!authHeader) {
    throw new Error("You must be logged in to upload media");
  }
  return await new Promise((resolve, reject) => {
    const xhr = new XMLHttpRequest();
    const uploadTimeout = setTimeout(() => {
      xhr.abort();
      reject(new Error("Upload timeout - please try again"));
    }, 3e4);
    xhr.onload = () => {
      clearTimeout(uploadTimeout);
      if (xhr.status >= 200 && xhr.status < 300) {
        try {
          const descriptor = JSON.parse(xhr.responseText);
          if (!descriptor?.url) {
            reject(new Error("Invalid response from Blossom server"));
            return;
          }
          resolve(descriptor);
        } catch {
          reject(new Error("Invalid response from Blossom server"));
        }
        return;
      }
      const reason = xhr.getResponseHeader("X-Reason");
      reject(
        new BlossomHttpError(`Mirror failed with status ${xhr.status}`, {
          status: xhr.status,
          reason
        })
      );
    };
    xhr.onerror = () => {
      clearTimeout(uploadTimeout);
      reject(new Error("Network error during upload"));
    };
    xhr.onabort = () => {
      clearTimeout(uploadTimeout);
      reject(new Error("Upload was cancelled"));
    };
    xhr.open("PUT", `${params.server}/mirror`);
    xhr.setRequestHeader("Authorization", authHeader);
    xhr.setRequestHeader("Content-Type", "application/json");
    xhr.send(JSON.stringify({ url: params.url }));
  });
}
async function uploadToBlossomServers(params) {
  const normalizedServers = params.servers.map((s) => normalizeBlossomServerUrl(s)).filter((s) => Boolean(s));
  if (normalizedServers.length === 0) {
    throw new Error("No Blossom servers configured");
  }
  const sha256 = params.sha256 ?? await sha256HexFromBlob(params.body);
  let primary = null;
  let lastError = null;
  for (const server of normalizedServers) {
    try {
      const descriptor = await putUpload({
        server,
        body: params.body,
        mimeType: params.mimeType,
        sha256,
        onProgress: params.onProgress
      });
      primary = { server, descriptor };
      break;
    } catch (e) {
      lastError = e;
    }
  }
  if (!primary) {
    throw lastError ?? new Error("Upload failed");
  }
  void mirrorToRemainingServers({
    servers: normalizedServers,
    primaryServer: primary.server,
    primaryUrl: primary.descriptor.url,
    body: params.body,
    mimeType: params.mimeType,
    sha256
  });
  return {
    url: primary.descriptor.url,
    primaryServer: primary.server,
    descriptor: primary.descriptor
  };
}
async function mirrorToRemainingServers(params) {
  const targets = params.servers.filter((s) => s !== params.primaryServer);
  const unsupportedMirrorStatuses = /* @__PURE__ */ new Set([404, 405, 501]);
  await Promise.allSettled(
    targets.map(async (server) => {
      try {
        await putMirror({
          server,
          url: params.primaryUrl,
          sha256: params.sha256
        });
        return;
      } catch (e) {
        if (e instanceof BlossomHttpError && unsupportedMirrorStatuses.has(e.status)) {
          try {
            await putUpload({
              server,
              body: params.body,
              mimeType: params.mimeType,
              sha256: params.sha256
            });
          } catch (fallbackError) {
            console.warn("Blossom mirror fallback upload failed", { server, error: fallbackError });
          }
          return;
        }
        console.warn("Blossom mirror request failed", { server, error: e });
      }
    })
  );
}
class MessagingService {
  debug = true;
  isFetchingHistory = false;
  lastHistoryFetch = 0;
  HISTORY_FETCH_DEBOUNCE = 5e3;
  // 5 seconds
  // First-time login history sync is intentionally capped.
  // Note: gift-wrap (NIP-59) `created_at` timestamps can be randomized by clients,
  // so the cutoff is approximate and should be treated as a best-effort window.
  FIRST_SYNC_BACKFILL_DAYS = 30;
  liveSeenEventIds = /* @__PURE__ */ new Set();
  activeSubscriptionUnsub = null;
  activeSubscriptionPubkey = null;
  // Timestamp (seconds) when the current session started. Used to suppress
  // notifications for messages sent before the app was opened.
  sessionStartedAt = 0;
  // Listen for incoming messages
  listenForMessages(publicKey) {
    const filters = [{
      kinds: [1059],
      // Kind 1059 Gift Wrap
      "#p": [publicKey]
    }];
    if (this.debug) console.log("Listening for messages...", filters);
    const unsub = connectionManager.subscribe(filters, async (event) => {
      if (this.debug) console.log("Received gift wrap event:", event);
      if (this.liveSeenEventIds.has(event.id)) {
        if (this.debug) console.log("Event already processed in live subscription, skipping");
        return;
      }
      this.liveSeenEventIds.add(event.id);
      if (this.liveSeenEventIds.size > 5e3) {
        this.liveSeenEventIds.clear();
        this.liveSeenEventIds.add(event.id);
      }
      if (await messageRepo.hasMessage(event.id)) {
        if (this.debug) console.log("Message already in cache, skipping");
        return;
      }
      await this.handleGiftWrap(event);
    });
    return unsub;
  }
  async startSubscriptionsForCurrentUser() {
    const s = get(signer);
    const user = get(currentUser);
    if (!s || !user) {
      if (this.debug) console.warn("Cannot start subscriptions: missing signer or user");
      return;
    }
    const pubkey = await s.getPublicKey();
    if (this.activeSubscriptionUnsub && this.activeSubscriptionPubkey === pubkey) {
      if (this.debug) console.log("Subscriptions already active for current user, skipping start");
      return;
    }
    if (this.activeSubscriptionUnsub) {
      try {
        this.activeSubscriptionUnsub();
      } catch (e) {
        console.error("Error while stopping previous subscription:", e);
      }
      this.activeSubscriptionUnsub = null;
      this.activeSubscriptionPubkey = null;
    }
    this.sessionStartedAt = Math.floor(Date.now() / 1e3);
    const unsub = this.listenForMessages(pubkey);
    this.activeSubscriptionUnsub = unsub;
    this.activeSubscriptionPubkey = pubkey;
    if (this.debug) console.log("Started app-global message subscriptions for current user");
  }
  stopSubscriptions() {
    if (this.activeSubscriptionUnsub) {
      try {
        this.activeSubscriptionUnsub();
      } catch (e) {
        console.error("Error while stopping subscriptions:", e);
      }
    }
    this.activeSubscriptionUnsub = null;
    this.activeSubscriptionPubkey = null;
    if (this.debug) console.log("Stopped app-global message subscriptions");
  }
  async handleGiftWrap(event) {
    const s = get(signer);
    if (!s) return;
    try {
      const decryptedGiftWrap = await s.decrypt(event.pubkey, event.content);
      const seal = JSON.parse(decryptedGiftWrap);
      if (seal.kind !== 13) throw new Error(`Expected Seal (Kind 13), got ${seal.kind}`);
      const decryptedSeal = await s.decrypt(seal.pubkey, seal.content);
      const rumor = JSON.parse(decryptedSeal);
      if (rumor.kind !== 14 && rumor.kind !== 15 && rumor.kind !== 7) {
        throw new Error(`Expected Rumor (Kind 14, 15, or 7), got ${rumor.kind}`);
      }
      const myPubkey = await s.getPublicKey();
      const pTag = rumor.tags.find((t) => t[0] === "p");
      if (!pTag || pTag[1] !== myPubkey) {
        if (rumor.pubkey !== myPubkey) {
          throw new Error("Received rumor p tag does not match my public key");
        }
      }
      if (rumor.kind === 7) {
        await this.processReactionRumor(rumor, event.id);
        return;
      }
      this.processRumor(rumor, event.id);
    } catch (e) {
      console.error("Failed to unwrap/decrypt message:", e);
    }
  }
  async processGiftWrapToMessage(event) {
    const s = get(signer);
    const user = get(currentUser);
    if (!s || !user) return null;
    try {
      const decryptedGiftWrap = await s.decrypt(event.pubkey, event.content);
      const seal = JSON.parse(decryptedGiftWrap);
      if (seal.kind !== 13) throw new Error(`Expected Seal (Kind 13), got ${seal.kind}`);
      const decryptedSeal = await s.decrypt(seal.pubkey, seal.content);
      const rumor = JSON.parse(decryptedSeal);
      if (rumor.kind !== 14 && rumor.kind !== 15 && rumor.kind !== 7) {
        throw new Error(`Expected Rumor (Kind 14, 15, or 7), got ${rumor.kind}`);
      }
      const myPubkey = await s.getPublicKey();
      const pTag = rumor.tags.find((t) => t[0] === "p");
      if (!pTag || pTag[1] !== myPubkey) {
        if (rumor.pubkey !== myPubkey) {
          throw new Error("Received rumor p tag does not match my public key");
        }
      }
      if (rumor.kind === 7) {
        await this.processReactionRumor(rumor, event.id);
        return null;
      }
      return await this.createMessageFromRumor(rumor, event.id);
    } catch (e) {
      console.error("Failed to process gift wrap:", e);
      return null;
    }
  }
  async createMessageFromRumor(rumor, originalEventId) {
    const s = get(signer);
    if (!s) return null;
    try {
      let direction;
      let partnerNpub;
      const myPubkey = await s.getPublicKey();
      if (rumor.pubkey === myPubkey) {
        direction = "sent";
        const pTag = rumor.tags.find((t) => t[0] === "p");
        const targetHex = pTag ? pTag[1] : myPubkey;
        partnerNpub = nip19.npubEncode(targetHex);
      } else {
        direction = "received";
        partnerNpub = nip19.npubEncode(rumor.pubkey);
      }
      const rumorId = getEventHash(rumor);
      if (rumor.kind === 15) {
        const fileTypeTag = rumor.tags.find((t) => t[0] === "file-type");
        const sizeTag = rumor.tags.find((t) => t[0] === "size");
        const hashTag = rumor.tags.find((t) => t[0] === "x");
        const plainHashTag = rumor.tags.find((t) => t[0] === "ox");
        const encAlgTag = rumor.tags.find((t) => t[0] === "encryption-algorithm");
        const keyTag = rumor.tags.find((t) => t[0] === "decryption-key");
        const nonceTag = rumor.tags.find((t) => t[0] === "decryption-nonce");
        const fileType = fileTypeTag?.[1];
        const fileSize = sizeTag ? parseInt(sizeTag[1], 10) || void 0 : void 0;
        return {
          recipientNpub: partnerNpub,
          message: rumor.content || "",
          sentAt: rumor.created_at * 1e3,
          eventId: originalEventId,
          rumorId,
          direction,
          createdAt: Date.now(),
          rumorKind: rumor.kind,
          fileUrl: rumor.content || void 0,
          fileType,
          fileSize,
          fileHashEncrypted: hashTag?.[1],
          fileHashPlain: plainHashTag?.[1],
          fileEncryptionAlgorithm: encAlgTag?.[1],
          fileKey: keyTag?.[1],
          fileNonce: nonceTag?.[1]
        };
      }
      const parentTag = rumor.tags.find((t) => t[0] === "e");
      const parentRumorId = parentTag?.[1];
      const location = this.parseLocationFromRumor(rumor);
      return {
        recipientNpub: partnerNpub,
        message: rumor.content,
        sentAt: rumor.created_at * 1e3,
        eventId: originalEventId,
        rumorId,
        direction,
        createdAt: Date.now(),
        rumorKind: rumor.kind,
        parentRumorId,
        location
      };
    } catch (e) {
      console.error("Failed to create message from rumor:", e);
      return null;
    }
  }
  parseLocationFromRumor(rumor) {
    const locationTag = rumor.tags.find((t) => t[0] === "location" && !!t[1]);
    const raw = locationTag?.[1] || (rumor.content?.startsWith("geo:") ? rumor.content.slice("geo:".length) : void 0);
    if (!raw) {
      return void 0;
    }
    const [latitudeText, longitudeText] = raw.split(",");
    const latitude = Number(latitudeText);
    const longitude = Number(longitudeText);
    if (!Number.isFinite(latitude) || !Number.isFinite(longitude)) {
      return void 0;
    }
    return { latitude, longitude };
  }
  async processRumor(rumor, originalEventId) {
    const s = get(signer);
    const user = get(currentUser);
    if (!s || !user) return;
    const message = await this.createMessageFromRumor(rumor, originalEventId);
    if (!message) return;
    if (this.debug) console.log(`Processed ${message.direction} message with ${message.recipientNpub}: ${message.message}`);
    await messageRepo.saveMessage(message);
    if (message.direction === "received") {
      const shouldPersistUnread = !isActivelyViewingConversation(message.recipientNpub);
      if (shouldPersistUnread) {
        addUnreadMessage(user.npub, message.recipientNpub, message.eventId);
      }
      if (!this.isFetchingHistory && rumor.created_at >= this.sessionStartedAt) {
        await notificationService.showNewMessageNotification(message.recipientNpub, message.message);
      }
      await this.autoAddContact(message.recipientNpub, true);
    }
  }
  async processReactionRumor(rumor, originalEventId) {
    const s = get(signer);
    const user = get(currentUser);
    if (!s || !user) return;
    try {
      const myPubkey = await s.getPublicKey();
      const pTag = rumor.tags.find((t) => t[0] === "p");
      if (!pTag || pTag[1] !== myPubkey && rumor.pubkey !== myPubkey) {
        return;
      }
      const eTag = rumor.tags.find((t) => t[0] === "e");
      if (!eTag || !eTag[1]) {
        return;
      }
      let content = (rumor.content || "").trim();
      if (!content) {
        return;
      }
      if (content === "+") {
        content = "👍";
      } else if (content === "-") {
        content = "👎";
      }
      const targetEventId = eTag[1];
      const authorNpub = nip19.npubEncode(rumor.pubkey);
      const reaction = {
        targetEventId,
        reactionEventId: originalEventId,
        authorNpub,
        emoji: content,
        createdAt: rumor.created_at * 1e3
      };
      await reactionRepo.upsertReaction(reaction);
      reactionsStore.applyReactionUpdate(reaction);
      const isFromOtherUser = rumor.pubkey !== myPubkey;
      if (isFromOtherUser) {
        const partnerPubkey = rumor.pubkey;
        const partnerNpub = nip19.npubEncode(partnerPubkey);
        const shouldPersistUnread = !isActivelyViewingConversation(partnerNpub);
        if (shouldPersistUnread) {
          addUnreadReaction(user.npub, partnerNpub, originalEventId);
        }
        try {
          await contactRepo.markActivity(partnerNpub, rumor.created_at * 1e3);
        } catch (activityError) {
          console.error("Failed to mark contact activity for reaction:", activityError);
        }
        if (!this.isFetchingHistory && rumor.created_at >= this.sessionStartedAt) {
          try {
            await notificationService.showReactionNotification(partnerNpub, content);
          } catch (notifyError) {
            console.error("Failed to show reaction notification:", notifyError);
          }
        }
      }
    } catch (e) {
      console.error("Failed to process reaction rumor:", e);
    }
  }
  // Check if this is a first-time sync (empty message cache)
  async isFirstTimeSync() {
    const count = await messageRepo.countMessages("ALL");
    return count === 0;
  }
  // Explicitly fetch history to fill gaps
  async fetchHistory() {
    const s = get(signer);
    if (!s) return { totalFetched: 0, processed: 0, messagesSaved: 0 };
    const now = Date.now();
    if (this.isFetchingHistory || now - this.lastHistoryFetch < this.HISTORY_FETCH_DEBOUNCE) {
      if (this.debug) console.log("History fetch debounced, skipping");
      return { totalFetched: 0, processed: 0, messagesSaved: 0 };
    }
    this.isFetchingHistory = true;
    this.lastHistoryFetch = now;
    try {
      const myPubkey = await s.getPublicKey();
      const isFirstSync = await this.isFirstTimeSync();
      startSync(isFirstSync);
      const relays = await this.getMessagingRelays(nip19.npubEncode(myPubkey));
      if (relays.length === 0) {
        console.warn("No user relays found, history fetching may be incomplete");
      }
      await this.waitForRelayConnection(relays);
      const nowSeconds = Math.floor(Date.now() / 1e3);
      const result = await this.fetchMessages({
        until: nowSeconds,
        limit: 50,
        maxBatches: isFirstSync ? 1e4 : 1,
        minUntil: isFirstSync ? nowSeconds - this.FIRST_SYNC_BACKFILL_DAYS * 86400 : void 0,
        abortOnDuplicates: !isFirstSync,
        // Only abort on duplicates for returning users
        markUnread: !isFirstSync
      });
      if (this.debug) console.log(`History fetch completed. Total fetched: ${result.totalFetched}`);
      return result;
    } finally {
      this.isFetchingHistory = false;
      endSync();
    }
  }
  // Fetch older messages for infinite scroll
  async fetchOlderMessages(until, options) {
    const s = get(signer);
    if (!s) return { totalFetched: 0, processed: 0, messagesSaved: 0, messagesSavedForChat: 0 };
    if (this.isFetchingHistory) {
      if (this.debug) console.log("Already fetching history, skipping fetchOlderMessages");
      return { totalFetched: 0, processed: 0, messagesSaved: 0, messagesSavedForChat: 0 };
    }
    this.isFetchingHistory = true;
    try {
      const myPubkey = await s.getPublicKey();
      const relays = await this.getMessagingRelays(nip19.npubEncode(myPubkey));
      await this.waitForRelayConnection(relays, 2e3);
      if (connectionManager.getConnectedRelays().length === 0) {
        return {
          totalFetched: 0,
          processed: 0,
          messagesSaved: 0,
          messagesSavedForChat: 0,
          reason: "no-connected-relays"
        };
      }
      const limit = options?.limit ?? 100;
      const timeoutMs = options?.timeoutMs ?? 5e3;
      const targetChatNpub = options?.targetChatNpub;
      const result = await this.fetchMessages({
        until,
        limit,
        maxBatches: 1,
        abortOnDuplicates: false,
        timeoutMs,
        targetChatNpub
      });
      if (this.debug) console.log(`Older messages fetch completed. Total fetched: ${result.totalFetched}`);
      return result;
    } finally {
      this.isFetchingHistory = false;
    }
  }
  async fetchMessages(options) {
    const s = get(signer);
    const user = get(currentUser);
    if (!s || !user) return { totalFetched: 0, processed: 0, messagesSaved: 0 };
    const myPubkey = await s.getPublicKey();
    let until = options.until;
    let hasMore = true;
    let totalFetched = 0;
    let messagesSaved = 0;
    let messagesSavedForChat = typeof options.targetChatNpub === "string" ? 0 : void 0;
    let batchCount = 0;
    const maxBatches = options.maxBatches ?? 1;
    while (hasMore && batchCount < maxBatches) {
      batchCount++;
      const filters = [{
        kinds: [1059],
        "#p": [myPubkey],
        limit: options.limit,
        until
      }];
      if (this.debug) console.log(`Fetching batch ${batchCount}... (until: ${until}, total: ${totalFetched})`);
      const events = await connectionManager.fetchEvents(filters, options.timeoutMs ?? 3e4);
      if (events.length === 0) {
        hasMore = false;
      } else {
        totalFetched += events.length;
        updateSyncProgress(totalFetched);
        const existingEventIds = await messageRepo.hasMessages(events.map((e) => e.id));
        const allDuplicates = events.length > 0 && existingEventIds.size === events.length;
        if (options.abortOnDuplicates && allDuplicates) {
          if (this.debug) console.log("Checkpoint reached: All events in batch are duplicates. Stopping fetch.");
          hasMore = false;
          break;
        }
        const newEvents = events.filter((event) => !existingEventIds.has(event.id));
        if (newEvents.length > 0) {
          if (this.debug) console.log(`Processing ${newEvents.length} new events in this batch...`);
          const batchResults = await Promise.allSettled(
            newEvents.map((event) => this.processGiftWrapToMessage(event))
          );
          const messagesToSave = [];
          for (const result of batchResults) {
            if (result.status === "fulfilled" && result.value) {
              messagesToSave.push(result.value);
            }
          }
          if (messagesToSave.length > 0) {
            await messageRepo.saveMessages(messagesToSave);
            if (this.debug) console.log(`Saved ${messagesToSave.length} messages from batch`);
            messagesSaved += messagesToSave.length;
            if (typeof messagesSavedForChat === "number") {
              messagesSavedForChat += messagesToSave.filter((message) => message.recipientNpub === options.targetChatNpub).length;
            }
            for (const message of messagesToSave) {
              await this.autoAddContact(message.recipientNpub, false);
              const shouldMarkUnread = !!options.markUnread && message.direction === "received" && !isActivelyViewingConversation(message.recipientNpub);
              if (shouldMarkUnread) {
                addUnreadMessage(user.npub, message.recipientNpub, message.eventId);
              }
            }
          }
        }
        const oldestEvent = events.reduce(
          (oldest, event) => event.created_at < oldest.created_at ? event : oldest
        );
        until = oldestEvent.created_at - 1;
        if (typeof options.minUntil === "number" && until < options.minUntil) {
          if (this.debug) console.log(`Cutoff reached (minUntil: ${options.minUntil}). Stopping history fetch.`);
          hasMore = false;
        }
      }
    }
    return {
      totalFetched,
      processed: totalFetched,
      messagesSaved,
      messagesSavedForChat
    };
  }
  async sendMessage(recipientNpub, text, parentRumorId, createdAtSeconds) {
    const s = get(signer);
    if (!s) throw new Error("Not authenticated");
    const senderPubkey = await s.getPublicKey();
    const senderNpub = nip19.npubEncode(senderPubkey);
    const { data: recipientPubkey } = nip19.decode(recipientNpub);
    const recipientRelays = await this.getMessagingRelays(recipientNpub);
    const senderRelays = await this.getMessagingRelays(senderNpub);
    if (recipientRelays.length === 0) {
      throw new Error("Contact has no messaging relays configured");
    }
    const allRelays = [.../* @__PURE__ */ new Set([...recipientRelays, ...senderRelays])];
    if (this.debug) console.log("Sending message to relays:", { recipientRelays, senderRelays });
    for (const url of allRelays) {
      connectionManager.addTemporaryRelay(url);
    }
    setTimeout(() => {
      connectionManager.cleanupTemporaryConnections();
    }, 15e3);
    const tags = [["p", recipientPubkey]];
    if (parentRumorId) {
      tags.push(["e", parentRumorId]);
    }
    const rumor = {
      kind: 14,
      pubkey: senderPubkey,
      created_at: createdAtSeconds ?? Math.floor(Date.now() / 1e3),
      content: text,
      tags
    };
    if (this.debug && parentRumorId) {
      console.log("Sending caption rumor", {
        content: rumor.content,
        tags: rumor.tags,
        parentRumorId
      });
    }
    const rumorId = getEventHash(rumor);
    const giftWrap = await this.createGiftWrap(rumor, recipientPubkey, s);
    if (this.debug && parentRumorId) {
      console.log("Caption giftWrap for recipient", {
        id: giftWrap.id,
        kind: giftWrap.kind,
        tags: giftWrap.tags
      });
    }
    initRelaySendStatus(giftWrap.id, recipientNpub, recipientRelays.length);
    const publishResult = await publishWithDeadline({
      connectionManager,
      event: giftWrap,
      relayUrls: recipientRelays,
      deadlineMs: 5e3,
      onRelaySuccess: (url) => registerRelaySuccess(giftWrap.id)
    });
    if (publishResult.successfulRelays.length === 0) {
      console.warn("DM send failed to reach any recipient relays", {
        recipientNpub,
        giftWrapId: giftWrap.id,
        failedRelays: publishResult.failedRelays,
        timedOutRelays: publishResult.timedOutRelays
      });
      throw new Error("Failed to send message to any relay");
    }
    if (publishResult.failedRelays.length > 0 || publishResult.timedOutRelays.length > 0) {
      console.warn("DM send did not reach some recipient relays", {
        recipientNpub,
        giftWrapId: giftWrap.id,
        successfulRelays: publishResult.successfulRelays,
        failedRelays: publishResult.failedRelays,
        timedOutRelays: publishResult.timedOutRelays
      });
    }
    const successfulRelaySet = new Set(publishResult.successfulRelays);
    for (const url of recipientRelays) {
      if (!successfulRelaySet.has(url)) {
        await retryQueue.enqueue(giftWrap, url);
      }
    }
    const selfGiftWrap = await this.createGiftWrap(rumor, senderPubkey, s);
    for (const url of senderRelays) {
      await retryQueue.enqueue(selfGiftWrap, url);
    }
    await messageRepo.saveMessage({
      recipientNpub,
      message: text,
      sentAt: (rumor.created_at || 0) * 1e3,
      eventId: selfGiftWrap.id,
      // Save SELF-WRAP ID to match incoming
      rumorId,
      // Save stable rumor ID
      direction: "sent",
      createdAt: Date.now(),
      rumorKind: 14,
      parentRumorId
    });
    await this.autoAddContact(recipientNpub);
    return rumorId;
  }
  async sendLocationMessage(recipientNpub, latitude, longitude, createdAtSeconds) {
    const s = get(signer);
    if (!s) throw new Error("Not authenticated");
    const senderPubkey = await s.getPublicKey();
    const senderNpub = nip19.npubEncode(senderPubkey);
    const { data: recipientPubkey } = nip19.decode(recipientNpub);
    const recipientRelays = await this.getMessagingRelays(recipientNpub);
    const senderRelays = await this.getMessagingRelays(senderNpub);
    if (recipientRelays.length === 0) {
      throw new Error("Contact has no messaging relays configured");
    }
    const allRelays = [.../* @__PURE__ */ new Set([...recipientRelays, ...senderRelays])];
    if (this.debug) console.log("Sending location message to relays:", { recipientRelays, senderRelays });
    for (const url of allRelays) {
      connectionManager.addTemporaryRelay(url);
    }
    setTimeout(() => {
      connectionManager.cleanupTemporaryConnections();
    }, 15e3);
    const locationValue = `${latitude},${longitude}`;
    const rumor = {
      kind: 14,
      pubkey: senderPubkey,
      created_at: createdAtSeconds ?? Math.floor(Date.now() / 1e3),
      content: `geo:${locationValue}`,
      tags: [
        ["p", recipientPubkey],
        ["location", locationValue]
      ]
    };
    const rumorId = getEventHash(rumor);
    const giftWrap = await this.createGiftWrap(rumor, recipientPubkey, s);
    initRelaySendStatus(giftWrap.id, recipientNpub, recipientRelays.length);
    const publishResult = await publishWithDeadline({
      connectionManager,
      event: giftWrap,
      relayUrls: recipientRelays,
      deadlineMs: 5e3,
      onRelaySuccess: (url) => registerRelaySuccess(giftWrap.id)
    });
    if (publishResult.successfulRelays.length === 0) {
      throw new Error("Failed to send message to any relay");
    }
    const successfulRelaySet = new Set(publishResult.successfulRelays);
    for (const url of recipientRelays) {
      if (!successfulRelaySet.has(url)) {
        await retryQueue.enqueue(giftWrap, url);
      }
    }
    const selfGiftWrap = await this.createGiftWrap(rumor, senderPubkey, s);
    for (const url of senderRelays) {
      await retryQueue.enqueue(selfGiftWrap, url);
    }
    await messageRepo.saveMessage({
      recipientNpub,
      message: rumor.content || "",
      sentAt: (rumor.created_at || 0) * 1e3,
      eventId: selfGiftWrap.id,
      rumorId,
      direction: "sent",
      createdAt: Date.now(),
      rumorKind: 14,
      location: {
        latitude,
        longitude
      }
    });
    await this.autoAddContact(recipientNpub);
    return rumorId;
  }
  mediaTypeToMime(type) {
    if (type === "image") {
      return "image/jpeg";
    }
    if (type === "video") {
      return "video/mp4";
    }
    return "audio/mpeg";
  }
  async uploadEncryptedMedia(encrypted, mediaType, mimeType, blossomServers) {
    const blob = new Blob([encrypted.ciphertext.buffer], { type: "application/octet-stream" });
    if (blossomServers.length === 0) {
      throw new Error("No Blossom servers configured");
    }
    const result = await uploadToBlossomServers({
      servers: blossomServers,
      body: blob,
      mimeType: "application/octet-stream",
      sha256: encrypted.hashEncrypted
    });
    return result.url;
  }
  async sendFileMessage(recipientNpub, file, mediaType, createdAtSeconds) {
    const s = get(signer);
    if (!s) throw new Error("Not authenticated");
    const senderPubkey = await s.getPublicKey();
    const senderNpub = nip19.npubEncode(senderPubkey);
    const { data: recipientPubkey } = nip19.decode(recipientNpub);
    const recipientRelays = await this.getMessagingRelays(recipientNpub);
    const senderRelays = await this.getMessagingRelays(senderNpub);
    if (recipientRelays.length === 0) {
      throw new Error("Contact has no messaging relays configured");
    }
    const allRelays = [.../* @__PURE__ */ new Set([...recipientRelays, ...senderRelays])];
    if (this.debug) console.log("Sending file message to relays:", { recipientRelays, senderRelays });
    for (const url of allRelays) {
      connectionManager.addTemporaryRelay(url);
    }
    setTimeout(() => {
      connectionManager.cleanupTemporaryConnections();
    }, 15e3);
    const encrypted = await encryptFileWithAesGcm(file);
    const mimeType = file.type || this.mediaTypeToMime(mediaType);
    const senderProfile = await profileRepo.getProfileIgnoreTTL(senderNpub);
    const blossomServers = senderProfile?.mediaServers ?? [];
    const fileUrl = await this.uploadEncryptedMedia(encrypted, mediaType, mimeType, blossomServers);
    const now = createdAtSeconds ?? Math.floor(Date.now() / 1e3);
    const rumor = {
      kind: 15,
      pubkey: senderPubkey,
      created_at: now,
      content: fileUrl,
      tags: [
        ["p", recipientPubkey],
        ["file-type", mimeType],
        ["encryption-algorithm", "aes-gcm"],
        ["decryption-key", encrypted.key],
        ["decryption-nonce", encrypted.nonce],
        ["size", encrypted.size.toString()],
        ["x", encrypted.hashEncrypted]
      ]
    };
    if (encrypted.hashPlain) {
      rumor.tags.push(["ox", encrypted.hashPlain]);
    }
    const rumorId = getEventHash(rumor);
    const giftWrap = await this.createGiftWrap(rumor, recipientPubkey, s);
    initRelaySendStatus(giftWrap.id, recipientNpub, recipientRelays.length);
    const publishResult = await publishWithDeadline({
      connectionManager,
      event: giftWrap,
      relayUrls: recipientRelays,
      deadlineMs: 5e3,
      onRelaySuccess: (url) => registerRelaySuccess(giftWrap.id)
    });
    if (publishResult.successfulRelays.length === 0) {
      console.warn("DM file send failed to reach any recipient relays", {
        recipientNpub,
        giftWrapId: giftWrap.id,
        failedRelays: publishResult.failedRelays,
        timedOutRelays: publishResult.timedOutRelays
      });
      throw new Error("Failed to send message to any relay");
    }
    if (publishResult.failedRelays.length > 0 || publishResult.timedOutRelays.length > 0) {
      console.warn("DM file send did not reach some recipient relays", {
        recipientNpub,
        giftWrapId: giftWrap.id,
        successfulRelays: publishResult.successfulRelays,
        failedRelays: publishResult.failedRelays,
        timedOutRelays: publishResult.timedOutRelays
      });
    }
    const successfulRelaySet = new Set(publishResult.successfulRelays);
    for (const url of recipientRelays) {
      if (!successfulRelaySet.has(url)) {
        await retryQueue.enqueue(giftWrap, url);
      }
    }
    const selfGiftWrap = await this.createGiftWrap(rumor, senderPubkey, s);
    for (const url of senderRelays) {
      await retryQueue.enqueue(selfGiftWrap, url);
    }
    await messageRepo.saveMessage({
      recipientNpub,
      message: "",
      sentAt: now * 1e3,
      eventId: selfGiftWrap.id,
      rumorId,
      direction: "sent",
      createdAt: Date.now(),
      rumorKind: 15,
      fileUrl,
      fileType: mimeType,
      fileSize: encrypted.size,
      fileHashEncrypted: encrypted.hashEncrypted,
      fileHashPlain: encrypted.hashPlain,
      fileEncryptionAlgorithm: "aes-gcm",
      fileKey: encrypted.key,
      fileNonce: encrypted.nonce
    });
    await this.autoAddContact(recipientNpub);
    return rumorId;
  }
  async sendReaction(recipientNpub, targetMessage, emoji) {
    const s = get(signer);
    if (!s) throw new Error("Not authenticated");
    if (!targetMessage.rumorId) {
      console.warn("Cannot react to message without rumorId (likely old message)");
      return;
    }
    const targetId = targetMessage.rumorId;
    const senderPubkey = await s.getPublicKey();
    const senderNpub = nip19.npubEncode(senderPubkey);
    const { data: recipientPubkey } = nip19.decode(recipientNpub);
    const recipientRelays = await this.getMessagingRelays(recipientNpub);
    const senderRelays = await this.getMessagingRelays(senderNpub);
    if (recipientRelays.length === 0) {
      throw new Error("Contact has no messaging relays configured");
    }
    const allRelays = [.../* @__PURE__ */ new Set([...recipientRelays, ...senderRelays])];
    if (this.debug) console.log("Sending reaction to relays:", { recipientRelays, senderRelays });
    for (const url of allRelays) {
      connectionManager.addTemporaryRelay(url);
    }
    await new Promise((r) => setTimeout(r, 500));
    const myPubkey = senderPubkey;
    let targetAuthorPubkey;
    if (targetMessage.direction === "received") {
      targetAuthorPubkey = recipientPubkey;
    } else {
      targetAuthorPubkey = myPubkey;
    }
    const rumor = {
      kind: 7,
      pubkey: senderPubkey,
      created_at: Math.floor(Date.now() / 1e3),
      content: emoji,
      tags: [
        ["p", targetAuthorPubkey],
        ["e", targetId, "", targetAuthorPubkey]
      ]
    };
    const giftWrap = await this.createGiftWrap(rumor, recipientPubkey, s);
    const selfGiftWrap = await this.createGiftWrap(rumor, senderPubkey, s);
    for (const url of recipientRelays) {
      await retryQueue.enqueue(giftWrap, url);
    }
    for (const url of senderRelays) {
      await retryQueue.enqueue(selfGiftWrap, url);
    }
    setTimeout(() => {
      connectionManager.cleanupTemporaryConnections();
    }, 15e3);
    const authorNpub = senderNpub;
    const reaction = {
      targetEventId: targetId,
      reactionEventId: selfGiftWrap.id,
      authorNpub,
      emoji,
      createdAt: Date.now()
    };
    await reactionRepo.upsertReaction(reaction);
    reactionsStore.applyReactionUpdate(reaction);
  }
  async createGiftWrap(rumor, recipientPubkey, s) {
    const rumorJson = JSON.stringify(rumor);
    const encryptedRumor = await s.encrypt(recipientPubkey, rumorJson);
    const seal = {
      kind: 13,
      pubkey: await s.getPublicKey(),
      created_at: Math.floor(Date.now() / 1e3),
      content: encryptedRumor,
      tags: []
    };
    const signedSeal = await s.signEvent(seal);
    const sealJson = JSON.stringify(signedSeal);
    const ephemeralPrivKey = generateSecretKey();
    const ephemeralPubkey = getPublicKey(ephemeralPrivKey);
    const conversationKey = nip44.v2.utils.getConversationKey(ephemeralPrivKey, recipientPubkey);
    const encryptedSeal = nip44.v2.encrypt(sealJson, conversationKey);
    const giftWrap = {
      kind: 1059,
      pubkey: ephemeralPubkey,
      created_at: Math.floor(Date.now() / 1e3) - Math.floor(Math.random() * 172800),
      // Randomize up to 2 days in past
      content: encryptedSeal,
      tags: [["p", recipientPubkey]]
    };
    return finalizeEvent(giftWrap, ephemeralPrivKey);
  }
  async getMessagingRelays(npub) {
    let profile = await profileRepo.getProfile(npub);
    if (!profile) {
      await discoverUserRelays(npub);
      profile = await profileRepo.getProfile(npub);
    }
    if (!profile) {
      return [];
    }
    const urls = new Set(profile.messagingRelays || []);
    return Array.from(urls);
  }
  async waitForRelayConnection(relayUrls, timeoutMs = 1e4) {
    if (relayUrls.length === 0) return;
    const startTime = Date.now();
    while (Date.now() - startTime < timeoutMs) {
      const connectedRelays = connectionManager.getConnectedRelays();
      if (connectedRelays.length > 0) {
        if (this.debug) console.log(`Found ${connectedRelays.length} connected relays, proceeding with history fetch`);
        return;
      }
      if (this.debug) console.log("Waiting for relays to connect before fetching history...");
      await new Promise((resolve) => setTimeout(resolve, 500));
    }
    console.warn("Timeout waiting for relay connections, proceeding with history fetch anyway");
  }
  async autoAddContact(npub, isUnread = false) {
    try {
      const existingContacts = await contactRepo.getContacts();
      const contactExists = existingContacts.some((contact) => contact.npub === npub);
      if (!contactExists) {
        await profileResolver.resolveProfile(npub, true);
        const now = Date.now();
        const lastReadAt = isUnread ? 0 : now;
        const lastActivityAt = now;
        await contactRepo.addContact(npub, lastReadAt, lastActivityAt);
        if (this.debug) console.log(`Auto-added new contact: ${npub}`);
      }
    } catch (error) {
      console.error("Failed to auto-add contact:", error);
    }
  }
}
const messagingService = new MessagingService();
function callHapticsSafely(invoke) {
  if (!isAndroidCapacitorShell()) {
    return;
  }
  try {
    const result = invoke();
    if (result && typeof result.catch === "function") {
      result.catch(() => {
      });
    }
  } catch {
  }
}
function hapticLightImpact() {
  callHapticsSafely(() => Haptics.impact({ style: ImpactStyle.Light }));
}
class MediaServerSettingsService {
  async updateSettings(mediaServers) {
    const currentUserData = get(currentUser);
    if (currentUserData) {
      const profile = await profileRepo.getProfileIgnoreTTL(currentUserData.npub);
      await profileRepo.cacheProfile(
        currentUserData.npub,
        profile?.metadata,
        {
          messagingRelays: profile?.messagingRelays ?? [],
          mediaServers
        },
        void 0
      );
    }
    return await this.publishServerList(mediaServers);
  }
  async publishServerList(mediaServers) {
    const currentSigner = get(signer);
    const currentUserData = get(currentUser);
    if (!currentSigner || !currentUserData) {
      throw new Error("User not authenticated");
    }
    const tags = [];
    const seen = /* @__PURE__ */ new Set();
    for (const server of mediaServers) {
      const normalized = normalizeBlossomServerUrl(server);
      if (!normalized || seen.has(normalized)) continue;
      seen.add(normalized);
      tags.push(["server", normalized]);
    }
    const event = {
      kind: 10063,
      tags,
      content: "",
      created_at: Math.floor(Date.now() / 1e3)
    };
    const signedEvent = await currentSigner.signEvent(event);
    const profile = await profileRepo.getProfileIgnoreTTL(currentUserData.npub);
    const messagingRelays = profile?.messagingRelays ?? [];
    const allRelays = /* @__PURE__ */ new Set([
      ...getDiscoveryRelays(),
      getBlasterRelayUrl(),
      ...connectionManager.getAllRelayHealth().map((h) => h.url),
      ...messagingRelays
    ]);
    let attempted = 0;
    let succeeded = 0;
    for (const relayUrl of allRelays) {
      attempted++;
      try {
        let relay = connectionManager.getRelayHealth(relayUrl)?.relay;
        if (!relay) {
          try {
            relay = await Relay.connect(relayUrl);
          } catch {
            console.warn(`Could not connect to ${relayUrl} to publish media servers`);
            continue;
          }
        }
        const relayAny = relay;
        if (!relayAny.onauth) {
          relayAny.onauth = async (eventTemplate) => {
            return await currentSigner.signEvent(eventTemplate);
          };
        }
        try {
          await relay.publish(signedEvent);
        } catch (e) {
          const message = e?.message || String(e);
          if (message.startsWith("auth-required")) {
            connectionManager.markRelayAuthRequired?.(relayUrl);
            if (connectionManager.getRelayHealth(relayUrl)) {
              await connectionManager.authenticateRelay(relayUrl);
            } else if (relayAny.auth && relayAny.onauth) {
              await relayAny.auth(relayAny.onauth);
            }
            await relay.publish(signedEvent);
          } else {
            throw e;
          }
        }
        console.log(`Published media server list to ${relayUrl}`);
        succeeded++;
      } catch (e) {
        console.error(`Failed to publish media server list to ${relayUrl}:`, e);
      }
    }
    return {
      attempted,
      succeeded,
      failed: attempted - succeeded
    };
  }
}
const mediaServerSettingsService = new MediaServerSettingsService();
async function ensureDefaultBlossomServersForCurrentUser() {
  const user = get(currentUser);
  if (!user?.npub) {
    throw new Error("Not authenticated");
  }
  const profile = await profileRepo.getProfileIgnoreTTL(user.npub);
  const servers = profile?.mediaServers ?? [];
  const defaultServers = getDefaultBlossomServers();
  if (servers.length > 0) {
    return { servers, didSetDefaults: false };
  }
  await mediaServerSettingsService.updateSettings([...defaultServers]);
  const refreshed = await profileRepo.getProfileIgnoreTTL(user.npub);
  return {
    servers: refreshed?.mediaServers ?? [...defaultServers],
    didSetDefaults: true
  };
}
function MediaUploadButton($$renderer, $$props) {
  $$renderer.component(($$renderer2) => {
    var $$store_subs;
    let {
      inline,
      variant: variantProp = "default"
    } = $$props;
    const variant = inline === true ? "chat" : inline === false ? "default" : variantProp;
    typeof window !== "undefined" && isAndroidNative();
    typeof window !== "undefined" && isMobileWeb();
    $$renderer2.push(`<div class="relative"><button type="button"${attr_class(clsx(variant === "chat" ? "flex-shrink-0 h-8 w-8 flex items-center justify-center text-gray-600 dark:text-gray-300 hover:text-gray-800 dark:hover:text-gray-100 disabled:opacity-50 disabled:cursor-not-allowed relative overflow-hidden active:scale-90 transition-transform duration-100 ease-out" : "flex-shrink-0 h-11 w-11 p-0 flex items-center justify-center rounded-full bg-[rgb(var(--color-lavender-rgb)/0.20)] dark:bg-[rgb(var(--color-lavender-rgb)/0.24)] text-[rgb(var(--color-text-rgb)/0.92)] shadow-sm hover:bg-[rgb(var(--color-lavender-rgb)/0.26)] dark:hover:bg-[rgb(var(--color-lavender-rgb)/0.30)] hover:shadow active:bg-[rgb(var(--color-lavender-rgb)/0.32)] dark:active:bg-[rgb(var(--color-lavender-rgb)/0.36)] disabled:opacity-50 disabled:pointer-events-none relative overflow-hidden transition-all duration-200 ease-out"))}${attr("title", store_get($$store_subs ??= {}, "$t", $format)("chat.mediaMenu.uploadMediaTooltip"))}><svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="text-gray-600 dark:text-gray-300"><path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"></path><polyline points="17 8 12 3 7 8"></polyline><line x1="12" y1="3" x2="12" y2="15"></line></svg></button> `);
    {
      $$renderer2.push("<!--[!-->");
    }
    $$renderer2.push(`<!--]--></div>`);
    if ($$store_subs) unsubscribe_stores($$store_subs);
  });
}
function CircularProgress($$renderer, $$props) {
  let { size = 48, strokeWidth = 4, class: className = "" } = $$props;
  $$renderer.push(`<div${attr_class(`circular-progress ${className}`, "svelte-1joswrh")}${attr_style(`width: ${stringify(size)}px; height: ${stringify(size)}px;`)} role="progressbar" aria-label="Loading"><svg viewBox="0 0 48 48" class="svelte-1joswrh"><circle cx="24" cy="24"${attr("r", 20)} fill="none"${attr("stroke-width", strokeWidth)} class="svelte-1joswrh"></circle></svg></div>`);
}
const imageViewerState = writable({
  url: null,
  originalUrl: null,
  fitToScreen: true
});
function openImageViewer(url, originalUrl) {
  imageViewerState.set({
    url,
    originalUrl: originalUrl ?? url,
    fitToScreen: true
  });
}
function clamp01(value) {
  if (!isFinite(value)) {
    return 0;
  }
  return Math.min(1, Math.max(0, value));
}
function downsamplePeaks(peaks, targetCount) {
  if (peaks.length === 0) {
    return [];
  }
  if (peaks.length === targetCount) {
    return peaks.map(clamp01);
  }
  const bucketSize = peaks.length / targetCount;
  const result = [];
  for (let i = 0; i < targetCount; i++) {
    const start = Math.floor(i * bucketSize);
    const end = Math.min(peaks.length, Math.floor((i + 1) * bucketSize));
    let max = 0;
    for (let j = start; j < end; j++) {
      max = Math.max(max, peaks[j] ?? 0);
    }
    result.push(clamp01(max));
  }
  return result;
}
function WaveformBars($$renderer, $$props) {
  $$renderer.component(($$renderer2) => {
    let {
      peaks = [],
      barCount = 60,
      progress = 0,
      heightPx = 32,
      seekable = false,
      onSeek,
      playedClass = "bg-blue-500/90 dark:bg-blue-400/90",
      unplayedClass = "bg-blue-300/60 dark:bg-blue-200/50"
    } = $$props;
    const normalizedPeaks = (() => {
      const normalized = peaks.map((p) => clamp01(p));
      if (barCount <= 0) {
        return [];
      }
      if (normalized.length === barCount) {
        return normalized;
      }
      if (normalized.length > barCount) {
        return normalized.slice(normalized.length - barCount);
      }
      const padValue = normalized.length > 0 ? normalized[0] ?? 0.2 : 0.2;
      const padding = Array.from({ length: barCount - normalized.length }, () => padValue);
      return [...padding, ...normalized];
    })();
    const clampedProgress = clamp01(progress);
    const playedCount = Math.round(clampedProgress * normalizedPeaks.length);
    $$renderer2.push(`<div${attr_class(`flex items-center justify-between gap-[2px] w-full ${seekable ? "cursor-pointer touch-none select-none" : ""}`)}${attr_style(`height: ${heightPx}px`)}><!--[-->`);
    const each_array = ensure_array_like(normalizedPeaks);
    for (let i = 0, $$length = each_array.length; i < $$length; i++) {
      let peak = each_array[i];
      $$renderer2.push(`<div class="flex-1 min-w-[2px] max-w-[4px]"><div${attr_class(`w-full rounded-full transition-colors ${i < playedCount ? playedClass : unplayedClass}`)}${attr_style(`height: ${Math.max(2, peak * heightPx)}px`)}></div></div>`);
    }
    $$renderer2.push(`<!--]--></div>`);
  });
}
function AudioWaveformPlayer($$renderer, $$props) {
  $$renderer.component(($$renderer2) => {
    let { url, isOwn = false } = $$props;
    const BAR_COUNT = 40;
    let isLoading = true;
    let peaks = [];
    const progress = 0;
    function handleSeek(nextProgress) {
      {
        return;
      }
    }
    function formatTime(seconds) {
      {
        return "0:00";
      }
    }
    $$renderer2.push(`<div${attr_class(`flex items-center gap-3 px-3 py-1.5 rounded-xl border text-[11px] min-w-[160px] max-w-full bg-white/20 dark:bg-slate-800/50 md:bg-white/10 md:dark:bg-slate-800/30 md:backdrop-blur-sm border-gray-200/50 dark:border-slate-700/50 transition-colors ${isOwn ? "text-blue-50" : "text-gray-900 dark:text-slate-100"}`)}><button type="button" class="flex-shrink-0 w-9 h-9 rounded-full flex items-center justify-center bg-blue-500 text-white shadow-sm hover:bg-blue-600 disabled:opacity-60"${attr("aria-label", "Play audio")}${attr("disabled", isLoading, true)}>`);
    {
      $$renderer2.push("<!--[!-->");
      $$renderer2.push(`<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><polygon points="8 5 19 12 8 19 8 5"></polygon></svg>`);
    }
    $$renderer2.push(`<!--]--></button> <div class="flex-1 flex flex-col gap-1 min-w-0">`);
    WaveformBars($$renderer2, {
      peaks,
      barCount: BAR_COUNT,
      heightPx: 32,
      progress,
      seekable: !isLoading,
      onSeek: handleSeek,
      playedClass: "bg-blue-500/90 dark:bg-blue-400/90",
      unplayedClass: "bg-blue-300/50 dark:bg-blue-200/40"
    });
    $$renderer2.push(`<!----> `);
    {
      $$renderer2.push("<!--[-->");
      $$renderer2.push(`<div class="typ-meta text-[10px] text-gray-500 dark:text-slate-400">Loading waveform…</div>`);
    }
    $$renderer2.push(`<!--]--></div> <div class="flex-shrink-0 text-[10px] tabular-nums text-right min-w-[56px]">`);
    {
      $$renderer2.push("<!--[!-->");
      $$renderer2.push(`${escape_html(formatTime())}`);
    }
    $$renderer2.push(`<!--]--></div></div> <audio${attr("src", url)} preload="none" class="sr-only" style="width: 1px; height: 1px; opacity: 0; pointer-events: none;"></audio>`);
  });
}
function extractYouTubeVideoId(url) {
  let parsed;
  try {
    parsed = new URL(url);
  } catch {
    return null;
  }
  const host = parsed.hostname.toLowerCase();
  const pathname = parsed.pathname;
  let candidate = null;
  if (host === "youtu.be" || host.endsWith(".youtu.be")) {
    const segments = pathname.split("/").filter(Boolean);
    candidate = segments[0] ?? null;
  } else if (host === "youtube.com" || host.endsWith(".youtube.com") || host === "m.youtube.com" || host === "music.youtube.com") {
    if (pathname === "/watch") {
      candidate = parsed.searchParams.get("v");
    } else if (pathname.startsWith("/shorts/")) {
      const parts = pathname.split("/").filter(Boolean);
      candidate = parts[1] ?? null;
    } else if (pathname.startsWith("/embed/")) {
      const parts = pathname.split("/").filter(Boolean);
      candidate = parts[1] ?? null;
    }
  }
  if (!candidate) {
    return null;
  }
  const cleaned = candidate.split("?")[0]?.split("&")[0]?.trim() ?? "";
  if (!/^[a-zA-Z0-9_-]{11}$/.test(cleaned)) {
    return null;
  }
  return cleaned;
}
function isYouTubeUrl(url) {
  return extractYouTubeVideoId(url) !== null;
}
function MessageContent($$renderer, $$props) {
  $$renderer.component(($$renderer2) => {
    var $$store_subs;
    let {
      content,
      highlight = void 0,
      isOwn = false,
      onImageClick,
      fileUrl = void 0,
      fileType = void 0,
      fileEncryptionAlgorithm = void 0,
      fileKey = void 0,
      fileNonce = void 0,
      location = void 0
    } = $$props;
    const urlRegex = /(https?:\/\/[^\s]+)/g;
    function isImage(url) {
      try {
        const u = new URL(url);
        return /\.(jpg|jpeg|png|gif|webp|svg)$/i.test(u.pathname);
      } catch {
        return false;
      }
    }
    function isVideo(url) {
      try {
        const u = new URL(url);
        return /\.(mp4|webm|mov|ogg)$/i.test(u.pathname);
      } catch {
        return false;
      }
    }
    function isAudio(url) {
      try {
        const u = new URL(url);
        return /\.mp3$/i.test(u.pathname);
      } catch {
        return false;
      }
    }
    function isImageMime(mime) {
      return !!mime && mime.startsWith("image/");
    }
    function isVideoMime(mime) {
      return !!mime && mime.startsWith("video/");
    }
    function isAudioMime(mime) {
      return !!mime && mime.startsWith("audio/");
    }
    function parseMarkdown(text) {
      text = text.replace(/^> (.+)$/gm, '<div class="border-l-2 border-gray-300 pl-3 italic">$1</div>');
      text = text.replace(/~~([^~]+)~~/g, "<del>$1</del>");
      text = text.replace(/\*\*([^*]+)\*\*/g, "<strong>$1</strong>");
      text = text.replace(/__([^_]+)__/g, "<strong>$1</strong>");
      text = text.replace(/\*([^*]+)\*/g, "<em>$1</em>");
      text = text.replace(/_([^_]+)_/g, "<em>$1</em>");
      return text;
    }
    const highlightNeedle = (highlight ?? "").trim();
    function buildOsmEmbedUrl(point) {
      const padding = 0.01;
      const left = point.longitude - padding;
      const right = point.longitude + padding;
      const bottom = point.latitude - padding;
      const top = point.latitude + padding;
      return `https://www.openstreetmap.org/export/embed.html?bbox=${left},${bottom},${right},${top}&layer=mapnik&marker=${point.latitude},${point.longitude}`;
    }
    function buildOsmOpenUrl(point) {
      return `https://www.openstreetmap.org/?mlat=${point.latitude}&mlon=${point.longitude}&zoom=15`;
    }
    const mapUrl = location ? buildOsmEmbedUrl(location) : null;
    const openMapUrl = location ? buildOsmOpenUrl(location) : null;
    function escapeRegExp(value) {
      return value.replace(/[.*+?^${}()|[\]\\]/g, "\\$&");
    }
    function applyHighlightToHtml(html2, needle) {
      if (!needle) {
        return html2;
      }
      if (typeof window === "undefined") {
        return html2;
      }
      try {
        const regex = new RegExp(escapeRegExp(needle), "gi");
        const parser = new DOMParser();
        const doc = parser.parseFromString(`<div>${html2}</div>`, "text/html");
        const root = doc.body.firstElementChild;
        if (!root) {
          return html2;
        }
        const walker = doc.createTreeWalker(root, NodeFilter.SHOW_TEXT);
        const textNodes = [];
        for (let node = walker.nextNode(); node; node = walker.nextNode()) {
          if (node.nodeType === Node.TEXT_NODE) {
            textNodes.push(node);
          }
        }
        const markClass = "bg-yellow-200/70 dark:bg-yellow-400/20 rounded px-0.5";
        for (const textNode of textNodes) {
          const text = textNode.nodeValue ?? "";
          regex.lastIndex = 0;
          if (!regex.test(text)) {
            continue;
          }
          regex.lastIndex = 0;
          const fragment = doc.createDocumentFragment();
          let lastIndex = 0;
          let match;
          while ((match = regex.exec(text)) !== null) {
            const start = match.index;
            const end = start + match[0].length;
            if (start > lastIndex) {
              fragment.appendChild(doc.createTextNode(text.slice(lastIndex, start)));
            }
            const mark = doc.createElement("mark");
            mark.setAttribute("class", markClass);
            mark.textContent = text.slice(start, end);
            fragment.appendChild(mark);
            lastIndex = end;
          }
          if (lastIndex < text.length) {
            fragment.appendChild(doc.createTextNode(text.slice(lastIndex)));
          }
          textNode.parentNode?.replaceChild(fragment, textNode);
        }
        return root.innerHTML;
      } catch {
        return html2;
      }
    }
    function getFirstNonMediaUrl(text) {
      const matches = text.match(urlRegex) ?? [];
      for (const candidate of matches) {
        if (!isImage(candidate) && !isVideo(candidate) && !isAudio(candidate) && !isYouTubeUrl(candidate)) {
          return candidate;
        }
      }
      return null;
    }
    function getUrlPreviewsEnabled() {
      if (typeof window === "undefined") {
        return true;
      }
      try {
        const raw = localStorage.getItem("nospeak-settings");
        if (!raw) {
          return true;
        }
        const parsed = JSON.parse(raw);
        if (typeof parsed.urlPreviewsEnabled === "boolean") {
          return parsed.urlPreviewsEnabled;
        }
        return true;
      } catch {
        return true;
      }
    }
    let parts = content.split(urlRegex);
    (() => {
      for (let i = 0; i < parts.length; i++) {
        const part = parts[i];
        if (!part.match(/^https?:\/\//)) {
          continue;
        }
        const videoId = extractYouTubeVideoId(part);
        if (videoId) {
          return { partIndex: i, videoId };
        }
      }
      return null;
    })();
    const singleEmojiRegex = new RegExp("^(\\p{Emoji_Presentation}|\\p{Emoji}\\uFE0F)$", "u");
    let isSingleEmoji = singleEmojiRegex.test(content.trim());
    getFirstNonMediaUrl(content);
    let isVisible = false;
    typeof window !== "undefined" && getUrlPreviewsEnabled() && isVisible;
    const NOSPEAK_INTERNAL_MEDIA_ORIGIN = "https://nospeak.chat";
    function isLegacyNospeakUserMediaUrl(url) {
      try {
        const u = new URL(url);
        return u.origin === NOSPEAK_INTERNAL_MEDIA_ORIGIN && u.pathname.startsWith("/api/user_media/");
      } catch {
        return false;
      }
    }
    $$renderer2.push(`<div${attr_class(`whitespace-pre-wrap break-words leading-relaxed ${isSingleEmoji ? "text-4xl" : ""}`)}>`);
    if (location) {
      $$renderer2.push("<!--[-->");
      $$renderer2.push(`<div class="my-1"><div class="flex items-center gap-2 typ-meta text-xs font-semibold text-gray-600 dark:text-slate-300 leading-none"><svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M21 10c0 7-9 13-9 13s-9-6-9-13a9 9 0 0118 0z"></path><circle cx="12" cy="10" r="3"></circle></svg> <span>${escape_html(store_get($$store_subs ??= {}, "$t", $format)("modals.locationPreview.title"))}</span></div> `);
      if (mapUrl) {
        $$renderer2.push("<!--[-->");
        $$renderer2.push(`<div class="rounded-xl overflow-hidden bg-gray-100/80 dark:bg-slate-800/80 border border-gray-200/60 dark:border-slate-700/60"><iframe${attr("src", mapUrl)} width="100%" height="220" frameborder="0" class="w-full" title="Location map"></iframe></div>`);
      } else {
        $$renderer2.push("<!--[!-->");
      }
      $$renderer2.push(`<!--]--> `);
      if (openMapUrl) {
        $$renderer2.push("<!--[-->");
        $$renderer2.push(`<a${attr("href", openMapUrl)} target="_blank" rel="noopener noreferrer" class="typ-meta text-xs underline hover:opacity-80">${escape_html(store_get($$store_subs ??= {}, "$t", $format)("modals.locationPreview.openInOpenStreetMap"))}</a>`);
      } else {
        $$renderer2.push("<!--[!-->");
      }
      $$renderer2.push(`<!--]--></div>`);
    } else {
      $$renderer2.push("<!--[!-->");
      if (fileUrl && fileEncryptionAlgorithm === "aes-gcm" && fileKey && fileNonce) {
        $$renderer2.push("<!--[-->");
        $$renderer2.push(`<div class="space-y-2">`);
        if (isLegacyNospeakUserMediaUrl(fileUrl)) {
          $$renderer2.push("<!--[-->");
          $$renderer2.push(`<div class="my-1 px-3 py-2 rounded-xl bg-gray-100/70 dark:bg-slate-800/60 border border-gray-200/60 dark:border-slate-700/60"><div class="typ-meta text-xs text-gray-600 dark:text-slate-300">${escape_html(store_get($$store_subs ??= {}, "$t", $format)("chat.mediaUnavailable"))}</div></div>`);
        } else {
          $$renderer2.push("<!--[!-->");
          {
            $$renderer2.push("<!--[!-->");
            {
              $$renderer2.push("<!--[!-->");
              {
                $$renderer2.push("<!--[!-->");
              }
              $$renderer2.push(`<!--]-->`);
            }
            $$renderer2.push(`<!--]-->`);
          }
          $$renderer2.push(`<!--]-->`);
        }
        $$renderer2.push(`<!--]--></div>`);
      } else {
        $$renderer2.push("<!--[!-->");
        if (fileUrl) {
          $$renderer2.push("<!--[-->");
          $$renderer2.push(`<div class="space-y-2">`);
          if (isLegacyNospeakUserMediaUrl(fileUrl)) {
            $$renderer2.push("<!--[-->");
            $$renderer2.push(`<div class="my-1 px-3 py-2 rounded-xl bg-gray-100/70 dark:bg-slate-800/60 border border-gray-200/60 dark:border-slate-700/60"><div class="typ-meta text-xs text-gray-600 dark:text-slate-300">${escape_html(store_get($$store_subs ??= {}, "$t", $format)("chat.mediaUnavailable"))}</div></div>`);
          } else {
            $$renderer2.push("<!--[!-->");
            if (isImageMime(fileType) || isImage(fileUrl)) {
              $$renderer2.push("<!--[-->");
              if (onImageClick) {
                $$renderer2.push("<!--[-->");
                $$renderer2.push(`<button type="button" class="block my-1 cursor-zoom-in"><img${attr("src", fileUrl)} alt="Attachment" class="max-w-full rounded max-h-[300px] object-contain" loading="lazy" onload="this.__e=event"/></button>`);
              } else {
                $$renderer2.push("<!--[!-->");
                $$renderer2.push(`<a${attr("href", fileUrl)} target="_blank" rel="noopener noreferrer" class="block my-1"><img${attr("src", fileUrl)} alt="Attachment" class="max-w-full rounded max-h-[300px] object-contain" loading="lazy" onload="this.__e=event"/></a>`);
              }
              $$renderer2.push(`<!--]-->`);
            } else {
              $$renderer2.push("<!--[!-->");
              if (isVideoMime(fileType) || isVideo(fileUrl)) {
                $$renderer2.push("<!--[-->");
                $$renderer2.push(`<div class="my-1"><video controls${attr("src", fileUrl)} class="max-w-full rounded max-h-[300px]" preload="metadata"></video></div>`);
              } else {
                $$renderer2.push("<!--[!-->");
                if (isAudioMime(fileType) || isAudio(fileUrl)) {
                  $$renderer2.push("<!--[-->");
                  $$renderer2.push(`<div class="mb-1">`);
                  AudioWaveformPlayer($$renderer2, { url: fileUrl, isOwn });
                  $$renderer2.push(`<!----></div>`);
                } else {
                  $$renderer2.push("<!--[!-->");
                  $$renderer2.push(`<a${attr("href", fileUrl)} target="_blank" rel="noopener noreferrer" class="underline hover:opacity-80 break-all">Download attachment</a>`);
                }
                $$renderer2.push(`<!--]-->`);
              }
              $$renderer2.push(`<!--]-->`);
            }
            $$renderer2.push(`<!--]-->`);
          }
          $$renderer2.push(`<!--]--></div>`);
        } else {
          $$renderer2.push("<!--[!-->");
          $$renderer2.push(`<!--[-->`);
          const each_array = ensure_array_like(parts);
          for (let i = 0, $$length = each_array.length; i < $$length; i++) {
            let part = each_array[i];
            if (part.match(/^https?:\/\//)) {
              $$renderer2.push("<!--[-->");
              {
                $$renderer2.push("<!--[!-->");
                if (isLegacyNospeakUserMediaUrl(part)) {
                  $$renderer2.push("<!--[-->");
                  $$renderer2.push(`<div class="my-1 px-3 py-2 rounded-xl bg-gray-100/70 dark:bg-slate-800/60 border border-gray-200/60 dark:border-slate-700/60"><div class="typ-meta text-xs text-gray-600 dark:text-slate-300">${escape_html(store_get($$store_subs ??= {}, "$t", $format)("chat.mediaUnavailable"))}</div></div>`);
                } else {
                  $$renderer2.push("<!--[!-->");
                  if (isImage(part)) {
                    $$renderer2.push("<!--[-->");
                    if (onImageClick) {
                      $$renderer2.push("<!--[-->");
                      $$renderer2.push(`<button type="button" class="block my-1 cursor-zoom-in"><img${attr("src", part)} alt="Attachment" class="max-w-full rounded max-h-[300px] object-contain" loading="lazy" onload="this.__e=event"/></button>`);
                    } else {
                      $$renderer2.push("<!--[!-->");
                      $$renderer2.push(`<a${attr("href", part)} target="_blank" rel="noopener noreferrer" class="block my-1"><img${attr("src", part)} alt="Attachment" class="max-w-full rounded max-h-[300px] object-contain" loading="lazy" onload="this.__e=event"/></a>`);
                    }
                    $$renderer2.push(`<!--]-->`);
                  } else {
                    $$renderer2.push("<!--[!-->");
                    if (isVideo(part)) {
                      $$renderer2.push("<!--[-->");
                      $$renderer2.push(`<div class="my-1"><video controls${attr("src", part)} class="max-w-full rounded max-h-[300px]" preload="metadata"></video></div>`);
                    } else {
                      $$renderer2.push("<!--[!-->");
                      if (isAudio(part)) {
                        $$renderer2.push("<!--[-->");
                        $$renderer2.push(`<div class="mb-1">`);
                        AudioWaveformPlayer($$renderer2, { url: part, isOwn });
                        $$renderer2.push(`<!----></div>`);
                      } else {
                        $$renderer2.push("<!--[!-->");
                        $$renderer2.push(`<a${attr("href", part)} target="_blank" rel="noopener noreferrer" class="underline hover:opacity-80 break-all">${escape_html(part)}</a>`);
                      }
                      $$renderer2.push(`<!--]-->`);
                    }
                    $$renderer2.push(`<!--]-->`);
                  }
                  $$renderer2.push(`<!--]-->`);
                }
                $$renderer2.push(`<!--]-->`);
              }
              $$renderer2.push(`<!--]-->`);
            } else {
              $$renderer2.push("<!--[!-->");
              $$renderer2.push(`<span>${html(applyHighlightToHtml(parseMarkdown(part), highlightNeedle))}</span>`);
            }
            $$renderer2.push(`<!--]-->`);
          }
          $$renderer2.push(`<!--]--> `);
          {
            $$renderer2.push("<!--[!-->");
          }
          $$renderer2.push(`<!--]-->`);
        }
        $$renderer2.push(`<!--]-->`);
      }
      $$renderer2.push(`<!--]-->`);
    }
    $$renderer2.push(`<!--]--></div>`);
    if ($$store_subs) unsubscribe_stores($$store_subs);
  });
}
function ContextMenu($$renderer, $$props) {
  $$renderer.component(($$renderer2) => {
    var $$store_subs;
    let {
      isOpen = false
    } = $$props;
    if (isOpen) {
      $$renderer2.push("<!--[-->");
      $$renderer2.push(`<div class="context-menu fixed bg-white/80 dark:bg-slate-900/80 backdrop-blur-xl border border-gray-200 dark:border-slate-700 rounded-lg shadow-xl py-1 z-[9999] min-w-[140px] outline-none"><div class="flex px-2 pt-1 pb-1 gap-1 border-b border-gray-200/70 dark:border-slate-700/70"><!--[-->`);
      const each_array = ensure_array_like(["👍", "👎", "❤️", "😂"]);
      for (let $$index = 0, $$length = each_array.length; $$index < $$length; $$index++) {
        let emoji = each_array[$$index];
        $$renderer2.push(`<button type="button" class="flex-1 px-1 py-1 rounded-md hover:bg-gray-100/70 dark:hover:bg-slate-700/70 text-lg text-center">${escape_html(emoji)}</button>`);
      }
      $$renderer2.push(`<!--]--></div> `);
      {
        $$renderer2.push("<!--[!-->");
      }
      $$renderer2.push(`<!--]--> <button class="w-full text-left px-4 py-2 hover:bg-gray-100/50 dark:hover:bg-slate-700/50 text-sm dark:text-white transition-colors">${escape_html(store_get($$store_subs ??= {}, "$t", $format)("chat.contextMenu.cite"))}</button> <button class="w-full text-left px-4 py-2 hover:bg-gray-100/50 dark:hover:bg-slate-700/50 text-sm dark:text-white transition-colors">${escape_html(store_get($$store_subs ??= {}, "$t", $format)("chat.contextMenu.copy"))}</button></div>`);
    } else {
      $$renderer2.push("<!--[!-->");
    }
    $$renderer2.push(`<!--]-->`);
    if ($$store_subs) unsubscribe_stores($$store_subs);
  });
}
function MessageReactions($$renderer, $$props) {
  $$renderer.component(($$renderer2) => {
    $$renderer2.push(`<div class="w-full h-0">`);
    {
      $$renderer2.push("<!--[!-->");
    }
    $$renderer2.push(`<!--]--></div>`);
  });
}
const VOICE_MESSAGE_MAX_DURATION_MS = 3 * 60 * 1e3;
const PREFERRED_OPUS_MIME_TYPES = [
  "audio/webm;codecs=opus",
  "audio/ogg;codecs=opus",
  "audio/webm"
];
function pickSupportedOpusMimeType(params) {
  const candidates = params.candidates ?? PREFERRED_OPUS_MIME_TYPES;
  for (const candidate of candidates) {
    if (params.isTypeSupported(candidate)) {
      return candidate;
    }
  }
  return null;
}
function getSupportedOpusMimeType() {
  if (typeof window === "undefined") {
    return null;
  }
  const mediaRecorder = window.MediaRecorder;
  if (!mediaRecorder || typeof mediaRecorder.isTypeSupported !== "function") {
    return null;
  }
  return pickSupportedOpusMimeType({
    isTypeSupported: (mimeType) => mediaRecorder.isTypeSupported(mimeType)
  });
}
function isVoiceRecordingSupported() {
  if (typeof window === "undefined") {
    return false;
  }
  const hasGetUserMedia = typeof navigator !== "undefined" && !!navigator.mediaDevices && typeof navigator.mediaDevices.getUserMedia === "function";
  if (!hasGetUserMedia) {
    return false;
  }
  return getSupportedOpusMimeType() !== null;
}
function formatDurationMs(ms) {
  if (!isFinite(ms) || ms <= 0) {
    return "0:00";
  }
  const totalSeconds = Math.floor(ms / 1e3);
  const mins = Math.floor(totalSeconds / 60);
  const secs = totalSeconds % 60;
  const padded = secs < 10 ? `0${secs}` : `${secs}`;
  return `${mins}:${padded}`;
}
const AndroidMicrophone = Capacitor.getPlatform() === "android" ? registerPlugin("AndroidMicrophone") : null;
function VoiceMessageSheet($$renderer, $$props) {
  $$renderer.component(($$renderer2) => {
    var $$store_subs;
    const translate = (key, vars) => get($format)(key, vars);
    let {
      isOpen,
      onCancel,
      onClose,
      onSend,
      maxDurationMs = VOICE_MESSAGE_MAX_DURATION_MS
    } = $$props;
    const effectiveMaxDurationMs = isFinite(maxDurationMs) && maxDurationMs > 0 ? maxDurationMs : VOICE_MESSAGE_MAX_DURATION_MS;
    const useNativeRecording = Capacitor.getPlatform() === "android" && AndroidMicrophone !== null;
    let recordingState = "idle";
    let error = null;
    let stream = null;
    let mediaRecorder = null;
    let mimeType = null;
    let chunks = [];
    let nativeFilePath = null;
    const WAVEFORM_BAR_COUNT = 40;
    let rawPeaks = [];
    let previewDuration = 0;
    let previewCurrentTime = 0;
    let startedAtMs = null;
    let pausedAtMs = null;
    let pausedTotalMs = 0;
    let elapsedMs = 0;
    let recordedBlob = null;
    let recordedUrl = null;
    let previewAudio = null;
    let isPreviewPlaying = false;
    let isStopping = false;
    function stopTracks() {
      for (const track of stream?.getTracks() ?? []) {
        try {
          track.stop();
        } catch {
        }
      }
      stream = null;
    }
    function cleanupPreview() {
      try {
        previewAudio?.pause();
      } catch {
      }
      if (recordedUrl && recordedUrl.startsWith("blob:")) {
        URL.revokeObjectURL(recordedUrl);
      }
      recordedUrl = null;
      recordedBlob = null;
      previewDuration = 0;
      previewCurrentTime = 0;
      isPreviewPlaying = false;
    }
    async function cleanupNativeListeners() {
    }
    async function cleanup() {
      cleanupPreview();
      await cleanupNativeListeners();
      nativeFilePath = null;
      if (mediaRecorder && mediaRecorder.state !== "inactive") {
        try {
          mediaRecorder.stop();
        } catch {
        }
      }
      mediaRecorder = null;
      stopTracks();
      chunks = [];
      mimeType = null;
      startedAtMs = null;
      pausedAtMs = null;
      pausedTotalMs = 0;
      elapsedMs = 0;
      isStopping = false;
      recordingState = "idle";
      error = null;
    }
    function computeElapsedMs(nowMs) {
      if (startedAtMs === null) {
        return 0;
      }
      if (recordingState === "paused" && pausedAtMs !== null) {
        return pausedAtMs - startedAtMs - pausedTotalMs;
      }
      return nowMs - startedAtMs - pausedTotalMs;
    }
    async function pauseRecording() {
      if (recordingState !== "recording") {
        return;
      }
      pausedAtMs = Date.now();
      if (useNativeRecording && AndroidMicrophone) {
        try {
          await AndroidMicrophone.pauseRecording();
          recordingState = "paused";
          console.info("[voice] native recording paused");
        } catch (e) {
          console.error("[voice] pauseRecording failed", e);
        }
      } else if (mediaRecorder && mediaRecorder.state === "recording") {
        try {
          mediaRecorder.pause();
          recordingState = "paused";
          void refreshPausedPreview();
        } catch {
        }
      }
    }
    async function resumeRecording() {
      if (recordingState !== "paused") {
        return;
      }
      cleanupPreview();
      if (pausedAtMs !== null) {
        pausedTotalMs += Date.now() - pausedAtMs;
      }
      pausedAtMs = null;
      if (useNativeRecording && AndroidMicrophone) {
        try {
          await AndroidMicrophone.resumeRecording();
          recordingState = "recording";
          console.info("[voice] native recording resumed");
        } catch (e) {
          console.error("[voice] resumeRecording failed", e);
        }
      } else if (mediaRecorder && mediaRecorder.state === "paused") {
        try {
          mediaRecorder.resume();
          recordingState = "recording";
        } catch {
        }
      }
    }
    async function stopRecording() {
      if (isStopping) {
        return;
      }
      isStopping = true;
      const finalElapsed = computeElapsedMs(Date.now());
      elapsedMs = Math.min(finalElapsed, effectiveMaxDurationMs);
      previewDuration = elapsedMs / 1e3;
      previewCurrentTime = 0;
      if (useNativeRecording && AndroidMicrophone) {
        await stopNativeRecording();
      } else {
        await stopWebRecording();
      }
      isStopping = false;
    }
    async function stopNativeRecording() {
      if (!AndroidMicrophone) {
        return;
      }
      try {
        console.info("[voice] stopping native recording");
        const result = await AndroidMicrophone.stopRecording();
        console.info("[voice] native recording stopped", result);
        nativeFilePath = result.filePath;
        console.info("[voice] reading file via Filesystem");
        const fs = window.Capacitor?.Plugins?.Filesystem;
        if (!fs) {
          throw new Error("Filesystem plugin not available");
        }
        const fileResult = await fs.readFile({ path: result.filePath });
        const base64Data = fileResult.data;
        const binaryString = atob(base64Data);
        const bytes = new Uint8Array(binaryString.length);
        for (let i = 0; i < binaryString.length; i++) {
          bytes[i] = binaryString.charCodeAt(i);
        }
        recordedBlob = new Blob([bytes], { type: "audio/mp4" });
        recordedUrl = URL.createObjectURL(recordedBlob);
        recordingState = "ready";
        console.info("[voice] file read successfully, blob size:", recordedBlob.size);
        await cleanupNativeListeners();
      } catch (e) {
        console.error("[voice] stopNativeRecording failed", e);
        error = translate("chat.voiceMessage.failedToStop");
        recordingState = "error";
      }
    }
    async function stopWebRecording() {
      if (!mediaRecorder) {
        return;
      }
      if (mediaRecorder.state === "inactive") {
        return;
      }
      const blob = await new Promise((resolve) => {
        const recorder = mediaRecorder;
        const onStop = () => {
          recorder.removeEventListener("stop", onStop);
          resolve(new Blob(chunks, { type: mimeType ?? "audio/webm" }));
        };
        recorder.addEventListener("stop", onStop);
        try {
          recorder.stop();
        } catch {
          recorder.removeEventListener("stop", onStop);
          resolve(new Blob(chunks, { type: mimeType ?? "audio/webm" }));
        }
      });
      recordedBlob = blob;
      if (recordedUrl) {
        URL.revokeObjectURL(recordedUrl);
      }
      recordedUrl = URL.createObjectURL(blob);
      recordingState = "ready";
      stopTracks();
    }
    async function refreshPausedPreview() {
      if (!mediaRecorder || !mimeType || chunks.length === 0) {
        return;
      }
      cleanupPreview();
      const recorder = mediaRecorder;
      await new Promise((resolve) => {
        let resolved = false;
        const finish = (extraData) => {
          if (resolved) return;
          resolved = true;
          if (extraData && extraData.size > 0) {
            chunks.push(extraData);
          }
          const blob = new Blob(chunks, { type: mimeType ?? "audio/webm" });
          recordedUrl = URL.createObjectURL(blob);
          resolve();
        };
        const timeoutId = window.setTimeout(() => finish(), 300);
        const handleData = (e) => {
          recorder.removeEventListener("dataavailable", handleData);
          clearTimeout(timeoutId);
          finish(e.data);
        };
        recorder.addEventListener("dataavailable", handleData);
        try {
          recorder.requestData();
        } catch {
          recorder.removeEventListener("dataavailable", handleData);
          clearTimeout(timeoutId);
          finish();
        }
      });
    }
    async function handleSend() {
      if (recordingState === "recording" || recordingState === "paused") {
        await stopRecording();
      }
      if (recordingState !== "ready" || !recordedBlob) {
        return;
      }
      const file = new File([recordedBlob], `voice-${Date.now()}.bin`, { type: mimeType ?? recordedBlob.type ?? "audio/mp4" });
      onSend(file);
      if (useNativeRecording && AndroidMicrophone && nativeFilePath) {
        try {
          await AndroidMicrophone.deleteRecordingFile({ filePath: nativeFilePath });
          console.info("[voice] deleted native recording file");
        } catch (e) {
          console.warn("[voice] failed to delete native recording file", e);
        }
      }
      onClose();
    }
    function togglePreview() {
      {
        return;
      }
    }
    function handlePreviewSeek(nextProgress) {
      {
        return;
      }
    }
    onDestroy(() => {
      void cleanup();
    });
    const timerLabel = formatDurationMs(elapsedMs);
    const canPause = recordingState === "recording" && !useNativeRecording;
    const canDone = recordingState === "recording" && useNativeRecording;
    const canResume = recordingState === "paused" && !useNativeRecording;
    const canPreview = (useNativeRecording ? recordingState === "ready" : recordingState === "paused" || recordingState === "ready") && !!recordedUrl;
    const waveformPeaks = downsamplePeaks(rawPeaks, WAVEFORM_BAR_COUNT);
    const previewProgress = previewDuration > 0 ? clamp01(previewCurrentTime / previewDuration) : 0;
    if (isOpen) {
      $$renderer2.push("<!--[-->");
      $$renderer2.push(`<div class="fixed inset-0 z-[70] flex items-end md:items-center justify-center bg-black/40 px-0 md:px-4" role="dialog" aria-modal="true" tabindex="-1"><div class="relative w-full bg-white/95 dark:bg-slate-900/95 border border-gray-200/80 dark:border-slate-700/80 shadow-2xl backdrop-blur-xl p-4 space-y-4 rounded-t-3xl md:max-w-md md:rounded-2xl"><div class="flex items-center justify-between"><h2 class="typ-title dark:text-white">${escape_html(store_get($$store_subs ??= {}, "$t", $format)("chat.voiceMessage.title"))}</h2> <div class="typ-meta text-xs tabular-nums text-gray-600 dark:text-slate-300">${escape_html(timerLabel)} / ${escape_html(formatDurationMs(effectiveMaxDurationMs))}</div></div> `);
      if (error) {
        $$renderer2.push("<!--[-->");
        $$renderer2.push(`<div class="typ-body text-sm text-red-600 dark:text-red-300">${escape_html(error)}</div>`);
      } else {
        $$renderer2.push("<!--[!-->");
      }
      $$renderer2.push(`<!--]--> <div class="rounded-xl bg-gray-100/80 dark:bg-slate-800/80 border border-gray-200/60 dark:border-slate-700/60 px-4 py-4">`);
      WaveformBars($$renderer2, {
        peaks: waveformPeaks,
        barCount: WAVEFORM_BAR_COUNT,
        heightPx: 40,
        progress: canPreview ? previewProgress : 0,
        seekable: canPreview,
        onSeek: handlePreviewSeek,
        playedClass: "bg-blue-500/90 dark:bg-blue-400/90",
        unplayedClass: "bg-blue-300/60 dark:bg-blue-200/50"
      });
      $$renderer2.push(`<!----> `);
      if (canPreview) {
        $$renderer2.push("<!--[-->");
        $$renderer2.push(`<div class="mt-3 flex items-center justify-center gap-3">`);
        Button($$renderer2, {
          size: "icon",
          variant: "filled-tonal",
          onclick: togglePreview,
          "aria-label": isPreviewPlaying ? store_get($$store_subs ??= {}, "$t", $format)("chat.voiceMessage.pausePreviewAria") : store_get($$store_subs ??= {}, "$t", $format)("chat.voiceMessage.playPreviewAria"),
          children: ($$renderer3) => {
            if (isPreviewPlaying) {
              $$renderer3.push("<!--[-->");
              $$renderer3.push(`<svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><rect x="6" y="4" width="4" height="16"></rect><rect x="14" y="4" width="4" height="16"></rect></svg>`);
            } else {
              $$renderer3.push("<!--[!-->");
              $$renderer3.push(`<svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><polygon points="8 5 19 12 8 19 8 5"></polygon></svg>`);
            }
            $$renderer3.push(`<!--]-->`);
          },
          $$slots: { default: true }
        });
        $$renderer2.push(`<!----> <span class="typ-meta text-xs tabular-nums text-gray-600 dark:text-slate-300">${escape_html(formatDurationMs(previewCurrentTime * 1e3))} / ${escape_html(formatDurationMs(previewDuration * 1e3))}</span></div>`);
      } else {
        $$renderer2.push("<!--[!-->");
      }
      $$renderer2.push(`<!--]--></div>  <audio${attr("src", recordedUrl ?? "")} class="sr-only"></audio> <div class="flex items-center justify-between gap-2">`);
      Button($$renderer2, {
        onclick: onCancel,
        disabled: isStopping,
        children: ($$renderer3) => {
          $$renderer3.push(`<!---->${escape_html(store_get($$store_subs ??= {}, "$t", $format)("chat.voiceMessage.cancelButton"))}`);
        },
        $$slots: { default: true }
      });
      $$renderer2.push(`<!----> <div class="flex items-center gap-2">`);
      if (canPause) {
        $$renderer2.push("<!--[-->");
        Button($$renderer2, {
          onclick: () => void pauseRecording(),
          disabled: isStopping,
          children: ($$renderer3) => {
            $$renderer3.push(`<!---->${escape_html(store_get($$store_subs ??= {}, "$t", $format)("chat.voiceMessage.pauseButton"))}`);
          },
          $$slots: { default: true }
        });
      } else {
        $$renderer2.push("<!--[!-->");
        if (canDone) {
          $$renderer2.push("<!--[-->");
          Button($$renderer2, {
            onclick: () => void stopRecording(),
            disabled: isStopping,
            children: ($$renderer3) => {
              $$renderer3.push(`<!---->${escape_html(store_get($$store_subs ??= {}, "$t", $format)("chat.voiceMessage.doneButton"))}`);
            },
            $$slots: { default: true }
          });
        } else {
          $$renderer2.push("<!--[!-->");
          if (canResume) {
            $$renderer2.push("<!--[-->");
            Button($$renderer2, {
              onclick: () => void resumeRecording(),
              disabled: isStopping,
              children: ($$renderer3) => {
                $$renderer3.push(`<!---->${escape_html(store_get($$store_subs ??= {}, "$t", $format)("chat.voiceMessage.resumeButton"))}`);
              },
              $$slots: { default: true }
            });
          } else {
            $$renderer2.push("<!--[!-->");
          }
          $$renderer2.push(`<!--]-->`);
        }
        $$renderer2.push(`<!--]-->`);
      }
      $$renderer2.push(`<!--]--> `);
      Button($$renderer2, {
        variant: "primary",
        onclick: () => void handleSend(),
        disabled: isStopping || recordingState === "error",
        children: ($$renderer3) => {
          $$renderer3.push(`<!---->${escape_html(store_get($$store_subs ??= {}, "$t", $format)("chat.voiceMessage.sendButton"))}`);
        },
        $$slots: { default: true }
      });
      $$renderer2.push(`<!----></div></div></div></div>`);
    } else {
      $$renderer2.push("<!--[!-->");
    }
    $$renderer2.push(`<!--]-->`);
    if ($$store_subs) unsubscribe_stores($$store_subs);
  });
}
function isCaptionMessage(messages, index) {
  const msg = messages[index];
  if (!msg || msg.rumorKind !== 14 || !msg.parentRumorId) return false;
  const parentIndex = messages.findIndex(
    (m) => m.rumorId === msg.parentRumorId
  );
  if (parentIndex === -1) return false;
  const parent = messages[parentIndex];
  if (!parent || parent.rumorKind !== 15) return false;
  if (parent.direction !== msg.direction) return false;
  if (index !== parentIndex + 1) return false;
  return true;
}
function getCaptionForParent(messages, index) {
  const msg = messages[index];
  if (!msg || msg.rumorKind !== 15) return null;
  const nextIndex = index + 1;
  const next = messages[nextIndex];
  if (!next || next.rumorKind !== 14 || !next.parentRumorId) return null;
  if (next.parentRumorId !== msg.rumorId) return null;
  if (next.direction !== msg.direction) return null;
  return next;
}
Capacitor.getPlatform() === "android" ? registerPlugin("AndroidLocation") : null;
function ChatView($$renderer, $$props) {
  $$renderer.component(($$renderer2) => {
    var $$store_subs;
    let {
      messages = [],
      partnerNpub,
      isFetchingHistory = false,
      canRequestNetworkHistory = false,
      networkHistoryStatus = "idle",
      networkHistorySummary = null
    } = $$props;
    let inputText = "";
    let isSearchOpen = false;
    let searchQuery = "";
    let searchResults = [];
    let isSearchingHistory = false;
    let isSearchActive = searchQuery.trim().length >= 3;
    function cancelPendingSearch() {
      isSearchingHistory = false;
    }
    function closeSearch() {
      cancelPendingSearch();
      isSearchOpen = false;
      searchQuery = "";
      searchResults = [];
    }
    function openSearch() {
      isSearchOpen = true;
      setTimeout(
        () => {
        },
        0
      );
    }
    function toggleSearch() {
      if (isSearchOpen) {
        closeSearch();
        return;
      }
      openSearch();
    }
    function escapeHtml(value) {
      return value.replaceAll("&", "&amp;").replaceAll("<", "&lt;").replaceAll(">", "&gt;").replaceAll('"', "&quot;").replaceAll("'", "&#39;");
    }
    function escapeRegExp(value) {
      return value.replace(/[.*+?^${}()|[\]\\]/g, "\\$&");
    }
    function highlightPlainTextToHtml(text, query) {
      const needle = query.trim();
      if (!needle) {
        return escapeHtml(text);
      }
      const regex = new RegExp(escapeRegExp(needle), "gi");
      const markClass = "bg-yellow-200/70 dark:bg-yellow-400/20 rounded px-0.5";
      let result = "";
      let lastIndex = 0;
      let match;
      while ((match = regex.exec(text)) !== null) {
        const start = match.index;
        const end = start + match[0].length;
        result += escapeHtml(text.slice(lastIndex, start));
        result += `<mark class="${markClass}">${escapeHtml(text.slice(start, end))}</mark>`;
        lastIndex = end;
      }
      result += escapeHtml(text.slice(lastIndex));
      return result;
    }
    let partnerPicture = void 0;
    let myPicture = void 0;
    let isSending = false;
    let optimisticMessages = [];
    let displayMessages = isSearchActive ? searchResults : [...messages, ...optimisticMessages];
    let isDestroyed = false;
    function revokeOptimisticResources(msg) {
      if (typeof window === "undefined") {
        return;
      }
      if (msg.fileUrl && msg.fileUrl.startsWith("blob:")) {
        URL.revokeObjectURL(msg.fileUrl);
      }
    }
    function makeOptimisticEventId() {
      return `optimistic:${Date.now()}:${Math.random().toString(16).slice(2)}`;
    }
    onDestroy(() => {
      isDestroyed = true;
      cancelPendingSearch();
      for (const optimistic of optimisticMessages) {
        revokeOptimisticResources(optimistic);
      }
    });
    let currentTime = Date.now();
    let unreadSnapshotMessageIds = [];
    let activeHighlightMessageIds = [];
    let unreadSnapshotMessageSet = new Set(unreadSnapshotMessageIds);
    let activeHighlightMessageSet = new Set(activeHighlightMessageIds);
    function clearEphemeralHighlights() {
      activeHighlightMessageIds = [];
    }
    function clearUnreadMarkersForChat() {
      if (!store_get($$store_subs ??= {}, "$currentUser", currentUser) || !partnerNpub) {
        return;
      }
      clearChatUnread(store_get($$store_subs ??= {}, "$currentUser", currentUser).npub, partnerNpub);
      unreadSnapshotMessageIds = [];
      clearEphemeralHighlights();
    }
    let showVoiceSheet = false;
    let contextMenu = { isOpen: false };
    const isAndroidShell = isAndroidCapacitorShell();
    const useFullWidthBubbles = isAndroidShell || isMobileWeb();
    const useSmallAvatars = useFullWidthBubbles;
    function getLastSentIndex(list) {
      let index = -1;
      for (let i = 0; i < list.length; i++) {
        if (list[i].direction === "sent") {
          index = i;
        }
      }
      return index;
    }
    function scrollToBottom() {
    }
    function getRelativeTime(timestamp) {
      const now = currentTime;
      const diff = now - timestamp;
      const seconds = Math.floor(diff / 1e3);
      const minutes = Math.floor(seconds / 60);
      const hours = Math.floor(minutes / 60);
      const days = Math.floor(hours / 24);
      const weeks = Math.floor(days / 7);
      const months = Math.floor(days / 30);
      const years = Math.floor(days / 365);
      const translate2 = get($format);
      if (seconds < 60) return translate2("chat.relative.justNow");
      if (minutes < 60) {
        const key2 = minutes === 1 ? "chat.relative.minutes" : "chat.relative.minutesPlural";
        return translate2(key2, { values: { count: minutes } });
      }
      if (hours < 24) {
        const key2 = hours === 1 ? "chat.relative.hours" : "chat.relative.hoursPlural";
        return translate2(key2, { values: { count: hours } });
      }
      if (days < 7) {
        const key2 = days === 1 ? "chat.relative.days" : "chat.relative.daysPlural";
        return translate2(key2, { values: { count: days } });
      }
      if (weeks < 4) {
        const key2 = weeks === 1 ? "chat.relative.weeks" : "chat.relative.weeksPlural";
        return translate2(key2, { values: { count: weeks } });
      }
      if (months < 12) {
        const key2 = months === 1 ? "chat.relative.months" : "chat.relative.monthsPlural";
        return translate2(key2, { values: { count: months } });
      }
      const key = years === 1 ? "chat.relative.years" : "chat.relative.yearsPlural";
      return translate2(key, { values: { count: years } });
    }
    function isSameDay(a, b) {
      const da = new Date(a);
      const db2 = new Date(b);
      return da.getFullYear() === db2.getFullYear() && da.getMonth() === db2.getMonth() && da.getDate() === db2.getDate();
    }
    function formatDateLabel(timestamp) {
      const now = new Date(currentTime);
      const target = new Date(timestamp);
      const startOfToday = new Date(now.getFullYear(), now.getMonth(), now.getDate()).getTime();
      const startOfTarget = new Date(target.getFullYear(), target.getMonth(), target.getDate()).getTime();
      const diffDays = Math.round((startOfToday - startOfTarget) / (1e3 * 60 * 60 * 24));
      const translateDate = get($format);
      if (diffDays === 0) {
        return translateDate("chat.dateLabel.today");
      }
      if (diffDays === 1) {
        return translateDate("chat.dateLabel.yesterday");
      }
      return target.toLocaleDateString(void 0, { year: "numeric", month: "short", day: "numeric" });
    }
    const translate = (key, vars) => get($format)(key, vars);
    function removeOptimisticMessage(eventId) {
      const target = optimisticMessages.find((m) => m.eventId === eventId);
      if (target) {
        revokeOptimisticResources(target);
      }
      optimisticMessages = optimisticMessages.filter((m) => m.eventId !== eventId);
    }
    const showVoiceButton = typeof window !== "undefined" && inputText.trim().length === 0 && isVoiceRecordingSupported();
    async function sendVoiceMessage(file) {
      if (!partnerNpub) {
        return;
      }
      try {
        await ensureDefaultBlossomServersForCurrentUser();
      } catch {
      }
      const createdAtSeconds = Math.floor(Date.now() / 1e3);
      const sentAtMs = createdAtSeconds * 1e3;
      const optimisticEventId = makeOptimisticEventId();
      const optimisticUrl = URL.createObjectURL(file);
      const optimistic = {
        recipientNpub: partnerNpub,
        message: "",
        sentAt: sentAtMs,
        eventId: optimisticEventId,
        direction: "sent",
        createdAt: Date.now(),
        rumorKind: 15,
        fileUrl: optimisticUrl,
        fileType: file.type || "audio/webm"
      };
      optimisticMessages = [...optimisticMessages, optimistic];
      void (async () => {
        try {
          await messagingService.sendFileMessage(partnerNpub, file, "audio", createdAtSeconds);
          if (isDestroyed) return;
          setTimeout(
            () => {
              if (!isDestroyed) {
                removeOptimisticMessage(optimisticEventId);
              }
            },
            0
          );
          clearUnreadMarkersForChat();
          scrollToBottom();
          hapticLightImpact();
        } catch (e) {
          if (isDestroyed) return;
          console.error("Failed to send voice message:", e);
          clearRelayStatus();
          removeOptimisticMessage(optimisticEventId);
          await nativeDialogService.alert({
            title: translate("chat.sendFailedTitle"),
            message: translate("chat.sendFailedMessagePrefix") + e.message
          });
        }
      })();
    }
    function openVoiceSheet() {
      if (!showVoiceButton) {
        return;
      }
      showVoiceSheet = true;
    }
    function closeVoiceSheet() {
      showVoiceSheet = false;
    }
    let $$settled = true;
    let $$inner_renderer;
    function $$render_inner($$renderer3) {
      head("may7r9", $$renderer3, ($$renderer4) => {
        if (partnerNpub) {
          $$renderer4.push("<!--[-->");
          $$renderer4.title(($$renderer5) => {
            $$renderer5.push(`<title>nospeak: chat with ${escape_html(partnerNpub.slice(0, 10) + "...")}</title>`);
          });
        } else {
          $$renderer4.push("<!--[!-->");
        }
        $$renderer4.push(`<!--]-->`);
      });
      $$renderer3.push(`<div class="relative flex flex-col h-full overflow-hidden bg-white/30 dark:bg-slate-900/30 backdrop-blur-sm">`);
      {
        $$renderer3.push("<!--[!-->");
      }
      $$renderer3.push(`<!--]--> `);
      {
        $$renderer3.push("<!--[!-->");
      }
      $$renderer3.push(`<!--]--> `);
      if (showVoiceSheet) {
        $$renderer3.push("<!--[-->");
        VoiceMessageSheet($$renderer3, {
          isOpen: showVoiceSheet,
          onCancel: closeVoiceSheet,
          onClose: closeVoiceSheet,
          onSend: (file) => void sendVoiceMessage(file)
        });
      } else {
        $$renderer3.push("<!--[!-->");
      }
      $$renderer3.push(`<!--]--> `);
      if (partnerNpub) {
        $$renderer3.push("<!--[-->");
        $$renderer3.push(`<div class="absolute top-0 left-0 right-0 z-20 p-2 pt-safe min-h-16 border-b border-gray-200/50 dark:border-slate-700/70 flex justify-between items-center bg-white/80 dark:bg-slate-900/80 backdrop-blur-xl shadow-sm transition-all duration-150 ease-out"><div class="flex items-center gap-3"><button class="md:hidden text-gray-500 hover:text-gray-700 dark:text-gray-400 dark:hover:text-gray-200 transition-colors duration-150 ease-out" aria-label="Back to contacts"><svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 19l-7-7 7-7"></path></svg></button> `);
        if (partnerNpub) {
          $$renderer3.push("<!--[-->");
          $$renderer3.push(`<button class="hover:opacity-80 transition-opacity duration-150 ease-out cursor-pointer" aria-label="Open contact profile">`);
          Avatar($$renderer3, {
            npub: partnerNpub,
            src: partnerPicture,
            size: "sm",
            class: "!w-8 !h-8 md:!w-9 md:!h-9 transition-all duration-150 ease-out"
          });
          $$renderer3.push(`<!----></button>`);
        } else {
          $$renderer3.push("<!--[!-->");
        }
        $$renderer3.push(`<!--]--> <button class="font-bold hover:underline dark:text-white text-left">${escape_html(partnerNpub.slice(0, 10) + "...")}</button></div> `);
        if (partnerNpub !== "ALL") {
          $$renderer3.push("<!--[-->");
          $$renderer3.push(`<div${attr_class(`md:hidden absolute bottom-2 h-11 left-24 right-16 z-30 transition-[opacity,transform] duration-200 ease-out ${isSearchOpen ? "opacity-100 translate-x-0 pointer-events-auto" : "opacity-0 translate-x-2 pointer-events-none"}`)}><input${attr("value", searchQuery)}${attr("placeholder", store_get($$store_subs ??= {}, "$t", $format)("chat.searchPlaceholder"))} class="w-full h-full px-4 border border-gray-200 dark:border-slate-700 rounded-full bg-white dark:bg-slate-800 text-gray-900 dark:text-white focus:outline-none focus:ring-2 focus:ring-blue-500/30 transition-all placeholder:text-gray-400 dark:placeholder:text-slate-500"${attr("aria-label", store_get($$store_subs ??= {}, "$t", $format)("chat.searchAriaLabel"))}/></div> <div class="flex items-center gap-2"><div${attr_class(`hidden md:block transition-[max-width,opacity] duration-200 ease-out ${isSearchOpen ? "max-w-56 opacity-100 overflow-visible" : "max-w-0 opacity-0 overflow-hidden pointer-events-none"}`)}><input${attr("value", searchQuery)}${attr("placeholder", store_get($$store_subs ??= {}, "$t", $format)("chat.searchPlaceholder"))} class="w-full px-4 h-11 border border-gray-200 dark:border-slate-700 rounded-full bg-white/90 dark:bg-slate-800/90 text-gray-900 dark:text-white focus:outline-none focus:ring-2 focus:ring-blue-500/30 transition-all placeholder:text-gray-400 dark:placeholder:text-slate-500"${attr("aria-label", store_get($$store_subs ??= {}, "$t", $format)("chat.searchAriaLabel"))}/></div> `);
          Button($$renderer3, {
            size: "icon",
            onclick: toggleSearch,
            class: "h-11 w-11 relative z-40",
            "aria-label": "Search chat",
            children: ($$renderer4) => {
              $$renderer4.push(`<svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 21l-4.35-4.35m1.85-5.15a7 7 0 11-14 0 7 7 0 0114 0z"></path></svg>`);
            },
            $$slots: { default: true }
          });
          $$renderer3.push(`<!----></div>`);
        } else {
          $$renderer3.push("<!--[!-->");
        }
        $$renderer3.push(`<!--]--></div>`);
      } else {
        $$renderer3.push("<!--[!-->");
      }
      $$renderer3.push(`<!--]--> <div tabindex="-1" class="flex-1 overflow-x-hidden overflow-y-auto px-4 pb-safe-offset-28 pt-[calc(5rem+env(safe-area-inset-top))] space-y-4 custom-scrollbar native-scroll focus:outline-none focus:ring-0">`);
      if (isSearchActive) {
        $$renderer3.push("<!--[-->");
        if (isSearchingHistory) {
          $$renderer3.push("<!--[-->");
          $$renderer3.push(`<div class="flex justify-center p-2"><div class="animate-spin rounded-full h-6 w-6 border-b-2 border-blue-500"></div></div>`);
        } else {
          $$renderer3.push("<!--[!-->");
          if (displayMessages.length === 0) {
            $$renderer3.push("<!--[-->");
            $$renderer3.push(`<div class="flex justify-center mt-10"><div class="px-4 py-2 rounded-2xl bg-white/80 dark:bg-slate-900/80 border border-gray-200/70 dark:border-slate-700/70 shadow-md backdrop-blur-xl typ-body text-gray-600 dark:text-slate-200">${escape_html(store_get($$store_subs ??= {}, "$t", $format)("chat.searchNoMatches"))}</div></div>`);
          } else {
            $$renderer3.push("<!--[!-->");
          }
          $$renderer3.push(`<!--]-->`);
        }
        $$renderer3.push(`<!--]-->`);
      } else {
        $$renderer3.push("<!--[!-->");
        if (canRequestNetworkHistory && displayMessages.length > 0) {
          $$renderer3.push("<!--[-->");
          $$renderer3.push(`<div class="flex flex-col items-center p-2 gap-2"><button class="typ-meta px-4 py-1.5 rounded-full bg-white/70 dark:bg-slate-800/80 backdrop-blur-sm border border-gray-200/60 dark:border-slate-700/60 text-gray-700 dark:text-slate-200 hover:bg-white/90 dark:hover:bg-slate-700/90 transition-all shadow-sm" type="button">${escape_html(store_get($$store_subs ??= {}, "$t", $format)("chat.history.fetchOlder"))}</button> `);
          if (networkHistorySummary) {
            $$renderer3.push("<!--[-->");
            $$renderer3.push(`<div class="px-3 py-1 rounded-full typ-meta bg-white/60 dark:bg-slate-800/60 border border-gray-200/60 dark:border-slate-700/60 text-gray-600 dark:text-slate-200 shadow-sm backdrop-blur-sm">${escape_html(get($format)("chat.history.summary", {
              values: {
                events: networkHistorySummary.eventsFetched,
                saved: networkHistorySummary.messagesSaved,
                chat: networkHistorySummary.messagesForChat
              }
            }))}</div>`);
          } else {
            $$renderer3.push("<!--[!-->");
          }
          $$renderer3.push(`<!--]--></div>`);
        } else {
          $$renderer3.push("<!--[!-->");
          if (networkHistoryStatus === "no-more" && displayMessages.length > 0) {
            $$renderer3.push("<!--[-->");
            $$renderer3.push(`<div class="flex justify-center p-2"><div class="px-3 py-1 rounded-full typ-meta bg-white/70 dark:bg-slate-800/80 border border-gray-200/70 dark:border-slate-700/70 text-gray-500 dark:text-slate-300 shadow-sm backdrop-blur-sm">${escape_html(store_get($$store_subs ??= {}, "$t", $format)("chat.history.none"))}</div></div>`);
          } else {
            $$renderer3.push("<!--[!-->");
            if (networkHistoryStatus === "error" && displayMessages.length > 0) {
              $$renderer3.push("<!--[-->");
              $$renderer3.push(`<div class="flex justify-center p-2"><div class="px-3 py-1 rounded-full typ-meta bg-red-50/80 dark:bg-red-900/40 border border-red-200/80 dark:border-red-500/70 text-red-600 dark:text-red-200 shadow-sm backdrop-blur-sm">${escape_html(store_get($$store_subs ??= {}, "$t", $format)("chat.history.error"))}</div></div>`);
            } else {
              $$renderer3.push("<!--[!-->");
            }
            $$renderer3.push(`<!--]-->`);
          }
          $$renderer3.push(`<!--]-->`);
        }
        $$renderer3.push(`<!--]--> `);
        if (isFetchingHistory) {
          $$renderer3.push("<!--[-->");
          $$renderer3.push(`<div class="flex justify-center p-2">`);
          CircularProgress($$renderer3, { size: 24, strokeWidth: 3 });
          $$renderer3.push(`<!----></div>`);
        } else {
          $$renderer3.push("<!--[!-->");
        }
        $$renderer3.push(`<!--]--> `);
        if (displayMessages.length === 0 && !isFetchingHistory) {
          $$renderer3.push("<!--[-->");
          $$renderer3.push(`<div class="flex justify-center mt-10"><div class="max-w-sm px-4 py-3 rounded-2xl bg-white/80 dark:bg-slate-900/80 border border-gray-200/70 dark:border-slate-700/70 shadow-md backdrop-blur-xl text-center space-y-1"><div class="typ-meta font-semibold uppercase text-gray-500 dark:text-slate-400">${escape_html(store_get($$store_subs ??= {}, "$t", $format)("chat.empty.noMessagesTitle"))}</div> <div class="typ-body text-gray-600 dark:text-slate-200">`);
          if (partnerNpub) {
            $$renderer3.push("<!--[-->");
            $$renderer3.push(`${escape_html(get($format)("chat.empty.forContact", {
              values: { name: partnerNpub.slice(0, 10) + "..." }
            }))}`);
          } else {
            $$renderer3.push("<!--[!-->");
            $$renderer3.push(`${escape_html(store_get($$store_subs ??= {}, "$t", $format)("chat.empty.generic"))}`);
          }
          $$renderer3.push(`<!--]--></div></div></div>`);
        } else {
          $$renderer3.push("<!--[!-->");
        }
        $$renderer3.push(`<!--]-->`);
      }
      $$renderer3.push(`<!--]--> <!--[-->`);
      const each_array = ensure_array_like(displayMessages);
      for (let i = 0, $$length = each_array.length; i < $$length; i++) {
        let msg = each_array[i];
        const caption = isCaptionMessage(displayMessages, i);
        const captionForThis = getCaptionForParent(displayMessages, i);
        const hasUnreadMarker = msg.direction === "received" && (unreadSnapshotMessageSet.has(msg.eventId) || captionForThis && unreadSnapshotMessageSet.has(captionForThis.eventId) || activeHighlightMessageSet.has(msg.eventId));
        const hasYouTubeLink = /https?:\/\/(www\.)?(youtube\.com|youtu\.be)\//.test(msg.message);
        const bubbleWidthClass = hasYouTubeLink ? "w-full max-w-full md:w-[560px] md:max-w-full" : useFullWidthBubbles ? "max-w-full" : "max-w-[70%]";
        if (i === 0 || !isSameDay(msg.sentAt, displayMessages[i - 1].sentAt)) {
          $$renderer3.push("<!--[-->");
          $$renderer3.push(`<div class="flex justify-center my-2"><div class="px-3 py-1 rounded-full typ-meta bg-white/70 dark:bg-slate-800/80 border border-gray-200/70 dark:border-slate-700/70 text-gray-600 dark:text-slate-200 shadow-sm backdrop-blur-sm">${escape_html(formatDateLabel(msg.sentAt))}</div></div>`);
        } else {
          $$renderer3.push("<!--[!-->");
        }
        $$renderer3.push(`<!--]-->   `);
        if (!caption) {
          $$renderer3.push("<!--[-->");
          $$renderer3.push(`<div${attr_class(`flex ${msg.direction === "sent" ? "justify-end" : "justify-start"} items-end gap-2`)}>`);
          if (msg.direction === "received" && partnerNpub && !caption) {
            $$renderer3.push("<!--[-->");
            $$renderer3.push(`<button class="mb-1 hover:opacity-80 transition-opacity duration-150 ease-out cursor-pointer">`);
            Avatar($$renderer3, {
              npub: partnerNpub,
              src: partnerPicture,
              size: "md",
              class: `${useSmallAvatars ? "!w-10 !h-10" : "!w-14 !h-14"} md:!w-10 md:!h-10 transition-all duration-150 ease-out`
            });
            $$renderer3.push(`<!----></button>`);
          } else {
            $$renderer3.push("<!--[!-->");
          }
          $$renderer3.push(`<!--]--> <div role="button" tabindex="0"${attr_class(`${bubbleWidthClass} p-3 shadow-sm cursor-pointer transition-all duration-150 ease-out relative ${isAndroidShell ? "select-none" : ""}
                          ${msg.direction === "sent" ? "bg-blue-50/10 dark:bg-blue-900/40 text-gray-900 dark:text-slate-100 border border-blue-500/10 dark:border-blue-400/10 rounded-2xl rounded-br-none hover:shadow-md" : "bg-white/95 dark:bg-slate-800/95 md:bg-white/80 md:dark:bg-slate-800/80 md:backdrop-blur-sm dark:text-white border border-gray-100 dark:border-slate-700/50 rounded-2xl rounded-bl-none hover:bg-white dark:hover:bg-slate-800"}`)}>`);
          if (hasUnreadMarker) {
            $$renderer3.push("<!--[-->");
            $$renderer3.push(`<div class="absolute left-0 top-2 bottom-2 w-1 rounded-r bg-emerald-400/70"></div>`);
          } else {
            $$renderer3.push("<!--[!-->");
          }
          $$renderer3.push(`<!--]--> `);
          MessageContent($$renderer3, {
            content: msg.message,
            highlight: isSearchActive ? searchQuery : void 0,
            isOwn: msg.direction === "sent",
            onImageClick: openImageViewer,
            fileUrl: msg.fileUrl,
            fileType: msg.fileType,
            fileEncryptionAlgorithm: msg.fileEncryptionAlgorithm,
            fileKey: msg.fileKey,
            fileNonce: msg.fileNonce,
            authorNpub: msg.direction === "sent" ? store_get($$store_subs ??= {}, "$currentUser", currentUser)?.npub : partnerNpub,
            location: msg.location
          });
          $$renderer3.push(`<!----> `);
          if (captionForThis) {
            $$renderer3.push("<!--[-->");
            $$renderer3.push(`<div class="mt-2 text-sm text-gray-900 dark:text-slate-100">${html(highlightPlainTextToHtml(captionForThis.message, isSearchActive ? searchQuery : ""))}</div>`);
          } else {
            $$renderer3.push("<!--[!-->");
          }
          $$renderer3.push(`<!--]--> `);
          MessageReactions($$renderer3, {
            targetEventId: msg.rumorId || "",
            isOwn: msg.direction === "sent"
          });
          $$renderer3.push(`<!----> <div${attr_class(`typ-meta mt-1 flex items-center justify-end gap-2 ${msg.direction === "sent" ? "text-blue-100" : "text-gray-400"}`)}><span class="cursor-help"${attr("title", new Date(msg.sentAt).toLocaleString())}>${escape_html(getRelativeTime(msg.sentAt))}</span> <button type="button" class="hidden md:inline-flex py-1 pr-0 pl-px rounded-l hover:bg-gray-100/50 dark:hover:bg-slate-700/50 transition-colors" aria-label="Message options"><svg width="14" height="14" viewBox="0 0 24 24" fill="currentColor"><circle cx="12" cy="5" r="2"></circle><circle cx="12" cy="12" r="2"></circle><circle cx="12" cy="19" r="2"></circle></svg></button></div> `);
          if (msg.direction === "sent" && i === getLastSentIndex(displayMessages) && partnerNpub) {
            $$renderer3.push("<!--[-->");
            if (store_get($$store_subs ??= {}, "$lastRelaySendStatus", lastRelaySendStatus) && store_get($$store_subs ??= {}, "$lastRelaySendStatus", lastRelaySendStatus).recipientNpub === partnerNpub) {
              $$renderer3.push("<!--[-->");
              $$renderer3.push(`<div class="typ-meta mt-0.5 text-right text-blue-100">`);
              if (store_get($$store_subs ??= {}, "$lastRelaySendStatus", lastRelaySendStatus).successfulRelays === 0) {
                $$renderer3.push("<!--[-->");
                $$renderer3.push(`${escape_html(store_get($$store_subs ??= {}, "$t", $format)("chat.relayStatus.sending"))}`);
              } else {
                $$renderer3.push("<!--[!-->");
                $$renderer3.push(`${escape_html(get($format)("chat.relayStatus.sentToRelays", {
                  values: {
                    successful: store_get($$store_subs ??= {}, "$lastRelaySendStatus", lastRelaySendStatus).successfulRelays,
                    desired: store_get($$store_subs ??= {}, "$lastRelaySendStatus", lastRelaySendStatus).desiredRelays
                  }
                }))}`);
              }
              $$renderer3.push(`<!--]--></div>`);
            } else {
              $$renderer3.push("<!--[!-->");
              if (msg.eventId && msg.eventId.startsWith("optimistic:")) {
                $$renderer3.push("<!--[-->");
                $$renderer3.push(`<div class="typ-meta mt-0.5 text-right text-blue-100">${escape_html(store_get($$store_subs ??= {}, "$t", $format)("chat.relayStatus.sending"))}</div>`);
              } else {
                $$renderer3.push("<!--[!-->");
              }
              $$renderer3.push(`<!--]-->`);
            }
            $$renderer3.push(`<!--]-->`);
          } else {
            $$renderer3.push("<!--[!-->");
          }
          $$renderer3.push(`<!--]--></div> `);
          if (msg.direction === "sent" && store_get($$store_subs ??= {}, "$currentUser", currentUser)) {
            $$renderer3.push("<!--[-->");
            $$renderer3.push(`<button class="mb-1 hover:opacity-80 transition-opacity duration-150 ease-out cursor-pointer">`);
            Avatar($$renderer3, {
              npub: store_get($$store_subs ??= {}, "$currentUser", currentUser).npub,
              src: myPicture,
              size: "md",
              class: `${useSmallAvatars ? "!w-10 !h-10" : "!w-14 !h-14"} md:!w-10 md:!h-10 transition-all duration-150 ease-out`
            });
            $$renderer3.push(`<!----></button>`);
          } else {
            $$renderer3.push("<!--[!-->");
          }
          $$renderer3.push(`<!--]--></div>`);
        } else {
          $$renderer3.push("<!--[!-->");
        }
        $$renderer3.push(`<!--]-->`);
      }
      $$renderer3.push(`<!--]--></div> <div class="absolute bottom-0 left-0 right-0 z-20 p-4 p-4-safe-bottom border-t border-gray-200/50 dark:border-slate-700/70 bg-white/80 dark:bg-slate-900/80 backdrop-blur-xl shadow-lg transition-all duration-150 ease-out"><form class="flex gap-3 items-end relative">`);
      {
        $$renderer3.push("<!--[!-->");
      }
      $$renderer3.push(`<!--]--> <div class="flex-1 flex items-center bg-white/90 dark:bg-slate-800/90 border border-gray-200 dark:border-slate-700 rounded-3xl px-4 py-1.5 gap-2 shadow-inner focus-within:ring-2 focus-within:ring-blue-500/50 transition-all">`);
      MediaUploadButton($$renderer3, {
        variant: "chat"
      });
      $$renderer3.push(`<!----> <textarea${attr("disabled", isSending, true)} rows="1" class="flex-1 bg-transparent border-0 focus:outline-none focus:ring-0 text-sm md:text-base dark:text-white disabled:opacity-50 resize-none overflow-hidden placeholder:text-gray-400 dark:placeholder:text-slate-500 py-1"${attr("placeholder", store_get($$store_subs ??= {}, "$t", $format)("chat.inputPlaceholder"))}>`);
      const $$body = escape_html(inputText);
      if ($$body) {
        $$renderer3.push(`${$$body}`);
      }
      $$renderer3.push(`</textarea></div> `);
      if (showVoiceButton) {
        $$renderer3.push("<!--[-->");
        Button($$renderer3, {
          type: "button",
          variant: "primary",
          size: "icon",
          class: "flex-shrink-0",
          onclick: openVoiceSheet,
          "aria-label": store_get($$store_subs ??= {}, "$t", $format)("chat.voiceMessage.recordAria"),
          children: ($$renderer4) => {
            $$renderer4.push(`<svg viewBox="0 0 24 24" class="w-5 h-5" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M12 1a3 3 0 0 0-3 3v8a3 3 0 0 0 6 0V4a3 3 0 0 0-3-3z"></path><path d="M19 10v2a7 7 0 0 1-14 0v-2"></path><line x1="12" y1="19" x2="12" y2="23"></line><line x1="8" y1="23" x2="16" y2="23"></line></svg>`);
          },
          $$slots: { default: true }
        });
      } else {
        $$renderer3.push("<!--[!-->");
      }
      $$renderer3.push(`<!--]--> `);
      if (inputText.trim().length > 0) {
        $$renderer3.push("<!--[-->");
        Button($$renderer3, {
          type: "submit",
          variant: "primary",
          size: "icon",
          class: "flex-shrink-0",
          disabled: isSending,
          "aria-label": "Send message",
          children: ($$renderer4) => {
            $$renderer4.push(`<svg viewBox="0 0 24 24" class="w-5 h-5" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><polyline points="9 6 15 12 9 18"></polyline></svg>`);
          },
          $$slots: { default: true }
        });
      } else {
        $$renderer3.push("<!--[!-->");
      }
      $$renderer3.push(`<!--]--></form></div></div> `);
      ContextMenu($$renderer3, {
        isOpen: contextMenu.isOpen
      });
      $$renderer3.push(`<!---->`);
    }
    do {
      $$settled = true;
      $$inner_renderer = $$renderer2.copy();
      $$render_inner($$inner_renderer);
    } while (!$$settled);
    $$renderer2.subsume($$inner_renderer);
    if ($$store_subs) unsubscribe_stores($$store_subs);
  });
}
function _page($$renderer, $$props) {
  $$renderer.component(($$renderer2) => {
    let messages = [];
    let currentPartner = page.params.npub;
    let isFetchingHistory = false;
    let cacheExhausted = false;
    let networkHistoryStatus = "idle";
    let networkHistorySummary = null;
    const canRequestNetworkHistory = cacheExhausted;
    $$renderer2.push(`<!---->`);
    {
      ChatView($$renderer2, {
        messages,
        partnerNpub: currentPartner,
        isFetchingHistory,
        canRequestNetworkHistory,
        networkHistoryStatus,
        networkHistorySummary
      });
    }
    $$renderer2.push(`<!---->`);
  });
}
export {
  _page as default
};
