import { error, json } from "@sveltejs/kit";
const MAX_CONTENT_LENGTH = 64 * 1024;
const REQUEST_TIMEOUT_MS = 5e3;
function extractBetween(html, regex) {
  const match = html.match(regex);
  if (!match || !match[1]) {
    return void 0;
  }
  const value = match[1].trim();
  return value.length > 0 ? value : void 0;
}
function getCharsetFromContentType(contentType) {
  if (!contentType) {
    return "utf-8";
  }
  const lower = contentType.toLowerCase();
  const match = lower.match(/charset\s*=\s*([^;]+)/);
  if (!match || !match[1]) {
    return "utf-8";
  }
  const charset = match[1].trim();
  if (!charset) {
    return "utf-8";
  }
  if (charset === "utf8") {
    return "utf-8";
  }
  return charset;
}
function decodeHtmlEntities(input) {
  if (!input) {
    return input;
  }
  const named = {
    amp: "&",
    lt: "<",
    gt: ">",
    quot: '"',
    apos: "'",
    nbsp: " ",
    auml: "ä",
    ouml: "ö",
    uuml: "ü",
    Auml: "Ä",
    Ouml: "Ö",
    Uuml: "Ü",
    szlig: "ß"
  };
  let result = input;
  result = result.replace(/&#(\d+);/g, (match, num) => {
    const code = Number(num);
    if (!Number.isFinite(code)) {
      return match;
    }
    try {
      return String.fromCodePoint(code);
    } catch {
      return match;
    }
  });
  result = result.replace(/&#x([0-9a-f]+);/gi, (match, hex) => {
    const code = parseInt(hex, 16);
    if (!Number.isFinite(code)) {
      return match;
    }
    try {
      return String.fromCodePoint(code);
    } catch {
      return match;
    }
  });
  result = result.replace(/&([a-zA-Z]+);/g, (match, name) => {
    if (Object.prototype.hasOwnProperty.call(named, name)) {
      return named[name];
    }
    return match;
  });
  return result;
}
function absolutizeUrl(maybeRelative, baseUrl) {
  try {
    return new URL(maybeRelative, baseUrl).toString();
  } catch {
    return maybeRelative;
  }
}
async function fetchUrlPreviewMetadata(url, fetchImpl = fetch) {
  let parsed;
  try {
    parsed = new URL(url);
  } catch {
    return null;
  }
  if (parsed.protocol !== "http:" && parsed.protocol !== "https:") {
    return null;
  }
  const controller = new AbortController();
  const timeout = setTimeout(() => controller.abort(), REQUEST_TIMEOUT_MS);
  try {
    const response = await fetchImpl(url, {
      method: "GET",
      signal: controller.signal,
      headers: {
        Accept: "text/html,application/xhtml+xml"
      }
    });
    if (!response.ok) {
      return null;
    }
    const reader = response.body?.getReader();
    if (!reader) {
      return null;
    }
    const contentType = "headers" in response && response.headers ? response.headers.get("content-type") : null;
    const charset = getCharsetFromContentType(contentType);
    const chunks = [];
    let totalLength = 0;
    while (totalLength < MAX_CONTENT_LENGTH) {
      const { done, value } = await reader.read();
      if (done || !value) {
        break;
      }
      let chunk = value;
      if (totalLength + chunk.length > MAX_CONTENT_LENGTH) {
        chunk = chunk.subarray(0, MAX_CONTENT_LENGTH - totalLength);
      }
      chunks.push(chunk);
      totalLength += chunk.length;
      if (totalLength >= MAX_CONTENT_LENGTH) {
        break;
      }
    }
    if (totalLength === 0) {
      return null;
    }
    const combined = new Uint8Array(totalLength);
    let offset = 0;
    for (const chunk of chunks) {
      combined.set(chunk, offset);
      offset += chunk.length;
    }
    let decoder;
    try {
      decoder = new TextDecoder(charset, { fatal: false });
    } catch {
      decoder = new TextDecoder("utf-8", { fatal: false });
    }
    const html = decoder.decode(combined);
    const rawTitle = extractBetween(html, /<meta[^>]+property=["']og:title["'][^>]*content=["']([^"']+)["'][^>]*>/i) || extractBetween(html, /<meta[^>]+name=["']twitter:title["'][^>]*content=["']([^"']+)["'][^>]*>/i) || extractBetween(html, /<meta[^>]+name=["']title["'][^>]*content=["']([^"']+)["'][^>]*>/i) || extractBetween(html, /<title[^>]*>([^<]{1,200})<\/title>/i);
    const rawDescription = extractBetween(html, /<meta[^>]+property=["']og:description["'][^>]*content=["']([^"']+)["'][^>]*>/i) || extractBetween(html, /<meta[^>]+name=["']twitter:description["'][^>]*content=["']([^"']+)["'][^>]*>/i) || extractBetween(html, /<meta[^>]+name=["']description["'][^>]*content=["']([^"']+)["'][^>]*>/i);
    const rawImage = extractBetween(html, /<meta[^>]+property=["']og:image:secure_url["'][^>]*content=["']([^"']+)["'][^>]*>/i) || extractBetween(html, /<meta[^>]+property=["']og:image["'][^>]*content=["']([^"']+)["'][^>]*>/i) || extractBetween(html, /<meta[^>]+name=["']twitter:image:src["'][^>]*content=["']([^"']+)["'][^>]*>/i) || extractBetween(html, /<meta[^>]+name=["']twitter:image["'][^>]*content=["']([^"']+)["'][^>]*>/i) || extractBetween(html, /<link[^>]+rel=["']image_src["'][^>]*href=["']([^"']+)["'][^>]*>/i);
    const title = rawTitle ? decodeHtmlEntities(rawTitle) : void 0;
    const description = rawDescription ? decodeHtmlEntities(rawDescription) : void 0;
    let image;
    if (rawImage) {
      const decodedImage = decodeHtmlEntities(rawImage);
      image = absolutizeUrl(decodedImage, url);
    }
    if (!title && !description) {
      return null;
    }
    return {
      url,
      title,
      description,
      image
    };
  } catch {
    return null;
  } finally {
    clearTimeout(timeout);
  }
}
const CORS_HEADERS = {
  "Access-Control-Allow-Origin": "*",
  "Access-Control-Allow-Methods": "GET,OPTIONS",
  "Access-Control-Allow-Headers": "Content-Type"
};
const GET = async ({ url }) => {
  const target = url.searchParams.get("url");
  if (!target) {
    throw error(400, "Missing url parameter");
  }
  const metadata = await fetchUrlPreviewMetadata(target);
  if (!metadata) {
    return new Response(null, { status: 204, headers: CORS_HEADERS });
  }
  return json(metadata, { headers: CORS_HEADERS });
};
const OPTIONS = async () => {
  return new Response(null, { status: 204, headers: CORS_HEADERS });
};
export {
  GET,
  OPTIONS
};
