/// <reference lib="webworker" />

import { cleanupOutdatedCaches, precacheAndRoute } from 'workbox-precaching';
import { clientsClaim } from 'workbox-core';

type PrecacheManifestEntry = {
    url: string;
    revision?: string;
};

declare let self: ServiceWorkerGlobalScope & {
    __WB_MANIFEST: Array<PrecacheManifestEntry | string> | undefined;
};

// Take over control immediately
self.skipWaiting();
clientsClaim();

// Clean up old caches
cleanupOutdatedCaches();

// Precache assets generated by Vite
// If the manifest injection fails for any reason, avoid crashing.
const wbManifest = self.__WB_MANIFEST;
precacheAndRoute(Array.isArray(wbManifest) ? wbManifest : []);

// Handle notification clicks
self.addEventListener('notificationclick', (event) => {
    event.notification.close();

    const urlToOpen = event.notification.data?.url;

    if (!urlToOpen) return;

    const promiseChain = self.clients.matchAll({
        type: 'window',
        includeUncontrolled: true
    }).then((windowClients) => {
        // Check if there is already a window/tab open with the target URL
        let matchingClient: WindowClient | null = null;

        for (const client of windowClients) {
            const clientUrl = new URL(client.url);
            const targetUrl = new URL(urlToOpen, self.location.origin);
            
            // Check if it's the same origin and path
            if (clientUrl.origin === targetUrl.origin) {
                matchingClient = client;
                // If it's the exact same page, we're done searching
                if (clientUrl.pathname === targetUrl.pathname) {
                    break;
                }
            }
        }

        if (matchingClient) {
            // Focus the existing window
            return matchingClient.focus().then(() => {
                // Navigate to the specific chat if needed
                if (matchingClient.url !== urlToOpen) {
                    return matchingClient.navigate(urlToOpen);
                }
            });
        } else {
            // If no window is open, open a new one
            return self.clients.openWindow(urlToOpen);
        }
    });

    event.waitUntil(promiseChain);
});
