import { json } from "@sveltejs/kit";
const DEFAULT_RUNTIME_CONFIG = {
  discoveryRelays: [
    "wss://nostr.data.haus",
    "wss://relay.damus.io",
    "wss://nos.lol",
    "wss://relay.primal.net",
    "wss://purplepag.es"
  ],
  defaultMessagingRelays: [
    "wss://nostr.data.haus",
    "wss://nos.lol",
    "wss://relay.damus.io"
  ],
  searchRelayUrl: "wss://relay.nostr.band",
  blasterRelayUrl: "wss://sendit.nosflare.com",
  defaultBlossomServers: [
    "https://blossom.data.haus",
    "https://blossom.primal.net"
  ],
  webAppBaseUrl: "https://nospeak.chat",
  robohashBaseUrl: "https://robohash.org/"
};
function parseCommaSeparated(value) {
  const trimmed = value ? value.trim() : "";
  if (!trimmed) {
    return [];
  }
  return trimmed.split(",").map((entry) => entry.trim()).filter((entry) => entry.length > 0);
}
function dedupePreserveOrder(values) {
  const seen = /* @__PURE__ */ new Set();
  const result = [];
  for (const value of values) {
    if (seen.has(value)) {
      continue;
    }
    seen.add(value);
    result.push(value);
  }
  return result;
}
function normalizeWssUrl(value) {
  try {
    const trimmed = value.trim();
    if (!trimmed) {
      return null;
    }
    const url = new URL(trimmed);
    if (url.protocol !== "wss:") {
      return null;
    }
    return url.href.replace(/\/$/, "");
  } catch {
    return null;
  }
}
function normalizeHttpsOrigin(value) {
  try {
    const trimmed = value.trim();
    if (!trimmed) {
      return null;
    }
    const url = new URL(trimmed);
    if (url.protocol !== "https:") {
      return null;
    }
    return url.origin;
  } catch {
    return null;
  }
}
function normalizeHttpsOriginWithTrailingSlash(value) {
  const origin = normalizeHttpsOrigin(value);
  if (!origin) {
    return null;
  }
  return `${origin}/`;
}
function parseWssList(value) {
  const entries = parseCommaSeparated(value);
  return dedupePreserveOrder(entries.map(normalizeWssUrl).filter((entry) => !!entry));
}
function parseHttpsOriginList(value) {
  const entries = parseCommaSeparated(value);
  return dedupePreserveOrder(entries.map(normalizeHttpsOrigin).filter((entry) => !!entry));
}
function resolveListOverride(envValue, parsed, fallback, name) {
  if (envValue && envValue.trim().length > 0 && parsed.length === 0) {
    console.warn(`Ignoring ${name} override; no valid entries`);
    return [...fallback];
  }
  return parsed.length > 0 ? parsed : [...fallback];
}
function resolveSingleOverride(envValue, parsed, fallback, name) {
  if (envValue && envValue.trim().length > 0 && !parsed) {
    console.warn(`Ignoring ${name} override; invalid value`);
    return fallback;
  }
  return parsed ?? fallback;
}
function getRuntimeConfigFromEnv(env) {
  const discoveryRelaysRaw = env.NOSPEAK_DISCOVERY_RELAYS;
  const defaultMessagingRelaysRaw = env.NOSPEAK_DEFAULT_MESSAGING_RELAYS;
  const defaultBlossomServersRaw = env.NOSPEAK_DEFAULT_BLOSSOM_SERVERS;
  const discoveryRelays = resolveListOverride(
    discoveryRelaysRaw,
    parseWssList(discoveryRelaysRaw),
    DEFAULT_RUNTIME_CONFIG.discoveryRelays,
    "NOSPEAK_DISCOVERY_RELAYS"
  );
  const defaultMessagingRelays = resolveListOverride(
    defaultMessagingRelaysRaw,
    parseWssList(defaultMessagingRelaysRaw),
    DEFAULT_RUNTIME_CONFIG.defaultMessagingRelays,
    "NOSPEAK_DEFAULT_MESSAGING_RELAYS"
  );
  const searchRelayUrl = resolveSingleOverride(
    env.NOSPEAK_SEARCH_RELAY,
    env.NOSPEAK_SEARCH_RELAY ? normalizeWssUrl(env.NOSPEAK_SEARCH_RELAY) : null,
    DEFAULT_RUNTIME_CONFIG.searchRelayUrl,
    "NOSPEAK_SEARCH_RELAY"
  );
  const blasterRelayUrl = resolveSingleOverride(
    env.NOSPEAK_BLASTER_RELAY,
    env.NOSPEAK_BLASTER_RELAY ? normalizeWssUrl(env.NOSPEAK_BLASTER_RELAY) : null,
    DEFAULT_RUNTIME_CONFIG.blasterRelayUrl,
    "NOSPEAK_BLASTER_RELAY"
  );
  const defaultBlossomServers = resolveListOverride(
    defaultBlossomServersRaw,
    parseHttpsOriginList(defaultBlossomServersRaw),
    DEFAULT_RUNTIME_CONFIG.defaultBlossomServers,
    "NOSPEAK_DEFAULT_BLOSSOM_SERVERS"
  );
  const webAppBaseUrl = resolveSingleOverride(
    env.NOSPEAK_WEB_APP_BASE_URL,
    env.NOSPEAK_WEB_APP_BASE_URL ? normalizeHttpsOrigin(env.NOSPEAK_WEB_APP_BASE_URL) : null,
    DEFAULT_RUNTIME_CONFIG.webAppBaseUrl,
    "NOSPEAK_WEB_APP_BASE_URL"
  );
  const robohashBaseUrl = resolveSingleOverride(
    env.NOSPEAK_ROBOHASH_BASE_URL,
    env.NOSPEAK_ROBOHASH_BASE_URL ? normalizeHttpsOriginWithTrailingSlash(env.NOSPEAK_ROBOHASH_BASE_URL) : null,
    DEFAULT_RUNTIME_CONFIG.robohashBaseUrl,
    "NOSPEAK_ROBOHASH_BASE_URL"
  );
  return {
    discoveryRelays,
    defaultMessagingRelays,
    searchRelayUrl,
    blasterRelayUrl,
    defaultBlossomServers,
    webAppBaseUrl,
    robohashBaseUrl
  };
}
function getRuntimeConfig() {
  return getRuntimeConfigFromEnv(process.env);
}
const NO_STORE_HEADERS = {
  "Cache-Control": "no-store"
};
const GET = async () => {
  const config = getRuntimeConfig();
  return json(config, { headers: NO_STORE_HEADERS });
};
export {
  GET
};
