import { Relay } from "nostr-tools";
import { w as writable, g as get } from "./index.js";
import { Capacitor, registerPlugin } from "@capacitor/core";
import "@capacitor/dialog";
import "@capacitor/share";
import Dexie from "dexie";
const relayHealths = writable([]);
const connectionStats = writable({
  connected: 0,
  total: 0,
  authRequiredConnected: 0,
  authAuthenticatedConnected: 0,
  authPendingConnected: 0,
  authFailedConnected: 0
});
const showRelayStatusModal = writable(false);
const signer = writable(null);
const currentUser = writable(null);
class NospeakDB extends Dexie {
  messages;
  profiles;
  // npub as primary key
  contacts;
  // npub as primary key
  retryQueue;
  reactions;
  constructor() {
    super("NospeakDB");
    this.version(1).stores({
      messages: "++id, [recipientNpub+sentAt], eventId, sentAt",
      profiles: "npub",
      retryQueue: "++id, nextAttempt"
    });
    this.version(2).stores({
      contacts: "npub"
    });
    this.version(3).stores({
      messages: "++id, [recipientNpub+sentAt], eventId, sentAt"
    }).upgrade(async (trans) => {
      const messages = await trans.table("messages").toArray();
      const seen = /* @__PURE__ */ new Set();
      const toDelete = [];
      for (const msg of messages) {
        if (seen.has(msg.eventId)) {
          toDelete.push(msg.id);
        } else {
          seen.add(msg.eventId);
        }
      }
      if (toDelete.length > 0) {
        await trans.table("messages").bulkDelete(toDelete);
      }
    });
    this.version(4).stores({
      messages: "++id, [recipientNpub+sentAt], &eventId, sentAt"
    });
    this.version(5).stores({
      contacts: "npub"
    });
    this.version(6).stores({
      reactions: "++id, targetEventId, reactionEventId, [targetEventId+authorNpub+emoji]"
    });
    this.version(7).stores({
      messages: "++id, [recipientNpub+sentAt], &eventId, sentAt, rumorId"
    });
    this.version(8).stores({
      contacts: "npub"
    });
    this.version(9).stores({
      profiles: "npub"
    }).upgrade(async (trans) => {
      const profiles = await trans.table("profiles").toArray();
      const updates = profiles.filter((p) => !Array.isArray(p.mediaServers)).map((p) => ({ ...p, mediaServers: [] }));
      if (updates.length > 0) {
        await trans.table("profiles").bulkPut(updates);
      }
    });
  }
  async clearAll() {
    await db.delete();
    await db.open();
    console.log("IndexedDB cleared");
  }
}
const db = new NospeakDB();
const DefaultRetryConfig = {
  maxRetries: 5,
  initialBackoff: 1e3,
  maxBackoff: 3e4,
  backoffMultiplier: 2,
  healthCheckInterval: 3e4,
  connectionTimeout: 3e3,
  pingInterval: 12e4,
  // 2 minutes (matches Android active profile)
  pingTimeout: 5e3
  // 5 seconds
};
class ConnectionManager {
  relays;
  config;
  defaultConfig;
  debug;
  isShutdown = false;
  backgroundModeEnabled = false;
  healthCheckTimer = null;
  uiUpdateTimer = null;
  pingTimer = null;
  pendingPings = /* @__PURE__ */ new Map();
  onRelayListUpdate = null;
  subscriptions = /* @__PURE__ */ new Set();
  authSigner = null;
  emitRelayUpdate() {
    if (this.onRelayListUpdate) {
      this.onRelayListUpdate(Array.from(this.relays.values()));
    }
  }
  constructor(config = DefaultRetryConfig, debug = false) {
    this.relays = /* @__PURE__ */ new Map();
    this.defaultConfig = { ...config };
    this.config = { ...config };
    this.debug = debug;
  }
  isRelayWebSocketOpen(relay) {
    const relayAny = relay;
    const ws = relayAny?.ws;
    if (!ws || typeof ws.readyState !== "number") {
      return true;
    }
    const openState = relayAny?._WebSocket?.OPEN ?? 1;
    return ws.readyState === openState;
  }
  preventSendsOnClosingWebSocket(relay) {
    if (this.isRelayWebSocketOpen(relay)) {
      return;
    }
    try {
      relay._connected = false;
    } catch {
    }
  }
  preventSendsOnIntentionalClose(relay) {
    try {
      relay._connected = false;
    } catch {
    }
    try {
      relay.connectionPromise = void 0;
    } catch {
    }
  }
  safeCloseRelay(relay) {
    this.preventSendsOnIntentionalClose(relay);
    try {
      relay.close();
    } catch (e) {
      if (this.debug) {
        console.warn(`Relay close failed for ${relay.url}`, e);
      }
    }
  }
  safeCloseSubscription(sub) {
    if (!sub || typeof sub.close !== "function") {
      return;
    }
    const relay = sub.relay;
    if (relay) {
      this.preventSendsOnClosingWebSocket(relay);
    }
    try {
      sub.close();
    } catch {
    }
  }
  setAuthSigner(authSigner) {
    this.authSigner = authSigner;
  }
  updateRelayAuthStatus(url, authStatus, lastAuthError = null) {
    const health = this.relays.get(url);
    if (!health) return;
    health.authStatus = authStatus;
    if (authStatus === "failed") {
      health.lastAuthAt = Date.now();
      health.lastAuthError = lastAuthError || "Unknown error";
    } else if (authStatus === "authenticated") {
      health.lastAuthAt = Date.now();
      health.lastAuthError = null;
    } else {
      health.lastAuthError = null;
    }
    this.emitRelayUpdate();
  }
  markRelayAuthRequired(url) {
    const health = this.relays.get(url);
    if (!health) return;
    if (health.authStatus === "not_required" || health.authStatus === "authenticated") {
      this.updateRelayAuthStatus(url, "required");
    }
  }
  async authenticateRelay(url) {
    const health = this.relays.get(url);
    if (!health?.relay || !health.isConnected) return false;
    const relayAny = health.relay;
    if (!relayAny.auth) return false;
    if (!relayAny.onauth) {
      this.updateRelayAuthStatus(url, "failed", "Missing signer");
      return false;
    }
    try {
      await relayAny.auth(relayAny.onauth);
      return true;
    } catch (e) {
      const message = e?.message || String(e);
      if (message.includes("Missing signer")) {
        this.updateRelayAuthStatus(url, "failed", "Missing signer");
      } else {
        this.updateRelayAuthStatus(url, "failed", message);
      }
      return false;
    }
  }
  attachAuthHandlers(url, relay) {
    const relayAny = relay;
    relayAny.onauth = async (eventTemplate) => {
      if (!this.authSigner) {
        this.updateRelayAuthStatus(url, "failed", "Missing signer");
        throw new Error("Missing signer");
      }
      try {
        return await this.authSigner(eventTemplate);
      } catch (e) {
        const message = e?.message || String(e);
        if (message.includes("Missing signer")) {
          this.updateRelayAuthStatus(url, "failed", "Missing signer");
        }
        throw e;
      }
    };
    if (typeof relayAny.auth === "function") {
      const originalAuth = relayAny.auth.bind(relayAny);
      relayAny.auth = async (signAuthEvent) => {
        this.markRelayAuthRequired(url);
        this.updateRelayAuthStatus(url, "authenticating");
        try {
          const result = await originalAuth(signAuthEvent);
          this.updateRelayAuthStatus(url, "authenticated");
          return result;
        } catch (e) {
          const message = e?.message || String(e);
          if (message.includes("Missing signer")) {
            this.updateRelayAuthStatus(url, "failed", "Missing signer");
          } else {
            this.updateRelayAuthStatus(url, "failed", message);
          }
          throw e;
        }
      };
    }
  }
  start() {
    if (this.healthCheckTimer) return;
    if (this.debug) console.log("Starting ConnectionManager loops");
    this.healthCheckTimer = setInterval(() => {
      this.checkAllRelayHealth();
    }, this.config.healthCheckInterval);
    this.uiUpdateTimer = setInterval(() => {
      this.emitRelayUpdate();
    }, 500);
    if (typeof document !== "undefined") {
      document.addEventListener("visibilitychange", this.handleVisibilityChange);
    }
    this.startPingLoop();
    if (typeof window !== "undefined") {
      window.addEventListener("online", this.handleOnline);
      window.addEventListener("offline", this.handleOffline);
    }
  }
  stop() {
    this.isShutdown = true;
    if (this.healthCheckTimer) clearInterval(this.healthCheckTimer);
    if (this.uiUpdateTimer) clearInterval(this.uiUpdateTimer);
    this.stopPingLoop();
    if (typeof document !== "undefined") {
      document.removeEventListener("visibilitychange", this.handleVisibilityChange);
    }
    if (typeof window !== "undefined") {
      window.removeEventListener("online", this.handleOnline);
      window.removeEventListener("offline", this.handleOffline);
    }
    for (const health of this.relays.values()) {
      if (health.relay) {
        this.safeCloseRelay(health.relay);
      }
    }
    if (this.debug) console.log("Connection manager stopped");
  }
  handleOnline = () => {
    if (this.debug) console.log("Network online, checking relays...");
    for (const [url, health] of this.relays.entries()) {
      if (!health.isConnected && health.type === 0) {
        this.handleReconnection(url);
      }
    }
  };
  handleOffline = () => {
    if (this.debug) console.log("Network offline");
  };
  handleVisibilityChange = () => {
    if (typeof document === "undefined") return;
    if (document.visibilityState === "visible") {
      if (this.debug) console.log("Tab became visible, verifying relay connections...");
      this.verifyAllConnections();
    }
  };
  verifyAllConnections() {
    for (const [url, health] of this.relays.entries()) {
      if (health.isConnected && health.relay) {
        if (!this.isRelayWebSocketOpen(health.relay)) {
          if (this.debug) console.log(`Relay ${url} socket dead on visibility check, triggering reconnect`);
          health.isConnected = false;
          health.relay = null;
          this.clearSubscriptionsForRelay(url);
          this.emitRelayUpdate();
          if (health.type === 0) {
            this.handleReconnection(url);
          }
        }
      }
    }
  }
  startPingLoop() {
    if (this.pingTimer) return;
    this.pingTimer = setInterval(() => {
      this.pingAllRelays();
    }, this.config.pingInterval);
  }
  stopPingLoop() {
    if (this.pingTimer) {
      clearInterval(this.pingTimer);
      this.pingTimer = null;
    }
    for (const [, pending] of this.pendingPings) {
      clearTimeout(pending.timeout);
      this.safeCloseSubscription(pending.sub);
    }
    this.pendingPings.clear();
  }
  pingAllRelays() {
    for (const [url, health] of this.relays.entries()) {
      if (health.isConnected && health.relay && health.type === 0) {
        this.pingRelay(url, health.relay);
      }
    }
  }
  pingRelay(url, relay) {
    if (this.pendingPings.has(url)) return;
    try {
      const sub = relay.subscribe(
        [{ kinds: [99999], limit: 1 }],
        {
          onevent: () => {
          },
          oneose: () => {
            this.clearPendingPing(url);
            if (this.debug) console.log(`Ping success for ${url}`);
          },
          onclose: () => {
            this.clearPendingPing(url);
          }
        }
      );
      const timeout = setTimeout(() => {
        if (this.debug) console.log(`Ping timeout for ${url}, marking as disconnected`);
        this.clearPendingPing(url);
        this.handlePingFailure(url);
      }, this.config.pingTimeout);
      this.pendingPings.set(url, { timeout, sub });
    } catch (e) {
      if (this.debug) console.log(`Ping failed for ${url}:`, e);
      this.handlePingFailure(url);
    }
  }
  clearPendingPing(url) {
    const pending = this.pendingPings.get(url);
    if (pending) {
      clearTimeout(pending.timeout);
      this.safeCloseSubscription(pending.sub);
      this.pendingPings.delete(url);
    }
  }
  handlePingFailure(url) {
    const health = this.relays.get(url);
    if (!health) return;
    health.isConnected = false;
    if (health.relay) {
      this.safeCloseRelay(health.relay);
      health.relay = null;
    }
    this.clearSubscriptionsForRelay(url);
    this.emitRelayUpdate();
    if (health.type === 0) {
      this.markRelayFailure(url);
    }
  }
  addPersistentRelay(url) {
    if (!this.relays.has(url)) {
      const health = {
        url,
        relay: null,
        isConnected: false,
        lastConnected: 0,
        lastAttempt: 0,
        successCount: 0,
        failureCount: 0,
        consecutiveFails: 0,
        type: 0,
        authStatus: "not_required",
        lastAuthAt: 0,
        lastAuthError: null
      };
      this.relays.set(url, health);
      if (this.debug) console.log(`Added persistent relay ${url}`);
      this.emitRelayUpdate();
      this.handleReconnection(url);
    } else {
      const health = this.relays.get(url);
      if (health.type === 1) {
        health.type = 0;
        if (this.debug) console.log(`Upgraded relay ${url} to persistent`);
        this.emitRelayUpdate();
      }
    }
  }
  addTemporaryRelay(url) {
    if (!this.relays.has(url)) {
      const health = {
        url,
        relay: null,
        isConnected: false,
        lastConnected: 0,
        lastAttempt: 0,
        successCount: 0,
        failureCount: 0,
        consecutiveFails: 0,
        type: 1,
        authStatus: "not_required",
        lastAuthAt: 0,
        lastAuthError: null
      };
      this.relays.set(url, health);
      if (this.debug) console.log(`Added temporary relay ${url}`);
      this.emitRelayUpdate();
      this.handleReconnection(url);
    }
  }
  removeRelay(url) {
    const health = this.relays.get(url);
    if (health) {
      if (health.relay) {
        this.safeCloseRelay(health.relay);
      }
      this.clearSubscriptionsForRelay(url);
      this.relays.delete(url);
      if (this.debug) console.log(`Removed relay ${url}`);
      this.emitRelayUpdate();
    }
  }
  clearSubscriptionsForRelay(url) {
    for (const sub of this.subscriptions) {
      if (sub.subMap.has(url)) {
        try {
          const s = sub.subMap.get(url);
          if (s && typeof s.close === "function") {
            this.safeCloseSubscription(s);
          }
        } catch (e) {
        }
        sub.subMap.delete(url);
      }
    }
  }
  cleanupTemporaryConnections() {
    const toRemove = [];
    for (const [url, health] of this.relays.entries()) {
      if (health.type === 1) {
        toRemove.push(url);
      }
    }
    for (const url of toRemove) {
      this.removeRelay(url);
    }
    if (this.debug && toRemove.length > 0) {
      console.log(`Cleaned up ${toRemove.length} temporary connections`);
    }
    this.emitRelayUpdate();
  }
  clearAllRelays() {
    const allUrls = Array.from(this.relays.keys());
    for (const url of allUrls) {
      this.removeRelay(url);
    }
    if (this.debug) {
      console.log(`Cleared all ${allUrls.length} relays`);
    }
    this.emitRelayUpdate();
  }
  getConnectedRelays() {
    const connected = [];
    for (const health of this.relays.values()) {
      if (health.isConnected && health.relay) {
        connected.push(health.relay);
      }
    }
    return connected;
  }
  getRelayHealth(url) {
    return this.relays.get(url);
  }
  getAllRelayHealth() {
    return Array.from(this.relays.values());
  }
  setUpdateCallback(callback) {
    this.onRelayListUpdate = callback;
    this.emitRelayUpdate();
  }
  setBackgroundModeEnabled(enabled) {
    this.backgroundModeEnabled = enabled;
    if (!enabled) {
      this.config = { ...this.defaultConfig };
      if (this.debug) {
        console.log("Background mode disabled, restored default retry config");
      }
      return;
    }
    this.config = {
      ...this.config,
      initialBackoff: Math.max(this.config.initialBackoff, 2e3),
      maxBackoff: Math.max(this.config.maxBackoff, 6e4)
    };
    if (this.debug) {
      console.log("Background mode enabled, applying conservative retry config", this.config);
    }
  }
  markRelayFailure(url) {
    const health = this.relays.get(url);
    if (!health) return;
    health.failureCount++;
    health.consecutiveFails++;
    health.lastAttempt = Date.now();
    if (health.type === 0) {
      this.handleReconnection(url);
    } else if (health.consecutiveFails >= 3) {
      this.handleReconnection(url);
    }
    if (this.debug) {
      console.log(`Marked failure for relay ${url} (consecutive: ${health.consecutiveFails})`);
    }
  }
  markRelaySuccess(url) {
    const health = this.relays.get(url);
    if (!health) return;
    health.successCount++;
    health.consecutiveFails = 0;
    health.lastConnected = Date.now();
    health.isConnected = true;
    if (this.debug) {
      console.log(`Marked success for relay ${url}`);
    }
    if (health.relay) {
      this.applySubscriptionsToRelay(health.relay);
    }
  }
  applySubscriptionsToRelay(relay) {
    for (const sub of this.subscriptions) {
      if (sub.subMap.has(relay.url)) continue;
      try {
        const s = relay.subscribe(sub.filters, {
          onevent: sub.onEvent,
          onclose: (reason) => {
            sub.subMap.delete(relay.url);
            if (typeof reason !== "string" || !reason.startsWith("auth-required")) {
              return;
            }
            this.markRelayAuthRequired(relay.url);
            if (sub.authRetryMap.get(relay.url)) {
              return;
            }
            sub.authRetryMap.set(relay.url, true);
            void (async () => {
              const authenticated = await this.authenticateRelay(relay.url);
              if (!authenticated) {
                return;
              }
              this.applySubscriptionsToRelay(relay);
            })();
          }
        });
        sub.subMap.set(relay.url, s);
        if (this.debug) console.log(`Applied subscription to ${relay.url}`);
      } catch (e) {
        console.error(`Failed to subscribe on ${relay.url}`, e);
      }
    }
  }
  // ... (rest of methods)
  calculateBackoff(consecutiveFails) {
    if (consecutiveFails <= 1) {
      return this.config.initialBackoff;
    }
    const exponential = 1 << consecutiveFails - 2;
    let delay = this.config.initialBackoff * exponential * this.config.backoffMultiplier;
    if (delay > this.config.maxBackoff) {
      delay = this.config.maxBackoff;
    }
    return delay;
  }
  async handleReconnection(url) {
    const health = this.relays.get(url);
    if (!health) return;
    if (health.type === 1) ;
    const backoffDelay = this.calculateBackoff(health.consecutiveFails);
    const timeSinceLastAttempt = Date.now() - health.lastAttempt;
    if (timeSinceLastAttempt < backoffDelay) {
      const waitTime = backoffDelay - timeSinceLastAttempt;
      if (this.debug) console.log(`Waiting ${waitTime}ms before reconnecting to ${url}`);
      await new Promise((resolve) => setTimeout(resolve, waitTime));
    }
    if (!this.relays.has(url)) return;
    try {
      await this.connectRelay(url);
    } catch (e) {
      if (this.debug) console.log(`Reconnection failed for ${url}: ${e}`);
      this.markRelayFailure(url);
    }
  }
  async connectRelay(url) {
    if (this.debug) console.log(`Attempting to connect to ${url}`);
    const health = this.relays.get(url);
    if (!health) throw new Error(`Relay ${url} not managed`);
    health.lastAttempt = Date.now();
    try {
      const relay = await Relay.connect(url);
      relay.onclose = () => {
        if (this.debug) console.log(`Relay ${url} disconnected`);
        health.isConnected = false;
        health.relay = null;
        this.clearSubscriptionsForRelay(url);
        if (health.authStatus === "authenticated" || health.authStatus === "authenticating") {
          health.authStatus = "required";
        }
        this.emitRelayUpdate();
        if (health.type === 0) {
          this.markRelayFailure(url);
        }
      };
      this.attachAuthHandlers(url, relay);
      health.relay = relay;
      health.isConnected = true;
      this.markRelaySuccess(url);
    } catch (e) {
      health.isConnected = false;
      if (health.relay) {
        this.safeCloseRelay(health.relay);
        health.relay = null;
      }
      throw e;
    }
  }
  checkAllRelayHealth() {
    for (const [url, health] of this.relays.entries()) {
      if (health.isConnected && health.relay) {
        if (!this.isRelayWebSocketOpen(health.relay)) {
          if (this.debug) console.log(`Health check: ${url} socket not open, clearing state`);
          health.isConnected = false;
          health.relay = null;
          this.clearSubscriptionsForRelay(url);
          this.emitRelayUpdate();
        }
      }
      if (!health.isConnected) {
        if (health.consecutiveFails < 5 || health.consecutiveFails % 5 === 0) {
          this.handleReconnection(url);
        }
      }
    }
  }
  async fetchEvents(filters, timeoutMs = 3e3) {
    const events = [];
    const connectedRelays = this.getConnectedRelays();
    if (connectedRelays.length === 0) return [];
    return new Promise((resolve) => {
      let completedRelays = 0;
      const subs = [];
      const ids = /* @__PURE__ */ new Set();
      const checkCompletion = () => {
        completedRelays++;
        if (completedRelays >= connectedRelays.length) {
          cleanup();
          resolve(events);
        }
      };
      const cleanup = () => {
        for (const { sub } of subs) {
          this.safeCloseSubscription(sub);
        }
        clearTimeout(timer);
      };
      const timer = setTimeout(() => {
        cleanup();
        resolve(events);
      }, timeoutMs);
      for (const relay of connectedRelays) {
        try {
          const sub = relay.subscribe(filters, {
            onevent(event) {
              if (!ids.has(event.id)) {
                ids.add(event.id);
                events.push(event);
              }
            },
            oneose() {
              checkCompletion();
            }
          });
          subs.push({ sub, relay: relay.url });
        } catch (e) {
          console.error(`Fetch failed on ${relay.url}`, e);
          checkCompletion();
        }
      }
    });
  }
  subscribe(filters, onEvent) {
    const subEntry = {
      filters,
      onEvent,
      subMap: /* @__PURE__ */ new Map(),
      authRetryMap: /* @__PURE__ */ new Map()
    };
    this.subscriptions.add(subEntry);
    for (const health of this.relays.values()) {
      if (health.relay && health.isConnected) {
        this.applySubscriptionsToRelay(health.relay);
      }
    }
    return () => {
      this.subscriptions.delete(subEntry);
      for (const s of subEntry.subMap.values()) {
        this.safeCloseSubscription(s);
      }
      subEntry.subMap.clear();
    };
  }
}
const connectionManager = new ConnectionManager(void 0, true);
connectionManager.setAuthSigner(async (eventTemplate) => {
  const currentSigner = get(signer);
  if (!currentSigner) {
    throw new Error("Missing signer");
  }
  return await currentSigner.signEvent(eventTemplate);
});
connectionManager.setUpdateCallback((relays) => {
  relayHealths.set(relays);
  const connectedRelays = relays.filter((relay) => relay.isConnected);
  const connected = connectedRelays.length;
  const authRequiredConnected = connectedRelays.filter((relay) => relay.authStatus !== "not_required").length;
  const authAuthenticatedConnected = connectedRelays.filter((relay) => relay.authStatus === "authenticated").length;
  const authPendingConnected = connectedRelays.filter(
    (relay) => relay.authStatus === "required" || relay.authStatus === "authenticating"
  ).length;
  const authFailedConnected = connectedRelays.filter((relay) => relay.authStatus === "failed").length;
  connectionStats.set({
    connected,
    total: relays.length,
    authRequiredConnected,
    authAuthenticatedConnected,
    authPendingConnected,
    authFailedConnected
  });
});
(() => {
  if (Capacitor.getPlatform() !== "android") {
    return null;
  }
  return registerPlugin("AndroidBackgroundMessaging");
})();
const { subscribe, update } = writable({});
Capacitor.getPlatform() === "android" ? registerPlugin("AndroidTapSound") : null;
export {
  currentUser as a,
  connectionStats as c,
  showRelayStatusModal as s
};
