/* Copyright (C) 2013 Laurent Destailleur <eldy@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 * or see http://www.gnu.org/
 */
package com.nltechno.dolidroidpro;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.Locale;
import java.util.Objects;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

// This are classes found with the useLibrary 'org.apache.http.legacy' in the build.gradle file.
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.DefaultHttpClient;

//import com.nltechno.utils.MySSLSocketFactory;
import com.nltechno.utils.Utils;

import android.Manifest;
import android.content.ActivityNotFoundException;
import android.content.ClipData;
import android.content.DialogInterface;
import android.content.pm.InstallSourceInfo;
import android.content.pm.PackageManager;
import android.media.MediaScannerConnection;
import android.net.Uri;
import android.net.http.SslError;
import android.os.AsyncTask;
import android.os.Build;
import android.os.Bundle;
import android.os.CountDownTimer;
import android.os.Environment;
import android.os.Parcelable;
import android.preference.PreferenceManager;
import android.annotation.SuppressLint;
import android.app.Activity;
import android.app.DownloadManager;
import android.content.Context;
import android.content.Intent;
import android.content.SharedPreferences;
import android.content.SharedPreferences.Editor;
import android.graphics.Bitmap;
import android.provider.MediaStore;
import android.util.Base64;
import android.util.Log;
import android.view.KeyEvent;
import android.view.Menu;
import android.view.MenuItem;
import android.webkit.ConsoleMessage;
import android.webkit.CookieManager;
import android.webkit.HttpAuthHandler;
import android.webkit.JavascriptInterface;
import android.webkit.SslErrorHandler;
import android.webkit.ValueCallback;
import android.webkit.WebBackForwardList;
import android.webkit.WebChromeClient;
import android.webkit.WebResourceError;
import android.webkit.WebResourceRequest;
import android.webkit.WebResourceResponse;
import android.webkit.WebSettings;
import android.webkit.WebView;
import android.webkit.WebViewClient;
import android.webkit.WebViewDatabase;
import android.widget.ProgressBar;
import android.widget.Toast;
import android.app.AlertDialog;

import androidx.core.content.FileProvider;
import androidx.security.crypto.EncryptedSharedPreferences;
import androidx.security.crypto.MasterKeys;
import androidx.swiperefreshlayout.widget.SwipeRefreshLayout;


/**
 * Second activity class
 */
@SuppressLint("SetJavaScriptEnabled")
public class SecondActivity extends Activity {

	private static final String LOG_TAG = "DoliDroidLogSecondActivity";
	public static final String VERSION_RESOURCES = "22.0";

    boolean isInstalledFromPlayStore = true;

    private WebView myWebView;
	WebViewClientDoliDroid myWebViewClientDoliDroid;
	WebChromeClientDoliDroid myWebChromeClientDoliDroid;
	WebBackForwardList mWebBackForwardList;

	private ValueCallback<Uri[]> mFilePathCallback;

	private String savedDolRootUrl;
	private String savedDolRootUrlWithSForced;
	private String savedDolRootUrlRel;
	private String savedDolScheme;
	private int savedDolPort;
	private String savedDolHost;
	private String savedDolUserInfoEncoded;
	private String savedDolBasedUrl;
	private String savedDolBasedUrlWithSForced;
	private String savedDolBasedUrlWithoutUserInfo;
    private String savedDolBasedUrlWithoutUserInfoWithSForced;
	private String savedAuthuser=null;
	private String savedAuthpass=null;
	private String savedUserAgent=null;

	private String saveQueryForonRequestPermissionsResult;
	private String saveUrlForonRequestPermissionsResult;
	private String saveListOfCookiesForonRequestPermissionsResult;

	private boolean prefAlwaysUseLocalResources=true;
	public boolean sslErrorWasAccepted=false;
    public boolean httpWarningWasViewed=false;

	private String lastversionfound;
	private String lastversionfoundforasset;


	// Variables used to manage cache and error retry
	private boolean tagToOverwriteLoginPass=true;
	private boolean tagLastLoginPassToSavedLoginPass=false;	// This is set to true after submitting login form
	private boolean tagToLogout=false;
	private String tagToShowInterruptMessage="";
	private int tagToShowInterruptCounter=0;
	private String tagToShowMessage="";
	private int tagToShowCounter=0;
    private int tagClearHistoryAfterFinished=0;

	private String cacheForMenu;
	private String cacheForQuickAccess;
    private String cacheForBookmarks;
    private String cacheForUploadFile;
    private String cacheForMultiCompany;
    private String cacheForVirtualCard;

	private String lastLoadUrl;
    private boolean	isBookmarkOn=true;
	private boolean	isMulticompanyOn=false;     // Not visible by default
    private boolean	isUploadFileOn=false;     // Not visible by default
    private boolean	isUserCardOn=false;      // Not visible by default
    private boolean	isVirtualCardOn=false;      // Not visible by default

	private Menu savMenu;
	private boolean messageNoPreviousPageShown =false;
	String listOfCookiesAfterLogon=null;

	private String mCameraPhotoPathString;
    Uri imageUri;
    Uri outputFileUri;

    final Activity activity = this;
    private ProgressBar progress;

    static final int RESULT_SECONDACTIVITY = RESULT_FIRST_USER;

    static final int RESULT_WEBVIEW =  RESULT_FIRST_USER+1;

    static final int REQUEST_ABOUT = RESULT_FIRST_USER;

    static final int REQUEST_INPUTFILE = RESULT_FIRST_USER+2;    // Use to trap file chooser on input field

    static final int REQUEST_ABOUT_INSTANCE = RESULT_FIRST_USER;

    static final int REQUEST_CODE_ASK_PERMISSIONS_WRITE_EXTERNAL_STORAGE = 123;

    // For inapp purchases (public key is found into menu "Services and API" for application into Google play publish center).
    //IabHelper iabHelper;
    //key dolidroid pro:
    //final String PUBLIC_KEY = "MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAtKWPkZ1rys0aYT9qQ7gHytljus58x9ZNwFUabsXgRAua2RwVkHnFfc8L2p68ojIb2tNHiRvMV6hYH2qViylftEMSYLFoKnuHzpL4tc+Ic+cTv/KtubP+ehUfISPQfYrZrukp3E8y0zM795Agsy8mefc2mmuOFJny/IZFLNyM5J+vjhoE6mO2l3jBmo08zu/3tz8Mbo/VYqJSs+P9UTppwF8ovB6u3fGPFeqblAdGize9WQ1L4SXNYblIjCklYj0rbXHFN3aJCjV9sSo0U+qdi6i+mT+CZgj09W1+U7RpkNJ6OczspTwhFh7/1nEev3Zci17TIFXNyP2v5aGMoBuCPwIDAQAB";
    //public static final String ITEM_SKU = "android.test.purchased";

    private final Pattern patternLoginHomePageForVersion = Pattern.compile(" (?:[@-]) (?:Doli[a-zA-Z]+ |)(\\d+)\\.(\\d+)\\.([^\\s]+)");     // Regex to extract version
    private final Pattern patternLoginHomePageForMulticompany = Pattern.compile("multicompany");                                    // Regex to know if multicompany module is on
    private final Pattern patternLoginPage = Pattern.compile("Login Doli[a-zA-Z]+ (\\d+)\\.(\\d+)\\.([^\\s]+)"); // No more used                    // To know page is login page with dolibarr <= 3.6
    private final Pattern patternLoginPage2 = Pattern.compile(" @ (?:Doli[a-zA-Z]+ |)(\\d+)\\.(\\d+)\\.([^\\s]+)");                  // To know page is login page with dolibarr >= 3.7
    
    private String nextAltHistoryStack = "";
    private String nextAltHistoryStackBis = "";
    ArrayList<String> altHistoryStack = new ArrayList<>();

    SwipeRefreshLayout swipe;

    // To store data for the download manager
    //final String strPref_Download_ID = "PREF_DOWNLOAD_ID";


    // This is a UI Thread
    @SuppressLint("SetJavaScriptEnabled")
    @Override
    public void onCreate(Bundle savedInstanceState) 
    {
        Log.i(LOG_TAG, "onCreate savedInstanceState="+savedInstanceState);

        super.onCreate(savedInstanceState);
        // To have the view SecondActivity with WebView included:
        setContentView(R.layout.activity_second);

        PackageManager packageManager = this.getPackageManager();
        String installerPackageName = null;
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.R) {
            try {
                InstallSourceInfo installSourceInfo = packageManager.getInstallSourceInfo(this.getPackageName());
                installerPackageName = installSourceInfo.getInstallingPackageName();
            } catch (PackageManager.NameNotFoundException e) {
                Log.e(LOG_TAG, "Could not get installer package name", e);
            }
        } else {
            // For older Android versions, you can keep the deprecated method,
            // though be aware of its limitations.
            // It's good practice to suppress the deprecation warning for this specific case.
            @SuppressWarnings("deprecation")
            String deprecatedInstallerPackageName = packageManager.getInstallerPackageName(this.getPackageName());
            installerPackageName = deprecatedInstallerPackageName;
        }

        isInstalledFromPlayStore = "com.android.vending".equals(installerPackageName);

        Log.d(LOG_TAG, "onCreate App is installed from: "+installerPackageName);

        // Read the non encrypted share preferences files
        //SharedPreferences sharedPrefs = PreferenceManager.getDefaultSharedPreferences(getApplicationContext());
        SharedPreferences sharedPrefs = getApplicationContext().getSharedPreferences("shared_prefs", Context.MODE_PRIVATE);

        boolean prefAlwaysAutoFill = sharedPrefs.getBoolean("prefAlwaysAutoFill", true);
        prefAlwaysUseLocalResources = sharedPrefs.getBoolean("prefAlwaysUseLocalResources", true);
        Log.d(LOG_TAG, "onCreate Read the non encrypted shared preferences file: prefAlwaysAutoFill="+prefAlwaysAutoFill+" prefAlwaysUseLocResouces="+prefAlwaysUseLocalResources);
        
        tagToOverwriteLoginPass=prefAlwaysAutoFill;

        // Define kind of menu we want to use
        boolean hasMenuHardware = Utils.hasMenuHardware(this);
        Log.d(LOG_TAG, "onCreate hasMenuHardware="+hasMenuHardware);

        Intent intent = getIntent();
        String dolRootUrl = intent.getStringExtra("dolRootUrl");
        String dolRequestUrl = intent.getStringExtra("dolRequestUrl");        

        this.savedDolRootUrl = dolRootUrl;      // this include user:pass of http basic urls. Always end with /. Example: hTtP://user:pass@testldr1.with.dolicloud.com:xxx/
        if (this.savedDolRootUrl != null) {
            this.savedDolRootUrl = this.savedDolRootUrl.replaceAll("^(?i)http(s?):", "http$1:");
        }
        this.savedDolScheme=Uri.parse(this.savedDolRootUrl).getScheme();                     // Example: http
        this.savedDolPort=Uri.parse(this.savedDolRootUrl).getPort();
        this.savedDolHost=Uri.parse(this.savedDolRootUrl).getHost();
        this.savedDolUserInfoEncoded=Uri.parse(this.savedDolRootUrl).getEncodedUserInfo();   // user:pass
        if (this.savedDolUserInfoEncoded == null) {
            this.savedDolUserInfoEncoded = "";
        }
        if (this.savedDolScheme != null)
            this.savedDolScheme = this.savedDolScheme.toLowerCase(Locale.ROOT);
        boolean includePort = (this.savedDolPort > 0);  // Do we have to include the port into the base url ?
        if ("http".equals(this.savedDolScheme) && this.savedDolPort == 80) {
            includePort = false;
            this.savedDolRootUrl = this.savedDolRootUrl.replace(":80", "");
        }
        if ("https".equals(this.savedDolScheme) && this.savedDolPort == 443) {
            includePort = false;
            this.savedDolRootUrl = this.savedDolRootUrl.replace(":443", "");
        }
        this.savedDolRootUrlWithSForced = "https:"+this.savedDolRootUrl.replace("http:", "").replace("https:", "");
        this.savedDolBasedUrl = this.savedDolScheme+"://"+this.savedDolUserInfoEncoded+("".equals(this.savedDolUserInfoEncoded) ? "" : "@")+this.savedDolHost+(includePort ? ":"+this.savedDolPort : "");   // Example: http://user:pass@testldr1.with.dolicloud.com:xxx
        this.savedDolBasedUrlWithSForced = "https:"+this.savedDolBasedUrl.replace("http:", "").replace("https:", "");
        this.savedDolBasedUrlWithoutUserInfo = this.savedDolScheme+"://"+this.savedDolHost+(includePort ? ":"+this.savedDolPort : "");	// Example: http://testldr1.with.dolicloud.com
        this.savedDolBasedUrlWithoutUserInfoWithSForced = "https:"+this.savedDolBasedUrlWithoutUserInfo.replace("http:", "").replace("https:", "");

		this.savedDolRootUrlRel = this.savedDolRootUrl.replace(this.savedDolBasedUrl, "");	// Rest of url, example: /

        try {
            URL uri=new URL(this.savedDolRootUrl);
            String userInfo=uri.getUserInfo();
            if (userInfo != null)
            {
                String[] credentials = userInfo.split(":", 2);
                //view.setHttpAuthUsernamePassword(getBaseDomain(url), "Restricted", credentials[0], credentials[1]);
                savedAuthuser=credentials[0];
                if (credentials.length > 1) {
                    savedAuthpass = credentials[1];
                }
                Log.d(LOG_TAG, "onCreate Saving basic authentication found into URL authuser="+savedAuthuser+" authpass="+savedAuthpass);
            }
            else
            {
                Log.d(LOG_TAG, "onCreate No basic authentication info into URL");
                savedAuthuser=null;
                savedAuthpass=null;
            }
        }
        catch (MalformedURLException e) {
            if (e.getMessage() == null) {
                Log.w(LOG_TAG, "Malformed URL Exception");
            } else {
                Log.w(LOG_TAG, e.getMessage());
            }
        }
        Log.d(LOG_TAG, "onCreate We have original root url = "+dolRootUrl);
        Log.d(LOG_TAG, "onCreate => savedDolRootUrl=" + this.savedDolRootUrl + " - savedDolRootUrlRel=" + this.savedDolRootUrlRel + " - savedDolRootUrlWithSForced = " + this.savedDolRootUrlWithSForced);
        Log.d(LOG_TAG, "onCreate => savedDolBasedUrl=" + this.savedDolBasedUrl + " - savedDolBasedUrlWithSForced=" + this.savedDolBasedUrlWithSForced);

        String urlToGo;
        if (! dolRequestUrl.contains("?") && ! dolRequestUrl.contains(".php")) urlToGo = dolRequestUrl+"index.php?dol_hide_topmenu=1&dol_hide_leftmenu=1&dol_optimize_smallscreen=1&dol_no_mouse_hover=1&dol_use_jmobile=1";
        else if (dolRequestUrl.contains("?")) urlToGo = dolRequestUrl+"&dol_hide_topmenu=1&dol_hide_leftmenu=1&dol_optimize_smallscreen=1&dol_no_mouse_hover=1&dol_use_jmobile=1";
        else urlToGo = dolRequestUrl+"?dol_hide_topmenu=1&dol_hide_leftmenu=1&dol_optimize_smallscreen=1&dol_no_mouse_hover=1&dol_use_jmobile=1";
        
        Log.d(LOG_TAG, "onCreate isDownloadManagerAvailable="+Utils.isDownloadManagerAvailable(this));
        Log.d(LOG_TAG, "onCreate We are in onCreate and will load URL urlToGo=" + urlToGo);

        progress = findViewById(R.id.progressBar1);
        progress.setMax(100);
        /*Drawable d=getResources().getDrawable(R.drawable.progressbar_style);
        ClipDrawable cd = new ClipDrawable(d, Gravity.START, ClipDrawable.HORIZONTAL);
        progress.setProgressDrawable(cd);*/

        myWebView = findViewById(R.id.webViewContent);
        myWebView.clearCache(true);
        myWebView.clearHistory();

        this.savedUserAgent = myWebView.getSettings().getUserAgentString() + " - " + getString(R.string.dolidroidUserAgent);

        myWebView.getSettings().setAllowContentAccess(true);
        myWebView.getSettings().setAllowFileAccess(true);
        //myWebView.getSettings().setAllowFileAccessFromFileURLs(true);
        //myWebView.getSettings().setAllowUniversalAccessFromFileURLs(true);
        myWebView.getSettings().setDomStorageEnabled(true);
        myWebView.getSettings().setSupportMultipleWindows(false);
        //myWebView.getSettings().setPluginsEnabled(true);
        //myWebView.getSettings().setJavaScriptCanOpenWindowsAutomatically(true);
        //myWebView.getSettings().setLoadWithOverviewMode(true);
        //myWebView.getSettings().setUseWideViewPort(false);
        //myWebView.getSettings().setSavePassword(false);
        //myWebView.getSettings().setSaveFormData(false);
        myWebView.getSettings().setUserAgentString(this.savedUserAgent);
        // Cache for no network (we don't want it, so we use default)
        //myWebView.getSettings().setAppCacheMaxSize(1024 * 1024 * 8);

        //myWebView.getSettings().setCacheMode(WebSettings.LOAD_CACHE_ELSE_NETWORK); //LOAD_DEFAULT
        //myWebView.getSettings().setCacheMode(WebSettings.LOAD_NO_CACHE);
        myWebView.getSettings().setCacheMode(WebSettings.LOAD_DEFAULT);

        //myWebView.getSettings().setRenderPriority(RenderPriority.HIGH);
        //myWebView.setLayerType(View.LAYER_TYPE_SOFTWARE, null);

        // Init code to allow js injection and execution
        myWebView.getSettings().setJavaScriptEnabled(true);
        final MyJavaScriptInterface myJavaScriptInterface = new MyJavaScriptInterface(activity);
        myWebView.addJavascriptInterface(myJavaScriptInterface, "HTMLOUT");

        myWebView.getSettings().setBuiltInZoomControls(true);
        Log.d(LOG_TAG, "onCreate Method setDisplayZoomControls exists. Set to false.");
        myWebView.getSettings().setDisplayZoomControls(false);          // Works with Android 3.0+ (level 11)
        this.myWebViewClientDoliDroid=new WebViewClientDoliDroid(this);
        myWebView.setWebViewClient(this.myWebViewClientDoliDroid);
        this.myWebChromeClientDoliDroid = new WebChromeClientDoliDroid();
        myWebView.setWebChromeClient(this.myWebChromeClientDoliDroid);
        
        lastLoadUrl=urlToGo;
        myWebView.loadUrl(urlToGo);

        // Add handler for the Swipe
        swipe = (SwipeRefreshLayout) findViewById(R.id.swipeContainer);
        swipe.setOnRefreshListener(new SwipeRefreshLayout.OnRefreshListener() {
            @Override
            public void onRefresh() {
                //myWebView.clearCache(true);
                //myWebView.clearHistory();
                //myWebView.reload();

                // Code similar to menu "Reload page"
                String urlToGo = myWebView.getUrl();
                if (urlToGo != null) {
                    if (urlToGo.startsWith("data:text") || urlToGo.startsWith("about:blank")) {
                        urlToGo = savedDolRootUrl;
                    }

                    if (! urlToGo.contains("dol_hide_topmenu=")) urlToGo = urlToGo + (urlToGo.contains("?")?"&":"?") + "dol_hide_topmenu=1";
                    if (! urlToGo.contains("dol_hide_leftmenu=")) urlToGo = urlToGo + (urlToGo.contains("?")?"&":"?") + "dol_hide_leftmenu=1";
                    if (! urlToGo.contains("dol_optimize_smallscreen=")) urlToGo = urlToGo + (urlToGo.contains("?")?"&":"?") + "dol_optimize_smallscreen=1";
                    if (! urlToGo.contains("dol_no_mouse_hover=")) urlToGo = urlToGo + (urlToGo.contains("?")?"&":"?") + "dol_no_mouse_hover=1";
                    if (! urlToGo.contains("dol_use_jmobile=")) urlToGo = urlToGo + (urlToGo.contains("?")?"&":"?") + "dol_use_jmobile=1";
                    Log.d(LOG_TAG, "LoadUrl after Swipe : Load url "+urlToGo);

                    // Disable the "swipe refresh" icon because we already have the progressBar called progress that is shown each time we load a page
                    swipe.setRefreshing(false);

                    lastLoadUrl=urlToGo;
                    myWebView.loadUrl(urlToGo);
                }
            }
        });
    }


	
	/**
	 * Called when activity start
	 */
	@Override
    public void onStart() 
	{	
    	Log.i(LOG_TAG, "onStart");

    	super.onStart();

    	// We must reload menu (it may have been changed into other activities)
		invalidateOptionsMenu();
    }

    /**
      *  Load SmartPhone menu.
      *  Note the onPrepareOptionsMenu is called at different moment compared to onCreateOptionsMenu called once at creation.
      *
      *  @param  Menu        menu    Object menu to initialize
      *  @return boolean             true
      */
    @Override
    public boolean onCreateOptionsMenu(Menu menu)
    {
        Log.d(LOG_TAG, "onCreateOptionsMenu");

        getMenuInflater().inflate(R.menu.activity_second, menu);    // Deploy android menu

        menu.findItem(R.id.menu_menu).setShowAsAction(MenuItem.SHOW_AS_ACTION_ALWAYS);
        menu.findItem(R.id.menu_search).setShowAsAction(MenuItem.SHOW_AS_ACTION_ALWAYS);
        menu.findItem(R.id.menu_bookmarks).setShowAsAction(MenuItem.SHOW_AS_ACTION_ALWAYS);
        menu.findItem(R.id.menu_uploadfile).setShowAsAction(MenuItem.SHOW_AS_ACTION_ALWAYS);
        menu.findItem(R.id.menu_back).setShowAsAction(MenuItem.SHOW_AS_ACTION_ALWAYS);

        //menu.findItem(R.id.menu_photo).setShowAsAction(MenuItem.SHOW_AS_ACTION_IF_ROOM);   // Not enough room so we force it on dropdown menu.
        //menu.findItem(R.id.menu_scan).setShowAsAction(MenuItem.SHOW_AS_ACTION_IF_ROOM);   // Not enough room so we force it on dropdown menu.
        menu.findItem(R.id.menu_multicompany).setShowAsAction(MenuItem.SHOW_AS_ACTION_IF_ROOM);   // Not enough room so we force it on dropdown menu.
        menu.findItem(R.id.menu_usercard).setShowAsAction(MenuItem.SHOW_AS_ACTION_IF_ROOM);  // Not enough room so we force it on dropdown menu.
        menu.findItem(R.id.menu_virtualcard).setShowAsAction(MenuItem.SHOW_AS_ACTION_IF_ROOM);  // Not enough room so we force it on dropdown menu.

        //menu.findItem(R.id.menu_photo).setVisible(false);
        //menu.findItem(R.id.menu_scan).setVisible(false);
        menu.findItem(R.id.menu_multicompany).setVisible(false);
        menu.findItem(R.id.menu_usercard).setVisible(false);
        menu.findItem(R.id.menu_virtualcard).setVisible(false);


        // Handle for the non encrypted shared preferences file
        //SharedPreferences sharedPrefs = PreferenceManager.getDefaultSharedPreferences(getApplicationContext());
        SharedPreferences sharedPrefs = getApplicationContext().getSharedPreferences("shared_prefs", Context.MODE_PRIVATE);

        // Hide menu show bar if phone too old, change label otherwise
        MenuItem menuItem2 = menu.findItem(R.id.always_autofill);
        boolean prefAlwaysAutoFill = sharedPrefs.getBoolean("prefAlwaysAutoFill", true);
        Log.d(LOG_TAG, "onCreateOptionsMenu prefAlwaysAutoFill value is "+prefAlwaysAutoFill);
        menuItem2.setChecked(prefAlwaysAutoFill);

        // Update menu label to add the number of predefined URL into label
        MenuItem tmpItem = menu.findItem(R.id.manage_all_urls);
        tmpItem.setVisible(false);
        if (MainActivity.listOfRootUrl != null) {
            tmpItem.setTitle(getString(R.string.menu_manage_all_urls) + " (" + MainActivity.listOfRootUrl.size() + ")");
        } else {
            tmpItem.setTitle(getString(R.string.menu_manage_all_urls) + " (0)");
        }

        MenuItem menuItem4 = menu.findItem(R.id.always_uselocalresources);
        Log.d(LOG_TAG, "onCreateOptionsMenu prefAlwaysUseLocalResources value is "+prefAlwaysUseLocalResources);
        menuItem4.setChecked(prefAlwaysUseLocalResources);

        if (isBookmarkOn) {
            Log.d(LOG_TAG, "onCreateOptionsMenu Bookmark feature must be on, we show picto");
            MenuItem menuItemBookmarks = menu.findItem(R.id.menu_bookmarks);
            if (menuItemBookmarks != null) {
                menuItemBookmarks.setVisible(true);
            }
        } else {
            Log.d(LOG_TAG, "onCreateOptionsMenu Bookmark feature must be disabled, we hide picto");
            MenuItem menuItemBookmarks = menu.findItem(R.id.menu_bookmarks);
            if (menuItemBookmarks != null) {
                menuItemBookmarks.setVisible(false);
            }
        }

        if (isMulticompanyOn) {
            Log.d(LOG_TAG, "onCreateOptionsMenu Module multicompany was found, we show picto");
            MenuItem menuItem5 = menu.findItem(R.id.menu_multicompany);
            if (menuItem5 != null) {
                menuItem5.setVisible(true);
            }
        } else {
            Log.d(LOG_TAG, "onCreateOptionsMenu Module multicompany was NOT found, we hide picto");
            MenuItem menuItem5 = menu.findItem(R.id.menu_multicompany);
            if (menuItem5 != null) {
                menuItem5.setVisible(false);
            }
        }

        if (isUploadFileOn) {
            Log.d(LOG_TAG, "onCreateOptionsMenu Upload file feature enabled, we show picto");
            MenuItem menuItem6 = menu.findItem(R.id.menu_uploadfile);
            if (menuItem6 != null) {
                menuItem6.setVisible(true);
            }
        } else {
            Log.d(LOG_TAG, "onCreateOptionsMenu Upload file feature is NOT enabled, we hide picto");
            MenuItem menuItem6 = menu.findItem(R.id.menu_uploadfile);
            if (menuItem6 != null) {
                menuItem6.setVisible(false);
            }
        }

        if (isUserCardOn) {
            Log.d(LOG_TAG, "onCreateOptionsMenu version enough for user card, we show menu/picto");
            MenuItem tmpmenu = menu.findItem(R.id.menu_usercard);
            if (tmpmenu != null) {
                tmpmenu.setVisible(true);
            }
        } else {
            Log.d(LOG_TAG, "onCreateOptionsMenu version NOT enough for virtual card, we hide menu/picto");
            MenuItem tmpmenu = menu.findItem(R.id.menu_usercard);
            if (tmpmenu != null) {
                tmpmenu.setVisible(false);
            }
        }

        if (isVirtualCardOn) {
            Log.d(LOG_TAG, "onCreateOptionsMenu version enough for virtual card, we show menu/picto");
            MenuItem tmpmenu = menu.findItem(R.id.menu_virtualcard);
            if (tmpmenu != null) {
                tmpmenu.setVisible(true);
            }
        } else {
            Log.d(LOG_TAG, "onCreateOptionsMenu version NOT enough for virtual card, we hide menu/picto");
            MenuItem tmpmenu = menu.findItem(R.id.menu_virtualcard);
            if (tmpmenu != null) {
                tmpmenu.setVisible(false);
            }
        }

        Log.d(LOG_TAG, "onCreateOptionsMenu Add menu About");
        MenuItem menuItemAbout = menu.findItem(R.id.about);
        if (menuItemAbout != null) {
            menuItemAbout.setVisible(true);
            //menuItemAbout.setIcon(getDrawable(R.drawable.ic_baseline_help_outline_24));
        }

        if (MainActivity.listOfRootUrl != null) {
            MenuItem tmpItem2 = menu.findItem(R.id.manage_all_urls);
            tmpItem2.setTitle(getString(R.string.menu_manage_all_urls) + " (" + MainActivity.listOfRootUrl.size() + ")");
            tmpItem2.setVisible(true);
        }

        Log.d(LOG_TAG, "onCreateOptionsMenu Add menu Clear cache");
        MenuItem menuItemClearCache = menu.findItem(R.id.clearcache);
        if (menuItemClearCache != null) {
            menuItemClearCache.setVisible(true);
            menuItemClearCache.setIcon(getDrawable(R.drawable.ic_baseline_clear_24));
        }

        Log.d(LOG_TAG, "onCreateOptionsMenu Add menu Reload page");
        MenuItem menuItemReloadPage = menu.findItem(R.id.refresh);
        if (menuItemReloadPage != null) {
            menuItemReloadPage.setVisible(true);
            menuItemReloadPage.setIcon(getDrawable(R.drawable.ic_baseline_refresh_24));
        }

        Log.d(LOG_TAG, "onCreateOptionsMenu Add menu Copy url");
        MenuItem menuItemAddLink = menu.findItem(R.id.menu_copy_url);
        if (menuItemAddLink != null) {
            menuItemAddLink.setVisible(true);
            menuItemAddLink.setIcon(getDrawable(R.drawable.ic_copy));
        }

        Log.d(LOG_TAG, "onCreateOptionsMenu Add menu About instance");
        MenuItem menuItemAboutInstance = menu.findItem(R.id.menu_aboutinstance);
        if (menuItemAboutInstance != null) {
            menuItemAboutInstance.setVisible(true);
            menuItemAboutInstance.setIcon(getDrawable(R.drawable.ic_baseline_question_mark_24));
        }

        Log.d(LOG_TAG, "onCreateOptionsMenu Add menu Logout");
        MenuItem menuItemLogout = menu.findItem(R.id.menu_logout);
        if (menuItemLogout != null) {
            menuItemLogout.setVisible(true);
            menuItemLogout.setIcon(getDrawable(R.drawable.ic_baseline_exit_to_app_24));
        }

        this.savMenu = menu;
        
        return true;
    }

    /**
     *  Once we selected a menu option
     *
     *  @param  MenuItem    item    Menu item selected
     *  @return boolean             True if we selected a menu managed, False otherwise
     */
    public boolean onOptionsItemSelected(MenuItem item)
    {
        Log.i(LOG_TAG, "SecondActivity::onOptionsItemSelected Click onto menu: item="+item.toString());

        SharedPreferences sharedPrefs;
        Editor editor;
        String urlToGo = ""; 

        // On which menu entry did you click ?
        switch (item.getItemId())
        {
            case R.id.menu_menu:
                return this.codeForMenu();
            case R.id.menu_search:
                return this.codeForQuickAccess();
            case R.id.menu_back:
                return this.codeForBack();
            case R.id.menu_bookmarks:
                return this.codeForBookmarks();
            case R.id.menu_multicompany:
                return this.codeForMultiCompany();
            case R.id.menu_uploadfile:
                return this.codeForUploadFile();
            case R.id.menu_usercard:
                return this.codeForUserCard();
            case R.id.menu_virtualcard:
                return this.codeForVirtualCard();
            case R.id.menu_copy_url:
                return this.codeForCopyUrl();
            case R.id.menu_aboutinstance:
                Log.i(LOG_TAG, "Start activity About instances");
                //myWebView = findViewById(R.id.webViewContent);
                Intent intentaboutinstance = new Intent(SecondActivity.this, AboutInstanceActivity.class);
                intentaboutinstance.putExtra("currentUrl", myWebView.getOriginalUrl());
                intentaboutinstance.putExtra("userAgent", myWebView.getSettings().getUserAgentString());
                intentaboutinstance.putExtra("savedDolRootUrl", this.savedDolRootUrl);
                intentaboutinstance.putExtra("lastversionfound", this.lastversionfound);
                intentaboutinstance.putExtra("lastversionfoundforasset", this.lastversionfoundforasset);
                intentaboutinstance.putExtra("title", myWebView.getTitle());
                intentaboutinstance.putExtra("savedAuthuser", this.savedAuthuser);
                intentaboutinstance.putExtra("savedAuthpass", this.savedAuthpass);
                Log.d(LOG_TAG, "startActivityForResult with requestCode="+REQUEST_ABOUT_INSTANCE);
                startActivityForResult(intentaboutinstance, REQUEST_ABOUT_INSTANCE);
                return true;
            case R.id.always_autofill:  // Switch menu bar on/off for "Save login/password"
                sharedPrefs = PreferenceManager.getDefaultSharedPreferences(getApplicationContext());
                //sharedPrefs = getApplicationContext().getSharedPreferences("shared_prefs", Context.MODE_PRIVATE);

                // Same code into MainActivity and SecondActivity
                boolean prefAlwaysAutoFill = sharedPrefs.getBoolean("prefAlwaysAutoFill", true);

                Log.i(LOG_TAG, "Click onto switch autofill, prefAlwaysAutoFill is "+prefAlwaysAutoFill);
                prefAlwaysAutoFill=!prefAlwaysAutoFill;

                editor = sharedPrefs.edit();
                editor.putBoolean("prefAlwaysAutoFill", prefAlwaysAutoFill);
                editor.apply();
                Log.d(LOG_TAG, "Switched value is now "+prefAlwaysAutoFill);
                // Update show bar or not
                if (prefAlwaysAutoFill) {
                    this.savMenu.findItem(R.id.always_autofill).setTitle(getString(R.string.menu_autofill_on));
                } else {
                    this.savMenu.findItem(R.id.always_autofill).setTitle(getString(R.string.menu_autofill_off));

                    // Clear saved login / pass
                    try {
                        String masterKeyAlias = MasterKeys.getOrCreate(MasterKeys.AES256_GCM_SPEC);
                        SharedPreferences sharedPrefsEncrypted = EncryptedSharedPreferences.create(
                                "secret_shared_prefs",
                                masterKeyAlias,
                                getApplicationContext(),
                                EncryptedSharedPreferences.PrefKeyEncryptionScheme.AES256_SIV,
                                EncryptedSharedPreferences.PrefValueEncryptionScheme.AES256_GCM
                        );
                        Editor editorEncrypted = sharedPrefsEncrypted.edit();
                        editorEncrypted.clear();
                        editorEncrypted.commit();

                        Log.d(LOG_TAG, "The encrypted shared preferences file has been cleared");
                    }
                    catch(Exception e) {
                        Log.w(LOG_TAG, "Failed to clear encrypted shared preferences file, we try by deleting the file");
                        File prefsFile = new File(getApplicationContext().getFilesDir(), "../shared_prefs/secret_shared_prefs.xml");
                        if (prefsFile.exists()) {
                            Log.d(LOG_TAG, "File "+prefsFile+" exists, we delete it");
                            try {
                                boolean deleted = prefsFile.delete();
                                Log.d(LOG_TAG, "File " + prefsFile + " deleted = " + deleted);
                            } catch(Exception e2) {
                                // Keep empty
                                Log.d(LOG_TAG, "Failed to delete file " + prefsFile);
                            }
                        }

                    }
                }
                invalidateOptionsMenu();
                return true;
            case R.id.always_uselocalresources:  // Switch menu bar on/off for "Use local static resources"
                Log.i(LOG_TAG, "Click onto switch uselocalresources, prefAlwaysUseLocalResources is "+prefAlwaysUseLocalResources);
                prefAlwaysUseLocalResources=!prefAlwaysUseLocalResources;

                sharedPrefs = PreferenceManager.getDefaultSharedPreferences(getApplicationContext());
                //sharedPrefs = getApplicationContext().getSharedPreferences("shared_prefs", Context.MODE_PRIVATE);

                editor = sharedPrefs.edit();
                editor.putBoolean("prefAlwaysUseLocalResources", prefAlwaysUseLocalResources);
                editor.apply();

                Log.d(LOG_TAG, "Switched value is now "+prefAlwaysUseLocalResources);
                // Update menu label
                if (prefAlwaysUseLocalResources) {
                    this.savMenu.findItem(R.id.always_uselocalresources).setTitle(getString(R.string.menu_uselocalresources_on));
                } else {
                    this.savMenu.findItem(R.id.always_uselocalresources).setTitle(getString(R.string.menu_uselocalresources_off));
                }
                invalidateOptionsMenu();
                return true;
            case R.id.manage_all_urls:
                Log.i(LOG_TAG, "Start activity Manage URLs");
                //myWebView = findViewById(R.id.webViewContent);
                Intent tmpintent1 = new Intent(SecondActivity.this, ManageURLActivity.class);
                tmpintent1.putExtra("currentUrl", myWebView.getOriginalUrl());
                tmpintent1.putExtra("userAgent", myWebView.getSettings().getUserAgentString());
                tmpintent1.putExtra("savedDolRootUrl", this.savedDolRootUrl);
                tmpintent1.putExtra("lastversionfound", this.lastversionfound);
                tmpintent1.putExtra("lastversionfoundforasset", this.lastversionfoundforasset);
                tmpintent1.putExtra("title", myWebView.getTitle());
                tmpintent1.putExtra("savedAuthuser", this.savedAuthuser);
                tmpintent1.putExtra("savedAuthpass", this.savedAuthpass);
                Log.d(LOG_TAG, "startActivityForResult with requestCode="+REQUEST_ABOUT);
                startActivityForResult(tmpintent1, REQUEST_ABOUT);
                return true;
            case R.id.about:
                Log.i(LOG_TAG, "Start activity About DoliDroid");
                //myWebView = findViewById(R.id.webViewContent);
                Intent intent = new Intent(SecondActivity.this, AboutActivity.class);
                intent.putExtra("currentUrl", myWebView.getOriginalUrl());
                intent.putExtra("userAgent", myWebView.getSettings().getUserAgentString());
                intent.putExtra("savedDolRootUrl", this.savedDolRootUrl);
                intent.putExtra("lastversionfound", this.lastversionfound);
                intent.putExtra("lastversionfoundforasset", this.lastversionfoundforasset);
                intent.putExtra("title", myWebView.getTitle());
                intent.putExtra("savedAuthuser", this.savedAuthuser);
                intent.putExtra("savedAuthpass", this.savedAuthpass);
                Log.d(LOG_TAG, "startActivityForResult with requestCode="+REQUEST_ABOUT);
                startActivityForResult(intent, REQUEST_ABOUT);
                return true;
            case R.id.menu_logout:
                tagToLogout=true;
                //myWebView = findViewById(R.id.webViewContent);
                urlToGo = this.savedDolRootUrl+"user/logout.php?noredirect=1&dol_hide_topmenu=1&dol_hide_leftmenu=1&dol_optimize_smallscreen=1&dol_no_mouse_hover=1&dol_use_jmobile=1";
                Log.i(LOG_TAG, "LoadUrl after select Logout : "+urlToGo);
                lastLoadUrl=urlToGo;

                myWebView.loadUrl(urlToGo);
                Log.i(LOG_TAG, "Clear caches and history of webView");
                myWebView.clearCache(true);
                myWebView.clearHistory();
                return true;
            case R.id.quit:
                Log.i(LOG_TAG, "Call finish activity, with setResult = "+RESULT_WEBVIEW);
                setResult(RESULT_WEBVIEW);
                finish();
                return true;
            case R.id.clearcache:   // Action to clear webview caches
                //myWebView = findViewById(R.id.webViewContent);
                Log.i(LOG_TAG, "Clear caches and history of webView");
                myWebView.clearFormData();
                myWebView.clearHistory();
                myWebView.clearCache(true);
                //WebStorage.getInstance().deleteAllData();
                this.cacheForMenu=null;
                this.cacheForQuickAccess=null;
                this.cacheForBookmarks=null;
                this.cacheForUploadFile=null;
                this.cacheForMultiCompany=null;
                this.cacheForVirtualCard=null;

                Toast.makeText(activity, R.string.CacheAndHistoryCleared, Toast.LENGTH_LONG).show();

                return true;
            case R.id.refresh:
                //myWebView = findViewById(R.id.webViewContent);
                urlToGo = myWebView.getUrl();
                Log.d(LOG_TAG, "urlToGo="+urlToGo);
                if (urlToGo != null) {
                    if (urlToGo.startsWith("data:text") || urlToGo.startsWith("about:blank")) {
                        urlToGo = savedDolRootUrl;
                    }

                    if (! urlToGo.contains("dol_hide_topmenu=")) urlToGo = urlToGo + (urlToGo.contains("?")?"&":"?") + "dol_hide_topmenu=1";
                    if (! urlToGo.contains("dol_hide_leftmenu=")) urlToGo = urlToGo + (urlToGo.contains("?")?"&":"?") + "dol_hide_leftmenu=1";
                    if (! urlToGo.contains("dol_optimize_smallscreen=")) urlToGo = urlToGo + (urlToGo.contains("?")?"&":"?") + "dol_optimize_smallscreen=1";
                    if (! urlToGo.contains("dol_no_mouse_hover=")) urlToGo = urlToGo + (urlToGo.contains("?")?"&":"?") + "dol_no_mouse_hover=1";
                    if (! urlToGo.contains("dol_use_jmobile=")) urlToGo = urlToGo + (urlToGo.contains("?")?"&":"?") + "dol_use_jmobile=1";
                    Log.d(LOG_TAG, "LoadUrl after select Refresh : Load url "+urlToGo);
                    lastLoadUrl=urlToGo;
                    myWebView.loadUrl(urlToGo);
                }
                return true;
         }

        Log.w(LOG_TAG, "Click onto unknown button "+item.getItemId());
        return false;
    }


    /**
     * Return a DefaultHTTPClient with option to support untrusted HTTPS
     * This object can be used to get content of a HTTP page from code.
     *
     * @return HttpClient       Object derivated from DefaultHttpClient
     */
    public HttpClient getNewHttpClient() {
        Log.d(LOG_TAG, "getNewHttpClient");

        try {
            /*
            Disabled, we do not use DefaultHttpClient with custom SSLSocketFactory to manage custom validation of certificate
            For getting page content from code, we use now the default DefaultHttpClient().
            It means it works only is HTTP certificate is valid.

            KeyStore trustStore = KeyStore.getInstance(KeyStore.getDefaultType());
            trustStore.load(null, null);
            //trustStore.setCertificateEntry("caCert", caCert);
            MySSLSocketFactory sf = new MySSLSocketFactory(trustStore);

            HttpParams params = new BasicHttpParams();
            HttpProtocolParams.setVersion(params, HttpVersion.HTTP_1_1);
            HttpProtocolParams.setContentCharset(params, HTTP.UTF_8);

            // Now we set the handler for http and https
            SchemeRegistry registry = new SchemeRegistry();
            registry.register(new Scheme("http", PlainSocketFactory.getSocketFactory(), 80));
            registry.register(new Scheme("https", (SocketFactory) sf, 443));

            ClientConnectionManager ccm = new ThreadSafeClientConnManager(params, null);

            return new DefaultHttpClient(ccm, params);
            */
            return new DefaultHttpClient(); // Will handle an error if HTTPS certificate is wrong.
        } catch (Exception e) {
            return new DefaultHttpClient();
        }
    }

    /**
     * Class to load an URL in background.
     * Used to load menu, quick search page and more...
     */
    private class DownloadWebPageTask extends AsyncTask<String, Void, String>
    {
        String mode;
        
        DownloadWebPageTask(String mode)
        {
            super();
            this.mode = mode;
        }

        /**
         * Launch download of urls. Return content of response.
         */
        @Override
        protected String doInBackground(String... urls) 
        {
            Log.d(LOG_TAG, "doInBackground");

            StringBuilder response = new StringBuilder();

            if (listOfCookiesAfterLogon != null) {      // We do not try to load url if cookies are not yet set
                for (String url : urls) {
                    //DefaultHttpClient client = new DefaultHttpClient();
                    // TODO Replace this with java.net.HttpURLConnection
                    HttpClient client = getNewHttpClient();
                    HttpGet httpGet = new HttpGet(url);
                    try {
                        Log.i(LOG_TAG, "doInBackground get url mode="+this.mode+" url="+url+" savedAuthuser="+savedAuthuser+" cookies="+listOfCookiesAfterLogon);

                        httpGet.setHeader("Cookie", listOfCookiesAfterLogon);
                        //httpGet.setHeader("Connection", "keep-alive");
                        httpGet.setHeader("User-Agent", savedUserAgent);
                        if (savedAuthuser != null) {
                            httpGet.setHeader("Authorization", "Basic " + Base64.encodeToString((savedAuthuser+":"+savedAuthpass).getBytes(), Base64.NO_WRAP));	// Add user/pass for basic authentication
                        }
                        String androlocale=Locale.getDefault().getLanguage();
                        if (! "".equals(androlocale)) {
                            httpGet.setHeader("Accept-Language", androlocale);
                        }

                        HttpResponse execute = client.execute(httpGet);
                        InputStream content = execute.getEntity().getContent();

                        BufferedReader buffer = new BufferedReader(new InputStreamReader(content));
                        String s;
                        while ((s = buffer.readLine()) != null) {
                            response.append(s);
                        }
                    }
                    catch (Exception e)
                    {
                        e.printStackTrace();
                    }
                }
            }

            return response.toString();
        }

        /**
         * When an url has been downloaded.
         * Used when download is done by the async Download manager after a DownloadWebPageTask.execute into codeForXXX for example.
         * Not called for common navigation (see instead shouldInterceptRequest, onPageStarted, onPageFinished)
         * 
         * @param   String      content downloaded
         */
        @Override
        protected void onPostExecute(String result) 
        {
            String stringforHistoryUrl = null;
            String stringToCheckInResult = null;

            if (result == null) {
                Log.i(LOG_TAG, "onPostExecute mode="+this.mode+" result=null");
            } else {
                Log.i(LOG_TAG, "onPostExecute mode="+this.mode+" result="+result.length());
            }

            if ("menu".equals(this.mode)) {     // Test that result is a menu
                stringforHistoryUrl = savedDolRootUrl+"core/get_menudiv.php?dol_hide_topmenu=1&dol_hide_leftmenu=1&dol_optimize_smallscreen=1&dol_no_mouse_hover=1&dol_use_jmobile=1";
                stringToCheckInResult = "<!-- Menu -->";
            }
            if ("quickaccess".equals(this.mode)) {     // Test that result is the search page
                stringforHistoryUrl = savedDolRootUrl+"core/search_page.php?dol_hide_topmenu=1&dol_hide_leftmenu=1&dol_optimize_smallscreen=1&dol_no_mouse_hover=1&dol_use_jmobile=1";
                stringToCheckInResult = "<!-- Quick access -->";
            }
            if ("bookmarks".equals(this.mode)) {     // Test that result is a bookmark page
                stringforHistoryUrl = savedDolRootUrl+"core/bookmarks_page.php?dol_hide_topmenu=1&dol_hide_leftmenu=1&dol_optimize_smallscreen=1&dol_no_mouse_hover=1&dol_use_jmobile=1";
                stringToCheckInResult = "<!-- Bookmarks -->";
            }
            if ("uploadfile".equals(this.mode)) {     // Test that result is a bookmark page
                stringforHistoryUrl = savedDolRootUrl+"core/upload_page.php?dol_hide_topmenu=1&dol_hide_leftmenu=1&dol_optimize_smallscreen=1&dol_no_mouse_hover=1&dol_use_jmobile=1";
                stringToCheckInResult = "<!-- Upload file -->";
            }
            if ("multicompany".equals(this.mode)) {     // Test that result is a multicompany selection page
                stringforHistoryUrl = savedDolRootUrl+"core/multicompany_page.php?dol_hide_topmenu=1&dol_hide_leftmenu=1&dol_optimize_smallscreen=1&dol_no_mouse_hover=1&dol_use_jmobile=1";
                stringToCheckInResult = "<!-- Multicompany selection  -->";
            }
            if ("virtualcard".equals(this.mode)) {     // Test that result is a multicompany selection page
                stringforHistoryUrl = savedDolRootUrl+"user/virtualcard.php?dol_hide_topmenu=1&dol_hide_leftmenu=1&dol_optimize_smallscreen=1&dol_no_mouse_hover=1&dol_use_jmobile=1";
                stringToCheckInResult = "<!-- Virtual card -->";
            }

            if (result != null && ! "".equals(result))
            {
                Log.d(LOG_TAG, "onPostExecute Check result of doInBackground mode="+this.mode+" savedDolBasedUrl="+savedDolBasedUrl+" stringforHistoryUrl="+stringforHistoryUrl);

                if (stringToCheckInResult != null) {     // Test that result is as expected
                    if (result.contains(stringToCheckInResult)) {
                        if (this.mode != null) {
                            // TODO Do not add same history url twice
                            nextAltHistoryStack = this.mode;       // this.mode is "menu", "quickaccess", ...
                        }

                        if ("menu".equals(this.mode)) {     // Test that result is a menu
                            cacheForMenu = result;
                        }
                        if ("quickaccess".equals(this.mode)) {     // Test that result is the quickaccess page
                            cacheForQuickAccess = result;
                        }
                        if ("bookmarks".equals(this.mode)) {     // Test that result is the quickaccess page
                            cacheForBookmarks = result;
                        }
                        if ("uploadfile".equals(this.mode)) {     // Test that result is the quickaccess page
                            cacheForUploadFile = result;
                        }
                        if ("virtualcard".equals(this.mode)) {     // Test that result is the quickaccess page
                            cacheForVirtualCard = result;
                        }
                        if ("multicompany".equals(this.mode)) {     // Test that result is the quickaccess page
                            cacheForMultiCompany = result;
                        }

                        // Generic download
                        Log.d(LOG_TAG, "onPostExecute Load content from result of doInBackground mode=" + this.mode + " savedDolBasedUrl=" + savedDolBasedUrl + " stringforHistoryUrl=" + stringforHistoryUrl);

                        myWebView.loadDataWithBaseURL(savedDolBasedUrl, result, "text/html", "UTF-8", stringforHistoryUrl);
                        //myWebView.loadData(result, "text/html", "UTF-8");     // This does not work
                    } else {
                        Log.d(LOG_TAG, "onPostExecute Failed to get page. Are you logged ?");
                        Toast.makeText(activity, getString(R.string.failedtogetpage), Toast.LENGTH_LONG).show();
                        Log.d(LOG_TAG, result);
                    }
                } else {    // Generic download
                    Log.d(LOG_TAG, "onPostExecute Load content from result of doInBackground mode=" + this.mode + " savedDolBasedUrl=" + savedDolBasedUrl + " stringforHistoryUrl=" + stringforHistoryUrl);
                    myWebView.loadDataWithBaseURL(savedDolBasedUrl, result, "text/html", "UTF-8", stringforHistoryUrl);
                    //myWebView.loadData(result, "text/html", "UTF-8");
                }
            }

            Log.d(LOG_TAG, "end onPostExecute (toremove)");
        }
    }

    
    /**
     * Once we click onto SmartPhone hardware key
     */
    @SuppressLint("GestureBackNavigation")
    @Override
    public boolean onKeyDown(int keyCode, KeyEvent event) 
    {
        if (event.getAction() == KeyEvent.ACTION_DOWN) {
            // Check if the key event was the Back button
            if ((keyCode == KeyEvent.KEYCODE_BACK)) {
                Log.d(LOG_TAG, "We pressed the key back on smartphone");
                return this.codeForBack();
            }
        }
        // If it wasn't the Back key or there's no web page history, bubble up to the default
        // system behavior (probably exit the activity)
        return super.onKeyDown(keyCode, event);
    } 


    /**
     * Common code for Menu
     * codeForMenu is in a UI thread
     * 
     * @return  boolean     true
     */
    private boolean codeForMenu() 
    {
        String urlToGo;
        boolean allowCacheForMenuPage = false;

        urlToGo = this.savedDolRootUrl+"core/get_menudiv.php?cache=600&dol_hide_topmenu=1&dol_hide_leftmenu=1&dol_optimize_smallscreen=1&dol_no_mouse_hover=1&dol_use_jmobile=1";

        // If not found into cache, call URL
        Log.d(LOG_TAG, "We called codeForMenu after click on Menu : savedDolBasedUrl="+this.savedDolBasedUrl+" urlToGo="+urlToGo);
        //myWebView = findViewById(R.id.webViewContent);

        /*
        if (allowCacheForMenuPage) {
            if (this.cacheForMenu != null && this.cacheForMenu.length() > 0) {
                String historyUrl = urlToGo;
                Log.d(LOG_TAG, "Got content from app cache this.cacheForMenu savedDolBasedUrl=" + this.savedDolBasedUrl + " historyUrl=" + historyUrl);
                //altHistoryStack.add("menu");  // TODO Do not add same history url twice
                nextAltHistoryStack = "menu";
                myWebView.loadDataWithBaseURL(this.savedDolBasedUrl, this.cacheForMenu, "text/html", "UTF-8", historyUrl);
            } else {
                DownloadWebPageTask task = new DownloadWebPageTask("menu");
                task.execute(new String[]{urlToGo});
            }
        } else { */
            myWebView.loadUrl(urlToGo);
        // }

        return true;
    }

    /**
     * Common code for Quick Access
     * codeForQuickAccess is in a UI thread
     *
     * @return  boolean     true
     */
    private boolean codeForQuickAccess() 
    {
        String urlToGo;
        boolean allowCacheForQuickAccessPage = false;

        urlToGo = this.savedDolRootUrl+"core/search_page.php?cache=600&dol_hide_topmenu=1&dol_hide_leftmenu=1&dol_optimize_smallscreen=1&dol_no_mouse_hover=1&dol_use_jmobile=1";

        // If not found into cache, call URL
        Log.d(LOG_TAG, "We called codeForQuickAccess after click on Search : savedDolBasedUrl="+this.savedDolBasedUrl+" urlToGo="+urlToGo);
        //myWebView = findViewById(R.id.webViewContent);

        /*
        if (allowCacheForQuickAccessPage) {
            if (this.cacheForQuickAccess != null && this.cacheForQuickAccess.length() > 0)
            {
                String historyUrl = urlToGo;
                Log.d(LOG_TAG, "Got content from app cache this.cacheForQuickAccess savedDolBasedUrl="+this.savedDolBasedUrl+" historyUrl="+historyUrl);
                //altHistoryStack.add("quickaccess");   // TODO Do not add same history url twice
                nextAltHistoryStack="quickaccess";
                myWebView.loadDataWithBaseURL(this.savedDolBasedUrl, this.cacheForQuickAccess, "text/html", "UTF-8", historyUrl);

                return true;
            }

            DownloadWebPageTask task = new DownloadWebPageTask("quickaccess");
            task.execute(new String[] { urlToGo });
        } else { */
            myWebView.loadUrl(urlToGo);
        // }

        return true;
    }

    /**
     * Common code for UploadFile
     * codeForUploadFile is in a UI thread
     *
     * @return  boolean             True
     */
    private boolean codeForUploadFile() {
        String urlToGo;
        boolean allowCacheForUploadFilePage = false;

        urlToGo = this.savedDolRootUrl+"core/upload_page.php?cache=600&dol_hide_topmenu=1&dol_hide_leftmenu=1&dol_optimize_smallscreen=1&dol_no_mouse_hover=1&dol_use_jmobile=1";

        // If not found into cache, call URL
        Log.d(LOG_TAG, "We called codeForUploadFile after click on Uploadfile : savedDolBasedUrl="+this.savedDolBasedUrl+" urlToGo="+urlToGo);

        //myWebView = findViewById(R.id.webViewContent);
        myWebView.loadUrl(urlToGo);

        return true;
    }

    /**
     * Common code for Bookmark
     * codeForBookmarks is in a UI thread
     *
     * @return  boolean             True
     */
    private boolean codeForBookmarks() {
        String urlToGo;
        boolean allowCacheForBookmarkPage = false;

        urlToGo = this.savedDolRootUrl+"core/bookmarks_page.php?cache=600&dol_hide_topmenu=1&dol_hide_leftmenu=1&dol_optimize_smallscreen=1&dol_no_mouse_hover=1&dol_use_jmobile=1";

        // If not found into cache, call URL
        Log.d(LOG_TAG, "We called codeForBookmarks after click on Bookmarks : savedDolBasedUrl="+this.savedDolBasedUrl+" urlToGo="+urlToGo);
        //myWebView = findViewById(R.id.webViewContent);

        /*if (allowCacheForBookmarkPage) {
            if (this.cacheForBookmarks != null && this.cacheForBookmarks.length() > 0) {
                String historyUrl = urlToGo;
                Log.d(LOG_TAG, "Got content from app cache this.cacheForBookmarks savedDolBasedUrl=" + this.savedDolBasedUrl + " historyUrl=" + historyUrl);
                //altHistoryStack.add("bookmarks");   // TODO Do not add same history url twice
                nextAltHistoryStack = "bookmarks";
                myWebView.loadDataWithBaseURL(this.savedDolBasedUrl, this.cacheForBookmarks, "text/html", "UTF-8", historyUrl);

                return true;
            }

            DownloadWebPageTask task = new DownloadWebPageTask("bookmarks");
            task.execute(new String[]{urlToGo});
        } else { */
            myWebView.loadUrl(urlToGo);
        //}

        return true;
    }

    /**
     * Common code for Back
     * codeForUserCard is in a UI thread
     *
     * @return  boolean             True
     */
    private boolean codeForUserCard() {
        String urlToGo;
        boolean allowCacheForUserCard = false;

        urlToGo = this.savedDolRootUrl+"user/card.php?dol_hide_topmenu=1&dol_hide_leftmenu=1&dol_optimize_smallscreen=1&dol_no_mouse_hover=1&dol_use_jmobile=1";

        // If not found into cache, call URL
        Log.d(LOG_TAG, "We called codeForUserCard after click on User Card : savedDolBasedUrl="+this.savedDolBasedUrl+" urlToGo="+urlToGo);
        //myWebView = findViewById(R.id.webViewContent);

        /*
        if (allowCacheForVirtualCard) {
            if (this.cacheForVirtualCard != null && this.cacheForVirtualCard.length() > 0) {
                String historyUrl = urlToGo;
                Log.d(LOG_TAG, "Got content from app cache this.cacheForVirtualCard savedDolBasedUrl=" + this.savedDolBasedUrl + " historyUrl=" + historyUrl);
                //altHistoryStack.add("bookmarks");   // TODO Do not add same history url twice
                nextAltHistoryStack = "bookmarks";
                myWebView.loadDataWithBaseURL(this.savedDolBasedUrl, this.cacheForVirtualCard, "text/html", "UTF-8", historyUrl);

                return true;
            }

            DownloadWebPageTask task = new DownloadWebPageTask("bookmarks");
            task.execute(new String[]{urlToGo});
        } else { */
            myWebView.loadUrl(urlToGo);
        //}

        return true;
    }

    /**
     * Common code for Back
     * codeForVirtualCard is in a UI thread
     *
     * @return  boolean             True
     */
    private boolean codeForVirtualCard() {
        String urlToGo;
        boolean allowCacheForVirtualCard = false;

        urlToGo = this.savedDolRootUrl+"user/virtualcard.php?cache=600&dol_hide_topmenu=1&dol_hide_leftmenu=1&dol_optimize_smallscreen=1&dol_no_mouse_hover=1&dol_use_jmobile=1";

        // If not found into cache, call URL
        Log.d(LOG_TAG, "We called codeForVirtualCard after click on Virtual Card : savedDolBasedUrl="+this.savedDolBasedUrl+" urlToGo="+urlToGo);
        //myWebView = findViewById(R.id.webViewContent);

        /*
        if (allowCacheForVirtualCard) {
            if (this.cacheForVirtualCard != null && this.cacheForVirtualCard.length() > 0) {
                String historyUrl = urlToGo;
                Log.d(LOG_TAG, "Got content from app cache this.cacheForVirtualCard savedDolBasedUrl=" + this.savedDolBasedUrl + " historyUrl=" + historyUrl);
                //altHistoryStack.add("bookmarks");   // TODO Do not add same history url twice
                nextAltHistoryStack = "bookmarks";
                myWebView.loadDataWithBaseURL(this.savedDolBasedUrl, this.cacheForVirtualCard, "text/html", "UTF-8", historyUrl);

                return true;
            }

            DownloadWebPageTask task = new DownloadWebPageTask("bookmarks");
            task.execute(new String[]{urlToGo});
        } else { */
            myWebView.loadUrl(urlToGo);
        //}

        return true;
    }

    /**
     * Common code for MultiCompany
     * codeForMultiCompany is in a UI thread
     *
     * @return  boolean     true
     */
    private boolean codeForMultiCompany()
    {
        String urlToGo;

        urlToGo = this.savedDolRootUrl+"core/multicompany_page.php?dol_hide_topmenu=1&dol_hide_leftmenu=1&dol_optimize_smallscreen=1&dol_no_mouse_hover=1&dol_use_jmobile=1";

        // If not found into cache, call URL
        Log.d(LOG_TAG, "We called codeForMultiCompany after click on Multicompany : savedDolBasedUrl="+this.savedDolBasedUrl+" urlToGo="+urlToGo);
        //myWebView = findViewById(R.id.webViewContent);

        // Clear also cache (we will need different content if we use different entities)
        Log.i(LOG_TAG, "Clear caches and history of webView");
        myWebView.clearCache(true);
        myWebView.clearHistory();
        this.cacheForMenu = null;
        this.cacheForQuickAccess = null;
        this.cacheForUploadFile = null;
        this.cacheForBookmarks = null;
        this.cacheForMultiCompany = null;
        //Log.d(LOG_TAG,"Clear also cookies");
        //this.myWebViewClientDoliDroid.deleteSessionCookies();

        myWebView.loadUrl(urlToGo);

        return true;
    }


    /**
     * Common code for CopyURL
     * codeForCopyUrl is in a UI thread
     *
     * @return  boolean     true
     */
    private boolean codeForCopyUrl()
    {
        String urlToGo;

        String currentUrl = myWebView.getUrl();

        // If not found into cache, call URL
        Log.d(LOG_TAG, "We called codeForCopyUrl after click on copyUrl : savedDolBasedUrl="+this.savedDolBasedUrl+" currentUrl="+currentUrl);
        //myWebView = findViewById(R.id.webViewContent);

        // Set currentUrl
        if (currentUrl != null) {
            currentUrl = currentUrl.replace("&dol_hide_topmenu=1", "");
            currentUrl = currentUrl.replace("dol_hide_topmenu=1&", "");
            currentUrl = currentUrl.replace("&dol_hide_leftmenu=1", "");
            currentUrl = currentUrl.replace("dol_hide_leftmenu=1&", "");
            currentUrl = currentUrl.replace("&dol_optimize_smallscreen=1", "");
            currentUrl = currentUrl.replace("dol_optimize_smallscreen=1&", "");
            currentUrl = currentUrl.replace("&dol_no_mouse_hover=1", "");
            currentUrl = currentUrl.replace("dol_no_mouse_hover=1&", "");
            currentUrl = currentUrl.replace("&dol_use_jmobile=1", "");
            currentUrl = currentUrl.replace("dol_use_jmobile=1&", "");
            // Another pass if it remains only 1 paramater
            currentUrl = currentUrl.replace("dol_hide_topmenu=1", "");
            currentUrl = currentUrl.replace("dol_hide_leftmenu=1", "");
            currentUrl = currentUrl.replace("dol_optimize_smallscreen=1", "");
            currentUrl = currentUrl.replace("dol_no_mouse_hover=1", "");
            currentUrl = currentUrl.replace("dol_use_jmobile=1", "");

            android.content.ClipboardManager clipboard = (android.content.ClipboardManager) getSystemService(Context.CLIPBOARD_SERVICE);
            android.content.ClipData clip = android.content.ClipData.newPlainText(getString(R.string.UrlCopied), currentUrl);
            clipboard.setPrimaryClip(clip);

            /*
            ClipboardManager clipboard = (ClipboardManager) getSystemService(Context.CLIPBOARD_SERVICE);
            ClipData clip = ClipData.newPlainText("DoliDroid copied URL", currentUrl);
            clipboard.setPrimaryClip(clip);
            */

            Toast.makeText(activity, R.string.UrlCopied, Toast.LENGTH_LONG).show();
        }
        
        return true;
    }

    /**
     * Common code for Back. Called for example by onOptionsItemSelected().
     * codeForBack is in a UI thread
     * 
     * @return  boolean             True
     */
    private boolean codeForBack() 
    {
        // Check if there is history
        String currentUrl;
        String previousUrl;
        boolean b;

        //myWebView = findViewById(R.id.webViewContent);

        currentUrl = myWebView.getUrl();
        previousUrl = "";
        b = myWebView.canGoBack();
        int indextoget = 0;

        mWebBackForwardList = myWebView.copyBackForwardList();
        if (mWebBackForwardList.getCurrentIndex() > 0) {
            indextoget = mWebBackForwardList.getCurrentIndex() - 1;
            previousUrl = mWebBackForwardList.getItemAtIndex(indextoget).getUrl();
        }

        Log.d(LOG_TAG, "We called codeForBack. canGoBack="+b+", indextoget="+indextoget+" currentUrl="+currentUrl+", previousUrl="+previousUrl);
        Log.d(LOG_TAG, "savedDolRootUrl="+savedDolRootUrl+", savedDolBasedUrl="+savedDolBasedUrl);

        if (indextoget <= 1 && ("".equals(previousUrl) || previousUrl.startsWith(savedDolRootUrl + "index.php"))) {
            // For an unknown reason, if we access home page passing by the login page, reaching the index.php page after, when we make a go page
            // back to reach this home page, we got an error of cache when making the goBack (even if we remove the myWebView.clearHistory after login).
            // So we disable the goBack for this case.
            Log.d(LOG_TAG, "We disable the goBack for this case. We replace it with the case there is no previous page.");
            if (previousUrl.isEmpty()) {
                // No previous page
                b = false;
            } else {
                // Previous page start with savedDolRootUrl + "index.php" so it's home page
                tagClearHistoryAfterFinished = 1;
                myWebView.loadUrl(savedDolRootUrl + "index.php");   // This will also add the index page into history
                return true;
            }
        }

        if (b) {
            if (currentUrl.contains("data:text/html")) {
                Log.d(LOG_TAG, "Previous Url may be a page with an history problem");
                if (altHistoryStack.size() > 0) {   // Should be true
                    nextAltHistoryStack = altHistoryStack.get(altHistoryStack.size() - 1);
                    Log.d(LOG_TAG, "Current page has &ui-page, we set nextAltHistoryStack to "+nextAltHistoryStack+" and consume the history stack");
                    altHistoryStack.remove(altHistoryStack.size() - 1);
                }
            } else {
                Log.d(LOG_TAG, "We clear nextAltHistoryStack"); 
                nextAltHistoryStack="";
            }

            Log.d(LOG_TAG, "We clear nextAltHistoryStackBis and make the goBack to "+previousUrl);
            nextAltHistoryStackBis = "";
            myWebView.goBack(); // This will call shouldInterceptRequest
        } else {
            if (! this.messageNoPreviousPageShown)
            {
                Toast.makeText(activity, getString(R.string.NoPreviousPageAgainToQuit), Toast.LENGTH_SHORT).show();
                this.messageNoPreviousPageShown = true;
            }
            else
            {
                Log.d(LOG_TAG, "Second click on Previous when no previous available.");
                Log.i(LOG_TAG, "We finish activity resultCode="+RESULT_SECONDACTIVITY);
                this.messageNoPreviousPageShown = false;
                setResult(RESULT_SECONDACTIVITY);   // We don't want to quit completely
                WebViewDatabase.getInstance(getBaseContext()).clearHttpAuthUsernamePassword();
                finish();
            }           
        }

        return true;
    }

    /**
     * Dump content of backforward webview list
     */
    public void dumpBackForwardList(WebView myWebView) 
    {
        // FOR DEBUG ONLY
        /*
        try {
            WebBackForwardList mWebBackForwardList = myWebView.copyBackForwardList();
            int nbelem=mWebBackForwardList.getSize();
            if (nbelem > 0)
            {
                for (int i=0; i < nbelem; i++)
                {
                    Log.v(LOG_TAG, "BackForward i="+i+(mWebBackForwardList.getCurrentIndex()==i?"*":"")+" url="+mWebBackForwardList.getItemAtIndex(i).getUrl());
                }
            }
            
            for (int i=0; i < altHistoryStack.size(); i++)
            {
                Log.v(LOG_TAG, "altHistoryStack i="+i+" "+altHistoryStack.get(i));
            }
            Log.v(LOG_TAG, "nextAltHistoryStack="+nextAltHistoryStack);
            Log.v(LOG_TAG, "nextAltHistoryStackBis="+nextAltHistoryStackBis);
        }
        catch(Exception e)
        {
            Log.e(LOG_TAG, "Error in dumpBackForwardList "+e.getMessage());
        }
        */
    }


    @Override
    public void onRequestPermissionsResult(int requestCode, String[] permissions, int[] grantResults) {
        Log.d(LOG_TAG, "onRequestPermissionsResult override requestCode="+requestCode+" grantResults length="+grantResults.length);

        switch (requestCode) {
            case REQUEST_CODE_ASK_PERMISSIONS_WRITE_EXTERNAL_STORAGE: {
                Log.d(LOG_TAG, Integer.toString(grantResults[0]));
                // If request is cancelled, the result arrays are empty.
                if (grantResults.length > 0 && grantResults[0] == PackageManager.PERMISSION_GRANTED) {
                    // permission was granted, yay! Do the task we need to do.
                    putDownloadInQueue(saveQueryForonRequestPermissionsResult, saveUrlForonRequestPermissionsResult, saveListOfCookiesForonRequestPermissionsResult);
                } else {
                    // permission denied, boo! Disable the
                    // functionality that depends on this permission.
                    Log.d(LOG_TAG, "Sorry, permission was not granted by user to do so.");

                    Toast.makeText(activity, "Sorry, permission to write files on storage was not granted by user.", Toast.LENGTH_SHORT).show();
                }
                return;
            }
            default:
                super.onRequestPermissionsResult(requestCode, permissions, grantResults);

                // other 'case' lines to check for other
                // permissions this app might request
        }
    }


    public boolean putDownloadInQueue(String query, String url, String listOfCookies)
    {
        Log.d(LOG_TAG, "putDownloadInQueue url to download = " + url);

        /*
        if (!url.startsWith("https")) {
            throw new IllegalArgumentException();
        }
        */

        // First create an object Request
        DownloadManager.Request request = new DownloadManager.Request(Uri.parse(url));

        String pathOfDownloadedFile = "download-dolidroid";
        if (query != null) {
            try {
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
                    pathOfDownloadedFile = URLDecoder.decode(query.replace("file=", ""), StandardCharsets.UTF_8);
                } else {
                    pathOfDownloadedFile = URLDecoder.decode(query.replace("file=", ""), "UTF-8");
                }
            } catch (UnsupportedEncodingException e) {
                pathOfDownloadedFile = "download-dolidroid";
            }
        }

        request.setTitle(pathOfDownloadedFile);
        request.setDescription(query);
        request.addRequestHeader("Cookie", listOfCookies);
        if (savedAuthuser != null) {
            Log.d(LOG_TAG, "putDownloadInQueue add header Authorization Basic for user "+savedAuthuser);
            request.addRequestHeader("Authorization", "Basic " + Base64.encodeToString((savedAuthuser+":"+savedAuthpass).getBytes(), Base64.NO_WRAP));   // Add user/pass for basic authentication
        }
        request.setAllowedNetworkTypes(DownloadManager.Request.NETWORK_WIFI | DownloadManager.Request.NETWORK_MOBILE);
        request.setAllowedOverRoaming(true);
        request.setVisibleInDownloadsUi(true);
        request.setNotificationVisibility(DownloadManager.Request.VISIBILITY_VISIBLE_NOTIFY_COMPLETED);
        request.allowScanningByMediaScanner();

        //File path = Environment.getExternalStorageDirectory();  // getExternalStorageDirectory+DIRECTORY_DOWNLOADS should be similar to dir of setDestinationInExternalPublicDir
        //String fullPath = path.getAbsolutePath(), "/"+Environment.DIRECTORY_DOWNLOADS;
        //String fullPath = Environment.getExternalStoragePublicDirectory(Environment.DIRECTORY_DOWNLOADS).getAbsolutePath();
        String fullPath = String.valueOf(getExternalFilesDir(Environment.DIRECTORY_DOWNLOADS));
        File tmpFolder = new File(fullPath); // Value is "/storage/emulated/0/Download"
        Log.d(LOG_TAG, tmpFolder.getAbsolutePath());

        if (!tmpFolder.exists()) {  // Should always exists because we use the default DIRECTORY_DOWNLOADS
            Log.d(LOG_TAG, "putDownloadInQueue Folder " + tmpFolder + " does not exists. We create it.");
            boolean resultmkdir = tmpFolder.mkdir();
            if (resultmkdir) {
                Log.d(LOG_TAG, "putDownloadInQueue Success to create dir");
            } else {
                Log.e(LOG_TAG, "putDownloadInQueue Failed to create dir");
            }
        }

        Log.d(LOG_TAG, "putDownloadInQueue Set output dirType=" + Environment.DIRECTORY_DOWNLOADS + ", subPath="+pathOfDownloadedFile+" (should be "+tmpFolder+")");
        // Storing in dedicated dir does not work
        // request.setDestinationInExternalFilesDir(getApplicationContext(), Environment.DIRECTORY_DOWNLOADS, pathOfDownloadeFile);
        // so we store file in common public download dir
        request.setDestinationInExternalPublicDir(Environment.DIRECTORY_DOWNLOADS, pathOfDownloadedFile);

        // Then create the object DownloadManager and enqueue the file
        // Complete tutorial on download manager on http://www.101apps.co.za/index.php/articles/using-the-downloadmanager-to-manage-your-downloads.html
        DownloadManager downloadManager = (DownloadManager) getSystemService(Context.DOWNLOAD_SERVICE);
        long downloadId = downloadManager.enqueue(request);
        Log.d(LOG_TAG, "putDownloadInQueue downloadId="+downloadId);


        // Once file is enqueue, you just have to wait until the event ACTION_DOWNLOAD_COMPLETE is triggered in DownloadBroadCasterReceiver.onReceive().


        // Now add also a timer to check regularly the status of download
        /*
        DownloadStatusChecker statusChecker = new DownloadStatusChecker(this);
        statusChecker.startMonitoringDownloads(downloadId, new DownloadStatusChecker.DownloadStatusListener() {
            @Override
            public void onDownloadStatusUpdated(int status) {
                if (status == DownloadManager.STATUS_SUCCESSFUL) {
                    // download is ok
                    Log.d(LOG_TAG, "DownloadStatusChecker downloadId "+downloadId+" Download STATUS_SUCCESSFUL");
                    statusChecker.stopMonitoringDownloads();    // destroy timer
                } else if (status == DownloadManager.STATUS_FAILED) {
                    // download failed
                    Log.d(LOG_TAG, "DownloadStatusChecker downloadId "+downloadId+" Download STATUS_FAILED");
                    statusChecker.stopMonitoringDownloads();    // destroy timer
                } else if (status == DownloadManager.STATUS_PAUSED) {
                    // download in pause
                    Log.d(LOG_TAG, "DownloadStatusChecker downloadId "+downloadId+" Download STATUS_PAUSED");
                } else {
                    Log.d(LOG_TAG, "DownloadStatusChecker downloadId "+downloadId+" Download manager status = "+status);
                }
            }
        }, 10000); // Check status every 10 seconds (10000 milliseconds)
        */

        return true;
    }



    /* ************************* */
    /* SUB-CLASSES WEBVIEW       */
    /* ************************* */ 
    
    /**
     * WebViewClientDoliDroid
     * 
     * Sequence of trigger called when we do myWebView.loadUrl(url):
     * 0) onPageStarted is called when we need to load a page (in cache or not)
     * 0) shouldInterceptRequest is called for all HTTP requests of pages and images (.php, .css, .js, .png, but not called when cache is used)
     * 1) shouldOverrideUrlLoading is called for HTTP requests of pages only (.php, .css, .js, but not called when cache is used). Also note that a redirect triggers this method but not if redirect is done with javascript window.location. 
     * 2) onLoadResource is called for all HTTP requests, even Ajax (but not called when cache is used)
     * 3) onReceivedError or onReceivedSslError
     * 4) onPageFinished is called when page with its resources are loaded (in cache or not)
     */
	class WebViewClientDoliDroid extends WebViewClient
	{
		int counthttpauth=0;

		final private SecondActivity secondActivity;
		String webViewtitle="";
		final String jsInjectCodeForLoginSubmit =
                "console.log('Execute jsInjectCodeForLoginSubmit');" +
		        "function dolidroidParseFormAfterSubmit(event) {" +
		        "    var form = this;" +
		        "    if (this.tagName.toLowerCase() != 'form') form = this.form;" +    
		        "    if (!form.method) form.method = 'get';" +
                "    var data = '';" +
		        "    data += 'method=' + form.method;" +
		        "    data += '&action=' + form.action;" +        
		        "    var inputs = document.forms[0].getElementsByTagName('input');" +
                "    for (var i = 0; i < inputs.length; i++) {" +
		        "         var field = inputs[i];" +
		        "         if (field.type != 'submit' && field.type != 'reset' && field.type != 'button')" +
		        "             data += '&' + field.name + '=' + field.value;" +
		        "    }" +
                "    console.log('Injected js code run: We have set a js variable data to '+data);" +
                "    console.log('Now we execute Java code functionJavaCalledByJsProcessFormSubmit(data)');" +
		        "    if (window.HTMLOUT) { window.HTMLOUT.functionJavaCalledByJsProcessFormSubmit(data); } else { console.error('HTMLOUT handler not valid'); } " +
                "    console.log('Injected js code finished');" +
		        "}" +
		        " " +
		        "for (var form_idx = 0; form_idx < document.forms.length; ++form_idx) {" +
		        "    document.forms[form_idx].addEventListener('submit', dolidroidParseFormAfterSubmit, false);" +
		        "}" +
		        "var inputs = document.getElementsByTagName('input'); " +
		        "for (var i = 0; i < inputs.length; i++) {" +
		        "    if (inputs[i].getAttribute('type') == 'button')" +
		        "        inputs[i].addEventListener('click', dolidroidParseFormAfterSubmit, false);" +
		        "}" +
                "console.log('End of jsInjectCodeForLoginSubmit');";

        public WebViewClientDoliDroid(SecondActivity secondActivity)
		{
			this.secondActivity = secondActivity;
		}
		
		/**
		 * onPageStarted
		 * 
		 * @param view		View
		 * @param url		URL
		 * @param favicon	Favicon
         * @see onPageFinished()
		 */
		@Override  
		public void onPageStarted(WebView view, String url, Bitmap favicon)
		{
		    try {
                Log.d(LOG_TAG, "onPageStarted url=" + url + " originalUrl=" + view.getOriginalUrl() + " view.getUrl=" + view.getUrl() + " savedDolBasedUrl=" + savedDolBasedUrl);
                String urltotest = view.getOriginalUrl();
                if (urltotest == null) {
                    urltotest = view.getUrl();
                }
                String urltotestWithoutBasicAuth = urltotest.replaceAll("://[^:]+:[^:]+@", "://");

                if (urltotest.startsWith("http:") && urltotestWithoutBasicAuth.startsWith(savedDolBasedUrl)) {
                    // If the urltotest (original url) is http:
                    //Log.d(LOG_TAG, "https:" + view.getUrl().substring(5));
                    //Log.d(LOG_TAG, url);
                    if (("https:" + urltotest.substring(5)).equals(url)) {
                        Log.w(LOG_TAG, "onPageStarted value of url is value of view.getUrl with a s added, we change the savedDolRootUrl");
                        //Toast.makeText(activity, "Warning: It seems your server forced a redirect to HTTPS page. Please check your connection URL and use the https directly if you can.", Toast.LENGTH_SHORT).show();
                        //savedDolBasedUrl = "http://"+savedDolBasedUrl.substring(4);

                        // Use Dialog instead of Toast for a longer message
                        AlertDialog alertDialog = new AlertDialog.Builder(activity).create();
                        alertDialog.setTitle(getString(R.string.Warning));
                        alertDialog.setMessage(getString(R.string.AlertHTTPS));
                        alertDialog.setButton(AlertDialog.BUTTON_NEUTRAL, "OK",
                                new DialogInterface.OnClickListener() {
                                    public void onClick(DialogInterface dialog, int which) {
                                        dialog.dismiss();
                                    }
                                });
                        alertDialog.show();
                    }

                    if (("http:" + urltotest.substring(5)).equals(url) && !this.secondActivity.httpWarningWasViewed) {
                        // Use Dialog instead of Toast for a longer message
                        AlertDialog alertDialog = new AlertDialog.Builder(activity).create();
                        alertDialog.setTitle(getString(R.string.Warning));
                        alertDialog.setMessage(getString(R.string.AlertHTTP));
                        alertDialog.setButton(AlertDialog.BUTTON_NEUTRAL, "OK",
                                new DialogInterface.OnClickListener() {
                                    public void onClick(DialogInterface dialog, int which) {
                                        dialog.dismiss();
                                    }
                                });
                        alertDialog.show();
                        this.secondActivity.httpWarningWasViewed = true;
                    }
                }
                //super.onPageStarted(view, url, favicon);
            } catch(Exception e) {
		        Log.e(LOG_TAG, e.getMessage());
            }
		}


		/**
		 * Return if we must intercept HTTP Request for pages (not called when cache is used)
		 * This method is called into a non-UI Thread (Android >= 3.0) so UI Thread function are not allowed.
		 * 
		 * @param 	WebView		        view
		 * @param 	WebResourceRequest	wrr     Object with	url. For example "http://192.168.0.1/xxx" or "data:image/png;base64,..."
		 * @return	boolean					    True or false if we must send request or not
		 */
		@Override
		public WebResourceResponse shouldInterceptRequest(WebView view, WebResourceRequest wrr)
		{
			// Get url relative to Dolibarr root.
            String url = null;
            if (wrr.getUrl() != null) {
                url = wrr.getUrl().toString();
            }

			String host=null;
			String fileName=null;
			String version=null;
			try {
				if (url != null && ! url.startsWith("data:"))   // for data:image
				{
					Uri uri=Uri.parse(url);
					version=uri.getQueryParameter("version");
					if (version != null)
					{
						String[] split = version.split("\\.");
						version=split[0]+"."+split[1];
					}
					fileName=uri.getPath();	// Return relative path of page (without params)
					host=uri.getHost();
				}
				
				// Format fileName to have a relative URL from root
				if (fileName != null) {
				    fileName=fileName.replaceFirst(this.secondActivity.savedDolRootUrlRel, "");
					if (fileName.startsWith("/")) {
					    fileName=fileName.substring(1);
                    }
				}
			}
			catch(Exception e)
			{
				Log.e(LOG_TAG, "shouldInterceptRequest Error into getting fileName or host");
				fileName=null;
			}

            Log.v(LOG_TAG, "shouldInterceptRequest url="+url+", host="+host+", fileName="+fileName+", savedDolBasedUrl="+savedDolBasedUrl+" version in url param (for js or css pages)="+version);

            Boolean isADownload = false;
            if ((url != null && (url.endsWith(".pdf") || url.endsWith(".odt") || url.endsWith(".ods")) && ! url.contains("action=") && ! url.contains("section=")) 	// Old way to detect a download (we do not make a download of link to delete or print or presend a file)
                    || "document.php".equals(fileName)									                       			                // The default wrapper to download files
                    || (url != null && url.contains("output=file"))) {																	// The new recommended parameter for pages that are not document.php like export.php that generate a file output
                isADownload = true;
            }

			if (isADownload && url != null && ! url.startsWith(savedDolBasedUrl) && url.startsWith(savedDolBasedUrlWithSForced)) {
				// In this case, we entered a HTTP login url but we were redirected to a HTTPS site.
			    Log.w(LOG_TAG, "shouldInterceptRequest AlertDownloadBadHTTPS Bad savedDolBasedUrl that does not allow download");
				// Can't make interaction here
				//Toast.makeText(activity, R.string.AlertDownloadBadHTTPS, Toast.LENGTH_LONG).show();
			}

			if (fileName != null && url.startsWith(savedDolBasedUrl)) {
				if (prefAlwaysUseLocalResources) {
					try {
						/* Example to return empty for files instead of some files */
						/*
						if (fileName.contains("dolibarr_logo.png"))	{
							Log.i(LOG_TAG, "Filename "+fileName+" discarded");
							return new WebResourceResponse("text/css", "UTF-8", new ByteArrayInputStream("".getBytes("UTF-8")));
						}*/

						String versionimg = VERSION_RESOURCES;                  // Set to the default value we want to use. Set "" to disable assets usage for img.
						//if (lastversionfoundforasset != null) versionimg = lastversionfoundforasset;
						String versionjscss = (version == null ? "" : version); // Set to "" to disable assets usage for js and css

						// Check if file need to be replaced by an asset file (if open file fails, throw exception and load from web).
						if ((fileName.endsWith("favicon.ico") || fileName.startsWith("theme/") || fileName.startsWith("includes/") || fileName.startsWith("public/demo/"))) {
							if (!versionimg.equals("") && (fileName.endsWith(".png") || fileName.endsWith(".jpg") || fileName.endsWith(".gif") || fileName.endsWith(".ico"))) {
								Log.d(LOG_TAG, "shouldInterceptRequest Filename " + fileName + " intercepted. Replaced with image assets file into " + versionimg);
								return new WebResourceResponse(null, null, getAssets().open(versionimg + "/" + fileName));
							} else if (!versionjscss.equals("") && fileName.endsWith(".js")) {
								Log.d(LOG_TAG, "shouldInterceptRequest Filename " + fileName + " intercepted. Replaced with js assets file into " + versionjscss);
								return new WebResourceResponse("application/x-javascript", "UTF-8", getAssets().open(versionjscss + "/" + fileName));
							} else if (!versionjscss.equals("") && fileName.endsWith(".css")) {
								Log.d(LOG_TAG, "shouldInterceptRequest Filename " + fileName + " intercepted. Replaced with css assets file into " + versionjscss);
								return new WebResourceResponse("text/css", "UTF-8", getAssets().open(versionjscss + "/" + fileName));
							}
						} else if (fileName.startsWith("data:text/html") || fileName.equals("")) {
							Log.d(LOG_TAG, "shouldInterceptRequest We make a back to go to a bad history url fileName=" + fileName);

							// Return last page that fails found into altHistoryStack
							if (altHistoryStack != null && altHistoryStack.size() > 0) {
								String lastelem = altHistoryStack.get(altHistoryStack.size() - 1);

								if ("menu".equals(lastelem) && cacheForMenu != null) {
									nextAltHistoryStack = "menu";
									altHistoryStack.remove(altHistoryStack.size() - 1);
									Log.d(LOG_TAG, "shouldInterceptRequest Return instead content of cacheForMenu");
									return new WebResourceResponse("text/html", "UTF-8", new ByteArrayInputStream(cacheForMenu.getBytes(StandardCharsets.UTF_8)));
								}
								if ("quickaccess".equals(lastelem) && cacheForQuickAccess != null) {
									nextAltHistoryStack = "quickaccess";
									altHistoryStack.remove(altHistoryStack.size() - 1);
									Log.d(LOG_TAG, "shouldInterceptRequest Return instead content of cacheQuickAccess");
									return new WebResourceResponse("text/html", "UTF-8", new ByteArrayInputStream(cacheForQuickAccess.getBytes(StandardCharsets.UTF_8)));
								}
							}

							Log.w(LOG_TAG, "shouldInterceptRequest Nothing to return instead");
						}
					} catch (IOException e) {
						Log.w(LOG_TAG, "shouldInterceptRequest Filename " + fileName + " intercepted but failed to find/open it from assets, we do standard process (so use cache of webview browser if not expired or download).");
					}
				} else {
					Log.v(LOG_TAG, "shouldInterceptRequest option is off");
				}
			}

            // Check if this is a download from a POST. Show a warning for that.
            if (isADownload) {
                // I comment this because we reach this page on both GET and POST links
                // In this case, we entered a HTTP login url but we were redirected to a HTTPS site.
                //Log.w(LOG_TAG, "shouldInterceptRequest AlertDownloadFromAPost Your app tried to make a download from a POST. This is forbidden by Dolibarr good practices. Not supported.");
                // Can't make interaction here
                //Toast.makeText(activity, R.string.AlertDownloadBadHTTPS, Toast.LENGTH_LONG).show();
            } else {
                Log.v(LOG_TAG, "shouldInterceptRequest "+url+" is not a download link");
            }

            return super.shouldInterceptRequest(view, wrr);
		}

		/**
		 * Handler to manage downloads.
		 * Seems called when we click on a href link or when calling loadUrl(). It is not called on POST methods (see shouldInterceptRequest, PageStarted or PageFinished for that).
         *
		 * @param	WebView		view		Web view
		 * @param	String		url			Url
		 * @return	boolean					True to mean URL has been handled by code, False to ask webview to handle it.
		 */
		@Override
        public boolean shouldOverrideUrlLoading(WebView view, WebResourceRequest request) {
            String url = null;
            if (request.getUrl() != null) {
                url = request.getUrl().toString();
            }

            Log.d(LOG_TAG, "shouldOverrideUrlLoading url=" + url + " originalUrl=" + view.getOriginalUrl());
            Log.d(LOG_TAG, "shouldOverrideUrlLoading savedDolRootUrl=" + savedDolRootUrl + " savedDolRootUrlWithSForced = " + savedDolRootUrlWithSForced + " savedDolBasedUrl=" + savedDolBasedUrl);

            //if (url != null) {
            //    String urlWithoutBasicAuth = url.replaceAll("://[^:]+:[^:]+@", "://");
            //    Log.d(LOG_TAG, "shouldOverrideUrlLoading urlWithoutBasicAuth=" + urlWithoutBasicAuth);
            //}

            // TODO Optimize performance by disabling loading of some url (ie: some jquery plugins)

            if (url.startsWith("tel:")) {  // Intercept phone urls
                Log.d(LOG_TAG, "Launch dialer : " + url);
                Intent intent = new Intent(Intent.ACTION_DIAL, Uri.parse(url));
                try {
                    startActivity(intent);
                } catch(ActivityNotFoundException e) {
                    Log.e(LOG_TAG, "No activity to manage Intent ACTION_DIAL");
                }
                return true;
            } else if (url.startsWith("geo:")) {  // Intercept geoloc url (map)
                Log.d(LOG_TAG, "Launch geo : " + url);
                Intent intent = new Intent(Intent.ACTION_VIEW, Uri.parse(url));
                startActivity(intent);
                return true;
            } else if (url.startsWith("mailto:")) {  // Intercept mailto urls
                Log.d(LOG_TAG, "Launch mailto : " + url);
                try {
                    String theEmail = "", theEmailCC = "", theEmailBCC = "", theSubject = "", theBody = "";
                    Boolean hasEmail = true, hasEmailCC = url.contains("&cc="), hasEmailBCC = url.contains("&bcc="), hasSubject = url.contains("&subject="), hasBody = url.contains("&body=");
                    int posEmail = 0, posEmailCC = hasEmailCC ? url.indexOf("&cc=") : 0, posEmailBCC = hasEmailBCC ? url.indexOf("&bcc=") : 0, posSubject = hasSubject ? url.indexOf("&subject=") : 0, posBody = hasBody ?  url.indexOf("&body=") : 0;

                    if (hasEmail && hasEmailCC) {
                        theEmail = url.substring(posEmail, posEmailCC - posEmail);
                    } else if (hasEmail && hasEmailBCC) {
                        theEmail = url.substring(posEmail, posEmailBCC - posEmail);
                    } else if (hasEmail && hasSubject) {
                        theEmail = url.substring(posEmail, posSubject - posEmail);
                    } else if (hasEmail && hasBody) {
                        theEmail = url.substring(posEmail, posBody - posEmail);
                    } else if (hasEmail) {
                        theEmail = url;
                    } else {
                        /*theEmail    = url;*/
                    }

                    if (hasEmailCC && hasEmailBCC) {
                        theEmailCC = url.substring(posEmailCC, posEmailBCC - posEmailCC);
                    } else if (hasEmailCC && hasSubject) {
                        theEmailCC = url.substring(posEmailCC, posSubject - posEmailCC);
                    } else if (hasEmailCC && hasBody) {
                        theEmailCC = url.substring(posEmailCC, posBody - posEmailCC);
                    } else if (hasEmailCC) {
                        theEmailCC = url.substring(posEmailCC);
                    } else {
                        /*theEmailCC  = url.substring(posEmailCC);*/
                    }
                    theEmailCC = theEmailCC.replace("&cc=", "");

                    if (hasEmailBCC && hasSubject) {
                        theEmailBCC = url.substring(posEmailBCC, posSubject - posEmailBCC);
                    } else if (hasEmailBCC && hasBody) {
                        theEmailBCC = url.substring(posEmailBCC, posBody - posEmailBCC);
                    } else if (hasEmailBCC) {
                        theEmailBCC = url.substring(posEmailBCC);
                    } else {
                        /*theEmailBCC = url.substring(posEmailBCC);*/
                    }
                    theEmailBCC = theEmailBCC.replace("&bcc=", "");

                    if (hasSubject && hasBody) {
                        theSubject = url.substring(posSubject, posBody - posSubject);
                    } else if (hasSubject) {
                        theSubject = url.substring(posSubject);
                    } else {
                        /*theSubject  = url.substring(posSubject);*/
                    }
                    theSubject = theSubject.replace("&subject=", "");

                    if (hasBody) {
                        theBody = url.substring(posBody);
                    } else {
                        /*theBody     = url.substring(posBody);*/
                    }
                    theBody = theBody.replace("&body=", "");
                    theSubject = theSubject.replace("%20", " ");
                    theBody = theBody.replace("%20", " ").replace("%0A", "\n");
                    String recipient = url.substring(url.indexOf(":") + 1);

                    Intent emailIntent = new Intent(Intent.ACTION_SEND, Uri.parse(url));
                    emailIntent.setDataAndType(Uri.parse(url), "message/rfc822");
                    emailIntent.putExtra(android.content.Intent.EXTRA_EMAIL, new String[]{recipient});
                    if (hasEmailCC) { emailIntent.putExtra(android.content.Intent.EXTRA_CC, theEmailCC); }
                    if (hasEmailBCC) { emailIntent.putExtra(android.content.Intent.EXTRA_BCC, theEmailBCC); }
                    if (hasSubject) { emailIntent.putExtra(android.content.Intent.EXTRA_SUBJECT, theSubject); }
                    if (hasBody) { emailIntent.putExtra(android.content.Intent.EXTRA_TEXT, theBody); }
                    startActivity(Intent.createChooser(emailIntent, "Email..."));
                } catch (Exception ex) {
                    // Nothing done
                }
                return true;
            } else if (! url.startsWith(savedDolBasedUrl) && ! url.startsWith(savedDolBasedUrlWithSForced)
                    && ! url.startsWith(savedDolBasedUrlWithoutUserInfo) && ! url.startsWith(savedDolBasedUrlWithoutUserInfoWithSForced)
                    //&& ! url.startsWith("https://accounts.google.com/o/oauth2/")
                    //&& ! url.startsWith("https://accounts.google.com/signin/")
            )
            {	// This is an external url
                Log.d(LOG_TAG, "Launch external url in the default browser : " + url);
                try {
                    Intent intent = new Intent(Intent.ACTION_VIEW, Uri.parse(url));
                    startActivity(intent);
                } catch (Exception ex) {
                    // Nothing done
                }
                return true;
            }

			// Without this, we got "download not supported" ("telechargement non pris en charge")
			if (((url.endsWith(".pdf") || url.endsWith(".odt") || url.endsWith(".ods")) && ! url.contains("action=")) 	// Old way to detect a download (we do not make a download of link to delete or print or presend a file)
					|| url.startsWith(savedDolRootUrl+"document.php?")									    			// The default wrapper to download files
                    || url.startsWith(savedDolRootUrlWithSForced+"document.php?")										// The default wrapper to download files
					|| url.contains("output=file"))																		// The new recommended parameter for pages that are not documents.php like export.php, emptyexample.php that generate a file output
	        {
				String query=Uri.parse(url).getQuery();
				if (query != null) {
                    // Example:
                    // query=file=myfilename.ext&...&output=file
                    // query=format=ical&...&output=file
                    query = query.replaceAll("^.*file=", "file=").replaceAll("&.*$", "");
                    query = query.replaceAll("^file=.*/", "file=");
                    if (!query.startsWith("file=")) {
                        // The parameter file=myfilename.ext was not provided so we use a generic file name
                        query = "Unknown-filename";
                    } else {
                        query = query.replaceAll("^file=", "");
                    }
                }
				Log.d(LOG_TAG, "shouldOverrideUrlLoading Start activity to download file="+query);
				/*Pattern p = Pattern.compile("file=(.*)\.(pdf|odt|ods)");
				Matcher m = p.matcher(urlquery);
				if (m.find()) {
				  url = m.group(1);  // The matched substring
				}*/

				if (! url.contains("dol_hide_topmenu=")) {
                    url = url + (url.contains("?")?"&":"?") + "dol_hide_topmenu=1";
                }
    			if (! url.contains("dol_hide_leftmenu=")) {
                    url = url + (url.contains("?")?"&":"?") + "dol_hide_leftmenu=1";
                }
    			if (! url.contains("dol_optimize_smallscreen=")) {
                    url = url + (url.contains("?")?"&":"?") + "dol_optimize_smallscreen=1";
                }
    			if (! url.contains("dol_no_mouse_hover=")) {
                    url = url + (url.contains("?")?"&":"?") + "dol_no_mouse_hover=1";
                }
    			if (! url.contains("dol_use_jmobile=")) {
                    url = url + (url.contains("?")?"&":"?") + "dol_use_jmobile=0";
                }
    			
				String listOfCookies=this.listCookies();

				// This call url and save content into a file
				try {
					saveQueryForonRequestPermissionsResult = query;
					saveUrlForonRequestPermissionsResult = url;
					saveListOfCookiesForonRequestPermissionsResult = listOfCookies;

					Log.d(LOG_TAG, "shouldOverrideUrlLoading ask permission to save file");

                    int version = Build.VERSION.SDK_INT;
                    if (version <= 32) {
                        // If API 23 to 32, we ask permission to user
                        int hasWriteContactsPermission = checkSelfPermission(Manifest.permission.WRITE_EXTERNAL_STORAGE);
                        if (hasWriteContactsPermission != PackageManager.PERMISSION_GRANTED) {
                            //boolean needEducativeInfo = shouldShowRequestPermissionRationale(Manifest.permission.WRITE_EXTERNAL_STORAGE);
                            //Log.d(LOG_TAG, "shouldOverrideUrlLoading hasWriteContactsPermission = " + hasWriteContactsPermission + " needEducativeInfo = " + needEducativeInfo);

                            // Now request the permission
                            requestPermissions(new String[]{Manifest.permission.WRITE_EXTERNAL_STORAGE}, REQUEST_CODE_ASK_PERMISSIONS_WRITE_EXTERNAL_STORAGE);

                            Log.d(LOG_TAG, "shouldOverrideUrlLoading requestPermissions was called, so we exit. The run of putDownloadInQueue will be done by result handler onRequestPermissionsResult()");
                            return false;
                        }
                    } else {
                        // If API 33, asking WRITE_EXTERNAL_STORAGE is useless. It is never allowed to write outside of app.
                        boolean isAllowPermissionAllFiles = Environment.isExternalStorageManager();    // Return if app has "All Files Access"
                        Log.i(LOG_TAG,"shouldOverrideUrlLoading isExternalStorageManager() " + isAllowPermissionAllFiles);

                        // We probably got "false" on isAllowPermissionAllFiles, but never mind, we should not need All Files Access,
                        // so we continue to try the putDownloadInQueue, we should be allowed on Download
                    }

					Log.d(LOG_TAG, "shouldOverrideUrlLoading Cal putDownloadInQueue for url to download = " + url);

					putDownloadInQueue(query, url, listOfCookies);

                    /*
                    DownloadManager.Request request = new DownloadManager.Request(Uri.parse(url));
                    request.setDescription(query);
                    request.setTitle(query);
                    request.addRequestHeader("Cookie", listOfCookies);
                    if (savedAuthuser != null) request.addRequestHeader("Authorization", "Basic " + Base64.encodeToString((savedAuthuser+":"+savedAuthpass).getBytes(), Base64.NO_WRAP));   // Add user/pass for basic authentication
                    //request.setVisibleInDownloadsUi(isVisible);
                    request.setAllowedNetworkTypes(DownloadManager.Request.NETWORK_WIFI | DownloadManager.Request.NETWORK_MOBILE);
                    request.setAllowedOverRoaming(true);
                    request.setVisibleInDownloadsUi(true);
                    request.allowScanningByMediaScanner();
                    request.setNotificationVisibility(DownloadManager.Request.VISIBILITY_VISIBLE);

                    Log.d(LOG_TAG, "shouldOverrideUrlLoading Set output dirType=" + Environment.DIRECTORY_DOWNLOADS + " subPath="+query);
                    request.setDestinationInExternalPublicDir(Environment.DIRECTORY_DOWNLOADS, query);
                    //request.setDestinationInExternalFilesDir(getApplicationContext(), null, query);
                    
                    // Get download service and enqueue file
                    // Complete tutorial on download manager on http://www.101apps.co.za/index.php/articles/using-the-downloadmanager-to-manage-your-downloads.html
                    DownloadManager dmanager = (DownloadManager) getSystemService(Context.DOWNLOAD_SERVICE);
                    long id = dmanager.enqueue(request);
                    */

					Log.d(LOG_TAG, "shouldOverrideUrlLoading URI has been added in queue - Now waiting event onReceive ACTION_DOWNLOAD_COMPLETE");
				}
				catch(IllegalArgumentException ie)
				{
					Log.e(LOG_TAG, Objects.requireNonNull(ie.getMessage(), "IllegalArgumentException message=null"));
					Toast.makeText(activity, ie.getMessage(), Toast.LENGTH_LONG).show();
				}
				catch(Exception e)
				{
					Log.e(LOG_TAG, Objects.requireNonNull(e.getMessage(), "Exception message=null"));
					Toast.makeText(activity, e.getMessage(), Toast.LENGTH_LONG).show();
				}

				return true;
	        } else {
			    Log.v(LOG_TAG, "Not a special link, not a download link");
            }

			return false;
		}

        @JavascriptInterface
        public void onAuthenticationComplete() {
            // Called when authentication is complete
            // Add code to go back to the Android application
            Log.v(LOG_TAG, "onAuthenticationComplete !!!!!!!!!!!!!!");
        }

		/**
		 * Called when a HTTP request is done (not called when cache is used)
		 * 
		 * @param	WebView		view		Web view
		 * @param	String		url			Url
		 */
		@Override
		public void onLoadResource(WebView view, String url)
		{
			Log.v(LOG_TAG, "onLoadResource url="+url+" originalUrl="+view.getOriginalUrl());
			//super.onLoadResource(view, url);
		}

		/**
		 * onPageFinished: Execute code just after a page was completely loaded
         *
         * @param	WebView		view		Web view
         * @param	String		url			Url
         * @see onPageStarted()
		 */
		@Override
	    public void onPageFinished(WebView view, String url)
	    {
			// super.onPageFinished(view, url);
			
			if (listOfCookiesAfterLogon == null)
			{
				Log.d(LOG_TAG, "onPageFinished Save session cookies for the download manager into var listOfCookiesAfterLogon");
				listOfCookiesAfterLogon=this.listCookies();	// Save cookie for
			}

            // If the swipe animation is still running
            swipe.setRefreshing(false);

			//myWebView = findViewById(R.id.webViewContent);
			boolean b = myWebView.canGoBack();
            WebBackForwardList mWebBackForwardList;

            Log.d(LOG_TAG, "onPageFinished Begin url="+url+" canGoBack="+b);
	        
		    if (tagToShowInterruptMessage.length() > 0 && tagToShowInterruptCounter > 0)	//onConsoleMessage is increased by onConsoleMessage function (javascript error)
			{
		    	// We should not go here. This manage loading errors when JMobile was not implemented correctly (no data-role defined into page). 
		    	String myErrorNum="JSERR001";
		    	String myMessage0="Error "+myErrorNum;
		    	String myMessage1="An error loading page was detected into some javascript part of your page.";
		    	String myMessage1b="This is commonly a problem on your server side, not into DoliDroid.";
		    	String myMessage2="DoliDroid will not work corrrectly until you solve this.";
		    	String myMessage3="-If you have installed external modules, try after disabling them (some external modules introduce bugs)";
		    	String myMessage4="-Try also to call login page from a standard PC by adding &dol_use_jmobile=1 as parameter of URL and check you don't have any javascript errors.";
		    	//String myMessage5="Go on <a href=\"http://wiki.dolibarr.org/index.php/Application_Android_-_DoliDroid\">here for online help to solve this</a>.";
		    	String myMessage5="For online help to solve this, go on page:<br><br>\n<strong><font size=\"-1\">http://wiki.dolibarr.org/index.php/Application_Android_-_DoliDroid#"+myErrorNum+"</font></strong>";
		    	String myMessage6="Note: Page you want to load is:<br><br>\n<font size=\"-1\">"+url+"</font>";
				Log.d(LOG_TAG, "Show user message "+myMessage0+"\n"+myMessage1+"\n"+myMessage1b+"\n"+myMessage2+"\n"+myMessage3+"\n"+myMessage4);

				tagToShowInterruptMessage="";
				
				myWebView.loadData(myMessage0+"<br><br>\n"+myMessage1+"<br>\n"+myMessage1b+"<br>\n"+myMessage2+"<br>\n<br>\n"+myMessage5+"<br><br><br>\n"+myMessage6, "text/html", "UTF-8");
			}
		    else if (tagToShowMessage.length() > 0 && tagToShowCounter > 0)	// tagToShowCounter is increased by onConsoleMessage function (javascript error)
			{
				Toast.makeText(activity, tagToShowMessage, Toast.LENGTH_LONG).show();

				tagToShowMessage="";
			}
			else
			{
				tagToShowInterruptCounter=0;		// Page was loaded, so we set count of number of try to 0

                // If we loaded page login.php, we check Dolibarr version
				// Title for login page is defined into login.tpl.php (with some part into dol_loginfunction in security2.lib.php)
				this.webViewtitle = myWebView.getTitle();

				if (this.webViewtitle != null)
				{
                    Matcher m = patternLoginHomePageForVersion.matcher(this.webViewtitle);
                    boolean foundVersion = m.find();

				    if (savMenu != null) {
                        MenuItem menuItemBookmarks = savMenu.findItem(R.id.menu_bookmarks);
                        MenuItem menuItemUserCard = savMenu.findItem(R.id.menu_usercard);
                        MenuItem menuItemVirtualCard = savMenu.findItem(R.id.menu_virtualcard);
                        MenuItem menuItemUploadFile = savMenu.findItem(R.id.menu_uploadfile);

                        if (foundVersion)    // if title ends with " Dolibarr x.y.z" or " Dolibarr x.y.z - multicompany or anytext from module hook setTitleHtml", this is login page or home page
                        {
                            lastversionfound = m.group(1) + ", " + m.group(2) + ", " + m.group(3);
                            lastversionfoundforasset = m.group(1) + "." + m.group(2);
                            Log.i(LOG_TAG, "onPageFinished Page title=" + this.webViewtitle + " - url=" + url + " - Found login or home page + version: " + lastversionfound + " - Suggest to use asset: " + lastversionfoundforasset);

                            MenuItem menuItemMultiCompany = savMenu.findItem(R.id.menu_multicompany);
                            if (menuItemMultiCompany != null) {
                                // Enable or disable menu entry for Multicompany
                                Matcher multicompanyRegex = patternLoginHomePageForMulticompany.matcher(this.webViewtitle);

                                if (multicompanyRegex.find() && m.group(1) != null && Integer.parseInt(m.group(1)) >= 15) {
                                    menuItemMultiCompany.setVisible(true);
                                    Log.d(LOG_TAG, "onPageFinished Module multicompany was found and Version major found >= 15, we enable the multicompany menu entry");
                                    isMulticompanyOn = true;
                                } else {
                                    menuItemMultiCompany.setVisible(false);
                                    Log.d(LOG_TAG, "onPageFinished Module multicompany was NOT found or major version < 15, we disable the multicompany menu entry");
                                    isMulticompanyOn = false;
                                }
                            } else {
                                isMulticompanyOn = false;
                            }

                            // Enable or disable menu entry for Upload File (available from Dolibarr v21)
                            try {
                                if (m.group(1) != null && Integer.parseInt(m.group(1)) >= 21 && !isInstalledFromPlayStore) {
                                    Log.d(LOG_TAG, "onPageFinished Version major found >= 21 and not installed from PlayStore , we enable the upload file entry");
                                    menuItemUploadFile.setVisible(true);
                                    isUploadFileOn = true;
                                } else {
                                    Log.d(LOG_TAG, "onPageFinished Version major found < 21 or installed from PlayStore, we disable the upload file entry");
                                    menuItemUploadFile.setVisible(false);
                                    isUploadFileOn = false;
                                }
                            } catch (Exception e) {
                                Log.d(LOG_TAG, "onPageFinished Failed to parse version found = " + m.group(1));
                                menuItemUploadFile.setVisible(false);
                            }

                            // Enable or disable menu entry for Bookmarks (available from Dolibarr v15)
                            try {
                                if (m.group(1) != null && Integer.parseInt(m.group(1)) >= 15) {
                                    Log.d(LOG_TAG, "onPageFinished Version major found >= 15, we enable the bookmark menu entry");
                                    menuItemBookmarks.setVisible(true);
                                    isBookmarkOn = true;
                                } else {
                                    Log.d(LOG_TAG, "onPageFinished Version major found < 15, we disable the bookmark menu entry");
                                    menuItemBookmarks.setVisible(false);
                                    isBookmarkOn = false;
                                }
                            } catch (Exception e) {
                                Log.d(LOG_TAG, "onPageFinished Failed to parse version found = " + m.group(1));
                                menuItemBookmarks.setVisible(false);
                            }

                            // Enable or disable menu entry for Virtual card (available from Dolibarr v18)
                            try {
                                if (m.group(1) != null && Integer.parseInt(m.group(1)) >= 18) {
                                    Log.d(LOG_TAG, "onPageFinished Version major found >= 18, we enable the virtual card menu entry");
                                    menuItemUserCard.setVisible(true);
                                    menuItemVirtualCard.setVisible(true);
                                    isUserCardOn = true;
                                    isVirtualCardOn = true;
                                } else {
                                    Log.d(LOG_TAG, "onPageFinished Version major found < 18, we disable the virtual card menu entry");
                                    menuItemUserCard.setVisible(false);
                                    menuItemVirtualCard.setVisible(false);
                                    isUserCardOn = false;
                                    isVirtualCardOn = false;
                                }
                            } catch (Exception e) {
                                Log.d(LOG_TAG, "onPageFinished Failed to parse version found = " + m.group(1));
                                menuItemBookmarks.setVisible(false);
                            }
                        }/* else {
                            Log.d(LOG_TAG, "Failed to find version");
                            menuItemBookmarks.setVisible(false);
                        }*/
                    }

                    // If title match pattern, this is a special page (login page or home page)
                    // patternLoginPage = Login Doli[a-zA-Z]+ (\d+)\.(\d+)\.([^\s]+)    for Dolibarr <= 3.6
                    // @ (?:Doli[a-zA-Z]+ |)(\d+)\.(\d+)\.([^\s]+)                      for Dolibarr > 3.7
                    if (patternLoginPage.matcher(this.webViewtitle).find() || patternLoginPage2.matcher(this.webViewtitle).find()) {
                        // This is login page, so we will read file "secret_shared_prefs" and inject result.
				    	if (url.equals(savedDolRootUrl)) {
							Log.w(LOG_TAG, "onPageFinished We ignore page since url is not a special page (not /index.php, not /mypage.php, ...)");
				    	} else {
							synchronized (this) 
							{
								boolean versionOk = true;	// Will be false if Dolibarr is < 6.0.*
								if (foundVersion) {
									try {
										if (m.group(1) != null && Integer.parseInt(m.group(1)) < 6) versionOk = false;
										if (m.group(1) != null && Integer.parseInt(m.group(1)) < 6 && m.group(2) != null && Integer.parseInt(m.group(2)) < 0)
											versionOk = false;
									}
									catch(Exception e)
									{
										versionOk = false;
									}
								}
								else {
									versionOk = false;
								}
								if (versionOk) {
								    Log.d(LOG_TAG, "onPageFinished Dolidroid is compatible with your Dolibarr "+lastversionfound);
                                } else {
									Log.w(LOG_TAG, "onPageFinished Dolidroid is NOT compatible with your Dolibarr "+lastversionfound);
									final Toast aToast = Toast.makeText(activity, getString(R.string.notCompatibleWithVersion, (lastversionfound == null ? this.webViewtitle : lastversionfound), "3.4"), Toast.LENGTH_SHORT);
									new CountDownTimer(5000, 1000)	// 5 seconds
									{
									    public void onTick(long millisUntilFinished) {aToast.show();}
									    public void onFinish() {aToast.show();}
									}.start();								
								}
	
								String jsInjectCodeForSetForm="";
								if (tagToOverwriteLoginPass)	// If we are allowed to overwrite username/pass into fields
								{
								    // Load username and password for the URL
							    	try {
                                        Log.d(LOG_TAG, "onPageFinished Open file to read shared preferences (secret_shared_prefs)");

                                        String masterKeyAlias = MasterKeys.getOrCreate(MasterKeys.AES256_GCM_SPEC);
                                        SharedPreferences sharedPrefsEncrypted = EncryptedSharedPreferences.create(
                                                "secret_shared_prefs",
                                                masterKeyAlias,
                                                getApplicationContext(),
                                                EncryptedSharedPreferences.PrefKeyEncryptionScheme.AES256_SIV,
                                                EncryptedSharedPreferences.PrefValueEncryptionScheme.AES256_GCM
                                        );

                                        String username = sharedPrefsEncrypted.getString(savedDolRootUrl + "-username", "");
                                        String password = sharedPrefsEncrypted.getString(savedDolRootUrl + "-password", "");

                                        if ((username != null && !"".equals(username)) || (password != null && !password.isEmpty())) {
                                            tagToOverwriteLoginPass = false;  // So we autofill form only the first time.
                                            Log.d(LOG_TAG, "onPageFinished Prepare js to autofill login form with username=" + username + " password=" + password.replaceAll(".", "*"));
                                            //Log.d(LOG_TAG, "onPageFinished Prepare js to autofill login form with username="+username+" password="+password);

                                            // This call inject JavaScript into the page which just finished loading.
                                            if (username != null && !"".equals(username))
                                                jsInjectCodeForSetForm += "document.getElementById('username').value='" + username + "';";    // Warning: This line makes Webkit fails with 2.3
                                            if (password != null && !"".equals(password))
                                                jsInjectCodeForSetForm += "document.getElementById('password').value='" + password + "';";    // Warning: This line makes Webkit fails with 2.3
                                        } else {
                                            Log.d(LOG_TAG, "onPageFinished No predefined login/pass to autofill login form");
                                        }
                                    }
                                    catch(IOException e) {
                                        Log.w(LOG_TAG, "onPageFinished Error IO on reading saved username/password");
                                    }
							    	catch(Exception e) {
                                        Log.w(LOG_TAG, "onPageFinished Error on reading saved username/password");
                                    }
								} else {
								    Log.d(LOG_TAG, "onPageFinished Do not autofill login form with login/pass. tagToOverwriteLoginPass is false.");
                                }

								// Force inject value of mobile parameters. This is required when session expired and login is available
								jsInjectCodeForSetForm+="document.getElementById('dol_hide_topmenu').value='1';";	// Warning: This line makes Webkit fails with 2.3
								jsInjectCodeForSetForm+="document.getElementById('dol_hide_leftmenu').value='1';";	// Warning: This line makes Webkit fails with 2.3
								jsInjectCodeForSetForm+="document.getElementById('dol_optimize_smallscreen').value='1';";	// Warning: This line makes Webkit fails with 2.3
								jsInjectCodeForSetForm+="document.getElementById('dol_no_mouse_hover').value='1';";	// Warning: This line makes Webkit fails with 2.3
								jsInjectCodeForSetForm+="document.getElementById('dol_use_jmobile').value='1';";	// Warning: This line makes Webkit fails with 2.3
								jsInjectCodeForSetForm+=jsInjectCodeForLoginSubmit;
								// Now inject js to catch submission of login
								Log.d(LOG_TAG, "onPageFinished Inject javascript jsInjectCodeForSetForm into page (to autofill form if allowed and to hook the submit of form to catch submitted params)");
								view.loadUrl("javascript:(function() { " + jsInjectCodeForSetForm + " })()");
							}
				    	}

                        myWebView.clearHistory();   // So it removes all history and the new loadUrl will be alone and first in list
                    } else { // This is not login page
                        Log.d(LOG_TAG, "onPageFinished tagLastLoginPassToSavedLoginPass="+tagLastLoginPassToSavedLoginPass);

				    	//Log.d(LOG_TAG, "Title of page is: "+myWebView.getTitle()+" - Login tag or Version not found");
				    	if (tagLastLoginPassToSavedLoginPass) {
				    		Log.i(LOG_TAG, "onPageFinished We have just received a page that is not Login page after submitting login form.");
				    		tagLastLoginPassToSavedLoginPass=false;

					    	//SharedPreferences sharedPrefs = PreferenceManager.getDefaultSharedPreferences(getApplicationContext());
                            SharedPreferences sharedPrefs = getApplicationContext().getSharedPreferences("shared_prefs", Context.MODE_PRIVATE);
					    	// shared pref file is into /data/data/package.name/shared_prefs/settings.xml but can be read by root only.

					    	boolean prefAlwaysAutoFill = sharedPrefs.getBoolean("prefAlwaysAutoFill", true);
					    	if (prefAlwaysAutoFill) {
						    	Log.d(LOG_TAG, "onPageFinished We save some fields of the submitted form (prefAlwaysAutoFill is true) into a file (secret_shared_prefs).");

						    	// Retrieve last values used submitted for username and password
                                // to save them with a name depending on URL.
                                try {
                                    String masterKeyAlias = MasterKeys.getOrCreate(MasterKeys.AES256_GCM_SPEC);
                                    SharedPreferences sharedPrefsEncrypted = EncryptedSharedPreferences.create(
                                            "secret_shared_prefs",
                                            masterKeyAlias,
                                            getApplicationContext(),
                                            EncryptedSharedPreferences.PrefKeyEncryptionScheme.AES256_SIV,
                                            EncryptedSharedPreferences.PrefValueEncryptionScheme.AES256_GCM
                                    );

                                    String username = sharedPrefsEncrypted.getString("lastsubmit-username", "");
                                    String password = sharedPrefsEncrypted.getString("lastsubmit-password", "");
                                    if ((username != null && !"".equals(username)) || (password != null && !"".equals(password))) {
                                        // Save username and password.
                                        SharedPreferences.Editor editor = sharedPrefsEncrypted.edit();
                                        Log.d(LOG_TAG, "onPageFinished Save " + savedDolRootUrl + "-username=" + username);
                                        editor.putString(savedDolRootUrl + "-username", username);
                                        Log.d(LOG_TAG, "onPageFinished Save " + savedDolRootUrl + "-password=" + password);
                                        editor.putString(savedDolRootUrl + "-password", password);
                                        editor.apply();
                                    }
                                }
                                catch(Exception e) {
                                    Log.w(LOG_TAG, "onPageFinished Failed to read or write into EncryptedSharedPreferences.");
                                    Log.w(LOG_TAG, e.getMessage());
                                }
					    	} else {
                                Log.d(LOG_TAG, "onPageFinished We don't save form fields (prefAlwaysAutoFill is false).");
                            }
                            tagToOverwriteLoginPass = prefAlwaysAutoFill;

				    		// Clear webview history
							Log.d(LOG_TAG,"onPageFinished We clear history to removes the login page history entry");

                            myWebView.clearHistory();   // So it removes the login page history entry (we don't want to have it when making go back)
				    	} else {
                            // This is a common page (no tag on login or version and not a page just after a login)
                            WebBackForwardList tmpWebBackForwardList = myWebView.copyBackForwardList();
                            //int currentindexinhistory = tmpWebBackForwardList.getCurrentIndex();
                            if (tagClearHistoryAfterFinished > 0) {
                                tagClearHistoryAfterFinished = 0;
                                myWebView.clearHistory();   // So it removes the login page history entry (we don't want to have it when making go back)
                            }
                        }
				    }
				}

				// If we loaded the page logout.php, we finished activity
				if (url.contains("logout.php")) {
					synchronized (this) 
					{
						if (tagToLogout)
						{
							Log.d(LOG_TAG, "onPageFinished End of logout page, tagToLogout="+tagToLogout);
							tagToLogout = false;	// Set to false to avoid infinite loop
							tagToOverwriteLoginPass = true;
							Log.i(LOG_TAG, "onPageFinished We finish activity resultCode="+RESULT_SECONDACTIVITY);
							setResult(RESULT_SECONDACTIVITY);
					    	WebViewDatabase.getInstance(getBaseContext()).clearHttpAuthUsernamePassword();
							finish();	// End activity
						}
					}
				}

				// If we loaded page get_menudiv.php, we trigger code to save content into cache
				/*
				if (url.contains("get_menudiv.php")) 
				{
					synchronized (this) 
					{
						if (tagToGetCacheForMenu)
						{
							Log.d(LOG_TAG, "Inject js to get html content, tagToGetCacheMenu="+tagToGetCacheForMenu);
							tagToGetCacheForMenu=false;	// Set to false to avoid infinite loop
					        // This call inject JavaScript into the page which just finished loading.
							view.loadUrl("javascript:window.HTMLOUT.functionJavaCalledByJsProcessHTML('menu','<head>'+document.getElementsByTagName('html')[0].innerHTML+'</head>');");	// Warning: This line makes Webkit fails with 2.3, that's why there is a version check before
						}
					}
				}
				// If we loaded page search_page.php, we trigger code to save content into cache
				if (url.contains("search_page.php")) 
				{
					synchronized (this) 
					{
						if (tagToGetCacheForQuickAccess)
						{
							Log.d(LOG_TAG, "onPageFinished Inject js to get html content, tagToGetCacheQuickAccess="+tagToGetCacheForQuickAccess);
							tagToGetCacheForQuickAccess=false;	// Set to false to avoid infinite loop
					        // This call inject JavaScript into the page which just finished loading.
							view.loadUrl("javascript:window.HTMLOUT.functionJavaCalledByJsProcessHTML('quickaccess','<head>'+document.getElementsByTagName('html')[0].innerHTML+'</head>');");	// Warning: This line makes Webkit fails with 2.3, that's why there is a version check before
						}
					}
				}
				*/

				if (! "".equals(nextAltHistoryStackBis)) {
					Log.d(LOG_TAG, "onPageFinished We now add an entry into history stack because nextAltHistoryStackBis="+nextAltHistoryStackBis);
					altHistoryStack.add(nextAltHistoryStackBis);
					nextAltHistoryStackBis="";
					if (url.contains("data:text/html")) {
                        nextAltHistoryStack="menu";
                    }
				}
				
				if (url.contains("data:text/html"))	{
					Log.d(LOG_TAG, "onPageFinished We finished to load a page with a bad history "+url);
					// If we go from a back, nextAltHistoryStack is ""
					//nextAltHistoryStackBis=(("".equals(nextAltHistoryStack) && url.contains("data:text/html"))?"menu":nextAltHistoryStack);
					nextAltHistoryStackBis=nextAltHistoryStack;	
					nextAltHistoryStack="";
				}

                messageNoPreviousPageShown = false;

				dumpBackForwardList(myWebView);
			}
	    }
		
		/**
		 * onReceivedHttpAuthRequest
		 */
	    @SuppressLint("AuthLeak")
		@Override
	    public void onReceivedHttpAuthRequest  (WebView view, HttpAuthHandler handler, String host, String realm)
	    { 
	    	Log.i(LOG_TAG, "A request to send http basic auth has been received");

	    	//String[] up = view.getHttpAuthUsernamePassword(host, realm); 

    		counthttpauth++;
            if (counthttpauth >= 3)
            {
                counthttpauth=0;
                Toast.makeText(getBaseContext(), "The server is protected by Basic Authentication. You must include login/pass into your login URL:\nhttp://login:password@mydomain.com", Toast.LENGTH_LONG).show();
                handler.cancel();
            }
            if (counthttpauth == 1)
            {
                counthttpauth++;
            }
            if (counthttpauth == 2) 
            {
                //Log.d(LOG_TAG, "We try to proceed with info from URL username="+savedAuthuser+" password="+savedAuthpass);
                Log.d(LOG_TAG, "We try to proceed with info from URL username="+savedAuthuser+" password=hidden");
                handler.proceed(savedAuthuser, savedAuthpass);
            }
	    	//webview.setHttpAuthUsernamePassword(host, realm, username, password);
	    }
	      
		/**
		 * onReceivedError
		 * This method is only called when network or webview errors occur, but never when a HTTP errors are received by WebView.
		 */
		@Override
        public void onReceivedError(WebView view, WebResourceRequest request, WebResourceError error)
		{
		    Log.e(LOG_TAG, "onReceivedError code: " + error.getErrorCode() + " on URL " + request.getUrl() + ": " + error.getDescription());
            super.onReceivedError(view, request, error);

            if ("net::ERR_ACCESS_DENIED".contentEquals(error.getDescription())) {
                Toast.makeText(activity, "Your WebView failed to gain permission for action (submit a captured file ?), for an unknown reason.", Toast.LENGTH_LONG).show();
            } else {
                Toast.makeText(activity, "Your Internet Connection may not be active Or " + error.getDescription() + ".", Toast.LENGTH_LONG).show();
            }
	    }
		
		/**
		 * onReceivedSslError
		 */
		@Override
		public void onReceivedSslError (WebView view, SslErrorHandler handler, SslError error) {
			String message = "SSL Certificate error.";
			switch (error.getPrimaryError()) {
				case SslError.SSL_UNTRUSTED:
					message = "The certificate authority is not trusted.";
					break;
				case SslError.SSL_EXPIRED:
					message = "The certificate has expired.";
					break;
				case SslError.SSL_IDMISMATCH:
					message = "The certificate Hostname mismatch.";
					break;
				case SslError.SSL_NOTYETVALID:
					message = "The certificate is not yet valid.";
					break;
				default:
					message = "Unknown certificate error " + error.getPrimaryError();
					break;
			}

			Log.w(LOG_TAG, "onReceivedSslError error message = " + message + " string = " + error);
			/*handler.proceed() ;
			handler.cancel(); */


			if (! this.secondActivity.sslErrorWasAccepted) {
				// Code to ask user how to handle error
				SslAlertDialog dialog = new SslAlertDialog(handler, this.secondActivity, message);
				dialog.show();
			}
			else
			{
				Log.w(LOG_TAG, "onReceivedSslError SSL error already accepted, we do handler.proceed()");
				handler.proceed();
			}

		}
		

		
		/**
		 * listCookies
		 * 
		 * @return	string		Return list of cookies with format name=value;name2=value2
		 */
		public String listCookies() 
		{
		    //CookieSyncManager.getInstance().sync();	// No more required with API 21
		    CookieManager cookie_manager = CookieManager.getInstance();

		    String cookie_string = cookie_manager.getCookie(savedDolRootUrl);
		    Log.v(LOG_TAG, "cookie_string (path " + savedDolRootUrl + ") = " + cookie_string);

		    return cookie_string;
		}
		
		/**
		 * deleteSessionCookies.
		 * Can be used to clear cookies to have a clean context for debug.
		 */
		public void deleteSessionCookies() 
		{
		    //CookieSyncManager.getInstance().sync();	// No more required with API 21
		    CookieManager cookie_manager = CookieManager.getInstance();

		    Log.v(LOG_TAG, "delete session cookies");
		    //cookie_manager.removeSessionCookie();
            cookie_manager.removeSessionCookies(null);
            //CookieSyncManager.getInstance().sync();	// No more required with API 21
	    
		    String cookie_string = cookie_manager.getCookie(savedDolRootUrl);
		    Log.v(LOG_TAG, "cookie_string (path " + savedDolRootUrl + ") = " + cookie_string);
		}		
	}


	/**
	 * WebChromeClientDoliDroid
	 */
	class WebChromeClientDoliDroid extends WebChromeClient
	{
		@Override
		public boolean onJsAlert(WebView view, String url, String message, final android.webkit.JsResult result)  
		{
			Log.d(LOG_TAG, message);
			//Toast.makeText(context, message, 3000).show();
			return true;
		}
		
		/**
		 * This can be called before of after the onPageFinished
		 * 
		 * @return		boolean		True if message is handled by client, false otherwise
		 */
		@Override
		public boolean onConsoleMessage(ConsoleMessage cm) 
		{
            if (cm != null && (cm.messageLevel() == ConsoleMessage.MessageLevel.TIP || cm.messageLevel() == ConsoleMessage.MessageLevel.LOG))	{
                Log.v(LOG_TAG, "onConsoleMessage "+cm.message() + " -- From line " + cm.lineNumber() + " of " + cm.sourceId());
            } else if (cm != null && cm.messageLevel() == ConsoleMessage.MessageLevel.DEBUG) {
                Log.d(LOG_TAG, "onConsoleMessage "+cm.message() + " -- From line " + cm.lineNumber() + " of " + cm.sourceId());
            } else if (cm != null && cm.messageLevel() == ConsoleMessage.MessageLevel.WARNING) {
                Log.w(LOG_TAG, "onConsoleMessage "+cm.message() + " -- From line " + cm.lineNumber() + " of " + cm.sourceId());
            } else if (cm != null && cm.messageLevel() == ConsoleMessage.MessageLevel.ERROR) {
                Log.e(LOG_TAG, "onConsoleMessage "+cm.message() + " -- From line " + cm.lineNumber() + " of " + cm.sourceId());
                // I don't know why we get this error with DoliDroid on web site ACE edit page. ACE seems to work correctly and we
                // don't have error on full browser mode. So i discard alert on this error message.
                if (cm.message() != null && !cm.message().contains("Failed to execute 'importScripts' on 'WorkerGlobalScope'")) {
                    tagToShowMessage = "Javascript error detected on page url = " + lastLoadUrl + " -- " + cm.message() + " -- From line " + cm.lineNumber() + " of " + cm.sourceId();
                    tagToShowCounter++;
                }
                return true;
            } else if (cm != null) {
                Log.e(LOG_TAG, "onConsoleMessage "+cm.message() + " -- From line " + cm.lineNumber() + " of " + cm.sourceId());
            } else {
                Log.e(LOG_TAG, "onConsoleMessage cm=null");
            }

			return false;
		}		
		
		/**
		 * Called during loading of page
		 */
		public void onProgressChanged (WebView view, int newProgress) 
		{
			//Log.i(LOG_TAG, "setProgress to "+(newProgress)+" current visibility="+(progress != null?progress.getVisibility():""));
			if (newProgress < 100 && progress.getVisibility() == ProgressBar.GONE)
			{
                progress.setVisibility(ProgressBar.VISIBLE);
            }
            progress.setProgress(newProgress);
            if (newProgress >= 100) 
            {
                progress.setVisibility(ProgressBar.GONE);
            }
        }

        /**
         * Called when clicked on input select file. With API >= 21 (Before it was openFileChooser)
         */
        public boolean onShowFileChooser(
                WebView webView, ValueCallback<Uri[]> filePathCallback,
                WebChromeClient.FileChooserParams fileChooserParams) {

            super.onShowFileChooser(webView, filePathCallback, fileChooserParams);

            Log.i(LOG_TAG, "onShowFileChooser fileChooserParams="+fileChooserParams);

            String[] acceptAttribute = fileChooserParams.getAcceptTypes();
            String nameAttribute = fileChooserParams.getFilenameHint();
            int multipleAttribute = fileChooserParams.getMode();

            // If capture attribute is not set, we use the default file chooser.
            // if capture attribute is set, we use the custom file chooser
            //boolean usecustomselect = fileChooserParams.isCaptureEnabled();
            boolean usecustomselect = true;

            // Log info on the input type=file attributes
            //int nbOfAttributes = acceptAttribute.length;
            for (String s : acceptAttribute) {
                // For example: s="image/*"
                Log.d(LOG_TAG, "acceptAttribute=" + s + " nameAttribute=" + nameAttribute);
            }

            if (mFilePathCallback != null) {
                mFilePathCallback.onReceiveValue(null);
            }
            mFilePathCallback = filePathCallback;

            if (!usecustomselect) {
                // Use the default selector.
                // If input has accept="image/*" the default selector already contains a filter.
                // However, the default selector does not allow to open the camera to take a picture
                Log.d(LOG_TAG, "onShowFileChooser use default selector");

                Intent intentDefault = fileChooserParams.createIntent();
                try {
                    startActivityForResult(intentDefault, REQUEST_INPUTFILE);
                } catch (ActivityNotFoundException e) {
                    mFilePathCallback = null;
                    Toast.makeText(getApplicationContext(), "Cannot open default file chooser", Toast.LENGTH_LONG).show();
                    return false;
                }
            } else {
                // The custom file selector
                Context context = getApplicationContext();
                PackageManager pm = context.getPackageManager();
                boolean enableCamera = pm.hasSystemFeature(PackageManager.FEATURE_CAMERA_ANY);
                //enableCamera = false;

                Log.d(LOG_TAG, "onShowFileChooser use custom selector enableCamera="+enableCamera);

                // Adjust the camera in a way that specifies the storage location for taking documents
                String filePath = getExternalFilesDir(Environment.DIRECTORY_PICTURES) + File.separator;

                Log.d(LOG_TAG, "filePath = "+filePath);

                // Create the storage directory if it does not exist
                File tmpDir = new File(filePath);
                if (!tmpDir.exists() && !tmpDir.mkdirs()) {
                    Log.e(LOG_TAG, "failed to create directory");
                } else {
                    Log.d(LOG_TAG, "directory already exists");
                }

                String timeStamp = new SimpleDateFormat("yyyyMMdd_HHmmss").format(new Date());
                String fileName = "Photo_"+timeStamp+".jpg";

                mCameraPhotoPathString = filePath + fileName;
                imageUri = Uri.fromFile(new File(mCameraPhotoPathString));

                Log.d(LOG_TAG, "onShowFileChooser imageUri for camera capture = "+imageUri);
                //    Intent i = new Intent(Intent.ACTION_GET_CONTENT);
                //    i.addCategory(Intent.CATEGORY_OPENABLE);
                //    i.setType("image/*");
                //    startActivityForResult(Intent.createChooser(i, "Image Chooser"), REQUEST_CODE_ABC);

                Intent intentDefault = fileChooserParams.createIntent();
                if (multipleAttribute == FileChooserParams.MODE_OPEN_MULTIPLE) {
                    intentDefault.putExtra(Intent.EXTRA_ALLOW_MULTIPLE, true);
                } else {
                    intentDefault.putExtra(Intent.EXTRA_ALLOW_MULTIPLE, false);
                }
                // Intent to get photos from photo galleries app (Google photo, ...)
                Intent chooserIntent = Intent.createChooser(intentDefault, "File Chooser");
                //chooserIntent.putExtra(Intent.EXTRA_INTENT, contentSelectionIntent);
                //chooserIntent.putExtra(Intent.EXTRA_TITLE, "File Chooser");

                // Add also the selector to capture a photo with name imageUri
                if (enableCamera) {
                    try {
                        String AuthorityString = context.getApplicationContext().getPackageName() + ".provider";
                        outputFileUri = FileProvider.getUriForFile(context, AuthorityString, new File(mCameraPhotoPathString));
                        context.getApplicationContext().grantUriPermission(context.getApplicationContext().getPackageName(), outputFileUri, Intent.FLAG_GRANT_READ_URI_PERMISSION | Intent.FLAG_GRANT_WRITE_URI_PERMISSION);

                        Log.d(LOG_TAG, "onShowFileChooser AuthorityString = " + AuthorityString + " imageUri = " + imageUri + " outputFileUri = " + outputFileUri);

                        /*
                        Option 1: Detect all intent available and forge chooserIntent with that

                        List<Intent> allIntents = new ArrayList();
                        Intent captureIntent = new Intent(android.provider.MediaStore.ACTION_IMAGE_CAPTURE);
                        List<android.content.pm.ResolveInfo> listCam = pm.queryIntentActivities(captureIntent, 0);
                        for (ResolveInfo res : listCam) {
                            Intent intent = new Intent(captureIntent);
                            intent.addFlags(Intent.FLAG_GRANT_READ_URI_PERMISSION | Intent.FLAG_GRANT_WRITE_URI_PERMISSION);
                            intent.setComponent(new ComponentName(res.activityInfo.packageName, res.activityInfo.name));
                            intent.setPackage(res.activityInfo.packageName);
                            if (outputFileUri != null) {
                                intent.putExtra(MediaStore.EXTRA_OUTPUT, outputFileUri);
                            }
                            allIntents.add(intent);
                        }

                        // collect all gallery intents
                        Intent galleryIntent = new Intent(Intent.ACTION_GET_CONTENT);       // Note ACTION_PICK should be considered deprecated
                        galleryIntent.setType("image/*");
                        List<android.content.pm.ResolveInfo> listGallery = pm.queryIntentActivities(galleryIntent, 0);
                        for (ResolveInfo res : listGallery) {
                            Intent intent = new Intent(galleryIntent);
                            intent.setComponent(new ComponentName(res.activityInfo.packageName, res.activityInfo.name));
                            intent.setPackage(res.activityInfo.packageName);
                            allIntents.add(intent);
                        }

                        // the main intent is the last in the list (fucking android) so pickup the useless one
                        Intent mainIntent = (Intent) allIntents.get(allIntents.size() - 1);
                        allIntents.remove(mainIntent);

                        // Create a chooser from the main intent
                        chooserIntent = Intent.createChooser(mainIntent, "File Chooser");

                        // Add all other intents
                        chooserIntent.putExtra(Intent.EXTRA_INITIAL_INTENTS, allIntents.toArray(new Parcelable[allIntents.size()]));
                        */

                        /*
                         Option 2 - Just add the ACTION_IMAGE_CAPTURE to default
                         */
                        Intent takePictureIntent = new Intent(android.provider.MediaStore.ACTION_IMAGE_CAPTURE);
                        takePictureIntent.addFlags(Intent.FLAG_GRANT_READ_URI_PERMISSION | Intent.FLAG_GRANT_WRITE_URI_PERMISSION);
                        //takePictureIntent.putExtra(MediaStore.EXTRA_OUTPUT, imageUri);    // Works on Android < 30 only
                        takePictureIntent.putExtra(MediaStore.EXTRA_OUTPUT, outputFileUri);

                        chooserIntent.putExtra(Intent.EXTRA_INITIAL_INTENTS, new Parcelable[]{takePictureIntent});
                        chooserIntent.addFlags(Intent.FLAG_GRANT_READ_URI_PERMISSION | Intent.FLAG_GRANT_WRITE_URI_PERMISSION);
                    }
                    catch(Exception e) {
                        Log.e(LOG_TAG, Objects.requireNonNull(e.getMessage(), "Exception"));
                    }
                }

                // Start activity to choose file
                try {
                    startActivityForResult(chooserIntent, REQUEST_INPUTFILE);
                } catch (ActivityNotFoundException e) {
                    mFilePathCallback = null;
                    Toast.makeText(getApplicationContext(), "Cannot open custom file chooser", Toast.LENGTH_LONG).show();
                    return false;
                } catch (Exception e) {
                    Log.e(LOG_TAG, "onShowFileChooser Failed to startActivityForResult");
                }
            }

            return true;
        }
    }

    /**
     * onResume
     */
    @Override
    protected void onResume()
    {
        Log.d(LOG_TAG, "onResume start cookie sync");
        //CookieSyncManager.getInstance().startSync();    // No more required with API 21
        super.onResume();
    }
    
    /**
     * onPause
     */
    @Override
    protected void onPause()
    {
        Log.d(LOG_TAG, "onPause stop cookie sync");
        //CookieSyncManager.getInstance().stopSync(); // No more required with API 21
        super.onPause();
    }
    
    /**
     * onStop
     */
    @Override
    protected void onStop() 
    {
        setResult(RESULT_WEBVIEW);
        super.onStop();
    }
    
    /**
     * onDestroy
     */
    @Override
    protected void onDestroy() 
    {
        // Delete iabHelper for InApp management
        //iabHelper.dispose();
        
        setResult(RESULT_WEBVIEW);
        super.onDestroy();
    }


    /**
     * An instance of this class will be registered as a JavaScript interface
     */
    public class MyJavaScriptInterface
    {
        private static final String LOG_TAG = "DoliDroidLogMyJavaScriptInterface";
        Context mContext;
        Activity activity;

        MyJavaScriptInterface(Context c) 
        {
             mContext = c;
        }

        MyJavaScriptInterface(Activity a)
        {
            activity = a;
        }

        /*
        @JavascriptInterface
        public void functionJavaCalledByJsProcessHTML(String page, String html)
        {
            // Process the html as needed by the app
            // Toast.makeText(mContext, toast, Toast.LENGTH_SHORT).show();
            Log.i(LOG_TAG, "functionJavaCalledByJsProcessHTML page="+page
                    //+" html="+(html == null ? "null" : html.replaceAll("\n", " "))
                    );
            if (html != null && page.equals("menu") && html.contains("<!-- Menu -->"))
            {
                cacheForMenu=html;
                Log.d(LOG_TAG, "Save menu content into cache");
            }
            else
            {
                Log.w(LOG_TAG, "Content does not look to be menu nor quick access page, so we don't cache it");
            }
        }
        */
        
        /*
         * Example of data:
         * method=post&action=http://192.168.0.1/index.php?dol_hide_topmenu=1&dol_hide_leftmenu=1&dol_optimize_smallscreen=1&dol_no_mouse_hover=1&dol_use_jmobile=1&mainmenu=home&token=cf51a99cad2639a6f2da61753748dc16&loginfunction=loginfunction&tz=0&tz_string=GMT&dst_observed=0&dst_first=&dst_second=&screenwidth=320&screenheight=460&dol_hide_topmenu=1&dol_hide_leftmenu=1&dol_optimize_smallscreen=1&dol_no_mouse_hover=1&dol_use_jmobile=1&username=admin&password=admin
         */
        @JavascriptInterface
        public void functionJavaCalledByJsProcessFormSubmit(String data)
        {
            Log.i(LOG_TAG, "functionJavaCalledByJsProcessFormSubmit execution of code injected by jsInjectCodeForSetForm with data="+data);
            String[] tmpdata = data.split("&");

            // Save the username and password into temporary var lastsubmit-username and lastsubmit-password
            try {
                String masterKeyAlias = MasterKeys.getOrCreate(MasterKeys.AES256_GCM_SPEC);

                File prefsFile = new File(getApplicationContext().getFilesDir(), "../shared_prefs/secret_shared_prefs.xml");
                if (prefsFile.exists()) {
                    Log.d(LOG_TAG, "The file "+prefsFile+" already exists");
                } else {
                    Log.d(LOG_TAG, "The file "+prefsFile+" does not exists yet");
                }

                SharedPreferences sharedPrefsEncrypted = EncryptedSharedPreferences.create(
                        "secret_shared_prefs",
                        masterKeyAlias,
                        getApplicationContext(),
                        EncryptedSharedPreferences.PrefKeyEncryptionScheme.AES256_SIV,
                        EncryptedSharedPreferences.PrefValueEncryptionScheme.AES256_GCM
                );

                SharedPreferences.Editor editor = sharedPrefsEncrypted.edit();
                for (String s : tmpdata) {
                    String[] keyval = s.split("=", 2);
                    if (keyval.length >= 2) {
                        String key = keyval[0];
                        String val = keyval[1];
                        if ("username".equals(key) || "password".equals(key)) {
                            tagLastLoginPassToSavedLoginPass = true;
                            if ("username".equals(key)) {
                                Log.d(LOG_TAG, "functionJavaCalledByJsProcessFormSubmit save lastsubmit-" + key + "=" + val);
                            } else {
                                Log.d(LOG_TAG, "functionJavaCalledByJsProcessFormSubmit save lastsubmit-" + key + "=hidden");
                            }
                            editor.putString("lastsubmit-" + key, val);
                        }
                    }
                }
                editor.apply();
            }
            catch(Exception e) {
                // Pb read/write file getApplicationContext().getFilesDir()."../shared_prefs/secret_shared_prefs"
                Log.w(LOG_TAG, "functionJavaCalledByJsProcessFormSubmit Failed to read the EncryptedSharedPreferences");
                Log.e(LOG_TAG, Objects.requireNonNull(e.getMessage()));
            }
        }
    }

    /**
     * This is called after another opened activity is finished.
     * We go back here.
     * For example, when we go back after selecting a file from the file selector
     */
    @Override
    public void onActivityResult(int requestCode, int resultCode, Intent data) 
    {
        Log.i(LOG_TAG, "onActivityResult requestCode = "+requestCode+" resultCode = "+resultCode + " mFilePathCallback = " + mFilePathCallback);

        if (requestCode != REQUEST_INPUTFILE) {
            Log.d(LOG_TAG, "onActivityResult not a return after an input file selection");
            // Not a file upload, we make standard action.
            super.onActivityResult(requestCode, resultCode, data);
            return;
        }
        if (mFilePathCallback == null) {
            Log.d(LOG_TAG, "onActivityResult return after input file selection but mFilePathCallback was empty");
            // Not a file upload, we make standard action.
            super.onActivityResult(requestCode, resultCode, data);
            return;
        }

        // Handle here if return comes from submit a file. mFilePathCallback is not null.
        Log.d(LOG_TAG, "onActivityResult we should have just selected a file from an external activity");

        if (data != null) {
            Log.d(LOG_TAG, "onActivityResult data = "+data);
        } else {
            Log.d(LOG_TAG, "onActivityResult data is null");
        }
        // Example: Intent { dat=content://com.android.providers.downloads.documents/document/74 flg=0x1 } after selection of file manager
        // Example: Intent { dat=content://com.android.providers.media.documents/document/image:88 flg=0x1 } after selection of file manager


        // Check that the response is a good one
        if (resultCode == Activity.RESULT_OK) 
        {
            Log.d(LOG_TAG, "onActivityResult result code is ok");

            Uri uri = null;
            ClipData uris = null;

            if (data != null) {
                /*
                // When we receive the image as a bitmap because we have created the chooser with a simple chooser and intent
                Bundle extras = data.getExtras();
                if (extras != null) {
                    Bitmap imageBitmap = (Bitmap) extras.get("data");
                    //imageView.setImageBitmap(imageBitmap);
                }
                */

                uri = data.getData();
                uris = data.getClipData();
            } else {
                // We refresh the gallery after taking a photo
                Log.d(LOG_TAG, "onActivityResult imageUri="+imageUri+" outputFileUri="+outputFileUri);

                // Refresh the media so it will see the new captured file. Old method
                Intent intent = new Intent(Intent.ACTION_MEDIA_SCANNER_SCAN_FILE);
                intent.setData(imageUri);
                sendBroadcast(intent);

                // Refresh the media so it will see the new captured file. New method
                // To test this, juste after using the camera to capture an image, switch to the Files application into Pictures directory and check you see the new file.
                File file = new File(imageUri.getPath());
                MediaScannerConnection.scanFile(getApplicationContext(), new String[]{file.getAbsolutePath()}, null,
                        new MediaScannerConnection.OnScanCompletedListener() {
                            public void onScanCompleted(String path, Uri uri) {
                                // now visible in gallery
                                Log.d(LOG_TAG, "MediaScannerConnection callback: Media have been refreshed");
                            }});
            }

            if (data == null || (uri == null && uris == null)) {
                // Case we have taken a photo
                Uri[] results = null;

                Log.d(LOG_TAG, "onActivityResult data or (uri and uris is null), mCameraPhotoPathString="+mCameraPhotoPathString+" imageUri="+imageUri);

                // If there is not data, then we may have taken a photo
                if (imageUri != null) {
                    //Uri[] results = null;
                    //results = new Uri[]{Uri.parse(mCameraPhotoPathString)};

                    results = new Uri[]{imageUri};

                    Log.d(LOG_TAG, "onActivityResult results="+ Arrays.toString(results));

                    mFilePathCallback.onReceiveValue(results);
                }
                mCameraPhotoPathString = null;
                imageUri = null;
            } else {
                // Case we have selected a file from file manager
                Uri[] results = null;

                Log.d(LOG_TAG, "onActivityResult data is not null");

                if (uri != null) {
                    Log.d(LOG_TAG, "onActivityResult uri="+uri);
                    // Example: content://com.android.providers.media.documents/document/image%3A88
                    results = new Uri[]{uri};
                    for (Uri uriTmp : results) {
                        if (uriTmp != null) {
                            Log.d(LOG_TAG, "onActivityResult system return URI:" + uriTmp);
                            // Example: URI:content://com.android.providers.downloads.documents/document/74
                            // Example: URI:content://com.android.providers.media.documents/document/image%3A88
                        }
                    }
                } else if (uris != null) {
                    Log.d(LOG_TAG, "onActivityResult uris="+uris);
                    // Example: ClipData { */* {U:content://com.android.providers.media.documents/document/image%3A112} {U:content://com.android.providers.media.documents/document/image%3A113} }
                    results = new Uri[uris.getItemCount()];
                    int count = uris.getItemCount();
                    for (int i = 0; i < count; i++) {
                        ClipData.Item item = uris.getItemAt(i);
                        Log.d(LOG_TAG, "onActivityResult system return URI:" + item.getUri());
                        // Example: URI:content://com.android.providers.media.documents/document/image%3A88
                        results[i] = item.getUri();
                    }
                }

                Log.d(LOG_TAG, "onActivityResult results="+ Arrays.toString(results));

                mFilePathCallback.onReceiveValue(results);
            }
        } else {
            Log.d(LOG_TAG, "onActivityResult resultCode is 0 (error)");
            mFilePathCallback.onReceiveValue(null);
        }

        Log.d(LOG_TAG, "onActivityResult end of management of external activity result");
        mFilePathCallback = null;

        // After this the onStart of SecondActivity should be executed
    }
}
