package com.nfcalarmclock.alarm.db

import android.content.Context
import android.os.Parcel
import android.os.Parcelable
import androidx.room.ColumnInfo
import androidx.room.Entity
import androidx.room.PrimaryKey
import com.nfcalarmclock.R
import com.nfcalarmclock.nfc.NacNfcTagDismissOrder
import com.nfcalarmclock.nfc.db.NacNfcTag
import com.nfcalarmclock.shared.NacSharedPreferences
import com.nfcalarmclock.system.NacCalendar
import com.nfcalarmclock.system.NacCalendar.Day
import com.nfcalarmclock.system.NacCalendar.calendarToString
import com.nfcalarmclock.system.daysToValue
import com.nfcalarmclock.system.removeToday
import com.nfcalarmclock.system.toCalendarField
import com.nfcalarmclock.system.toDay
import com.nfcalarmclock.system.toDays
import com.nfcalarmclock.view.quickToast
import dagger.Module
import dagger.Provides
import dagger.hilt.InstallIn
import dagger.hilt.components.SingletonComponent
import java.util.Calendar
import java.util.EnumSet
import java.util.Locale

/**
 * Normalize a name.
 */
fun String.normalizeName(): String
{
	return if (this.isNotEmpty())
	{
		this.replace("\n", " ")
	}
	else
	{
		this
	}
}

/**
 * Next alarm object.
 *
 * Really just a container for a NacAlarm and Calendar.
 */
class NacNextAlarm(

	/**
	 * Alarm object corresponding to the next alarm that will run.
	 */
	val alarm: NacAlarm,

	/**
	 * Calendar object corresponding to the next date and time the alarm will run.
	 */
	val calendar: Calendar

)

/**
 * Alarm.
 */
@Entity(tableName = "alarm")
open class NacAlarm()
	: Comparable<NacAlarm>,
	Parcelable
{

	/**
	 * Unique alarm ID.
	 * <p>
	 * When setting the Id manually, it must be 0 to be autogenerated.
	 */
	@PrimaryKey(autoGenerate = true)
	@ColumnInfo(name = "id")
	var id: Long = 0

	/**
	 * Whether the alarm is currently active or not.
	 */
	@ColumnInfo(name = "is_active", defaultValue = "0")
	var isActive: Boolean = false

	/**
	 * Amount of time, in milliseconds, the alarm has been active for.
	 *
	 * This will typically only change when the alarm is snoozed.
	 */
	@ColumnInfo(name = "time_active", defaultValue = "0")
	var timeActive: Long = 0

	/**
	 * Number of times the alarm has been snoozed.
	 */
	@ColumnInfo(name = "snooze_count", defaultValue = "0")
	var snoozeCount: Int = 0

	/**
	 * Whether the alarm is enabled or not.
	 */
	@ColumnInfo(name = "is_enabled", defaultValue = "0")
	var isEnabled: Boolean = false

	/**
	 * Hour at which to run the alarm.
	 */
	@ColumnInfo(name = "hour", defaultValue = "0")
	var hour: Int = 0

	/**
	 * Minute at which to run the alarm.
	 */
	@ColumnInfo(name = "minute", defaultValue = "0")
	var minute: Int = 0

	/**
	 * Days on which to run the alarm.
	 */
	@ColumnInfo(name = "days")
	var days: EnumSet<Day> = EnumSet.noneOf(Day::class.java)

	/**
	 * Date on which to run the alarm, in format of YYYY-MM-DD.
	 */
	@ColumnInfo(name = "date", defaultValue = "")
	var date: String = ""

	/**
	 * Whether the alarm should be repeated or not.
	 */
	@ColumnInfo(name = "should_repeat", defaultValue = "0")
	var shouldRepeat: Boolean = false

	/**
	 * Frequency at which to repeat the alarm.
	 */
	@ColumnInfo(name = "repeat_frequency", defaultValue = "1")
	var repeatFrequency: Int = 1

	/**
	 * Units for the frequency at which to repeat the alarm.
	 *
	 * 1 = Minutes
	 * 2 = Hours
	 * 3 = Days
	 * 4 = Weeks
	 * 5 = Months
	 */
	@ColumnInfo(name = "repeat_frequency_units", defaultValue = "4")
	var repeatFrequencyUnits: Int = 4

	/**
	 * Units for the frequency at which to repeat the alarm.
	 *
	 * 1 = Minutes
	 * 2 = Hours
	 * 3 = Days
	 * 4 = Weeks
	 * 5 = Months
	 */
	@ColumnInfo(name = "repeat_frequency_days_to_run_before_starting", defaultValue = "127")
	var repeatFrequencyDaysToRunBeforeStarting: EnumSet<Day> = Day.WEEK

	/**
	 * Whether the alarm should vibrate the phone or not.
	 */
	@ColumnInfo(name = "should_vibrate", defaultValue = "0")
	var shouldVibrate: Boolean = false

	/**
	 * Duration to vibrate the device for.
	 */
	@ColumnInfo(name = "vibrate_duration", defaultValue = "500")
	var vibrateDuration: Long = 500

	/**
	 * Amount of time to wait in between vibrations.
	 */
	@ColumnInfo(name = "vibrate_wait_time", defaultValue = "1000")
	var vibrateWaitTime: Long = 1000

	/**
	 * Whether to vibrate using a pattern or not.
	 */
	@ColumnInfo(name = "should_vibrate_pattern", defaultValue = "0")
	var shouldVibratePattern: Boolean = false

	/**
	 * Number of times to repeat the vibration.
	 */
	@ColumnInfo(name = "vibrate_repeat_pattern", defaultValue = "3")
	var vibrateRepeatPattern: Int = 3

	/**
	 * Amount of time to wait after the vibration has been repeated the set number of
	 * times.
	 */
	@ColumnInfo(name = "vibrate_wait_time_after_pattern", defaultValue = "3000")
	var vibrateWaitTimeAfterPattern: Long = 3000

	/**
	 * Whether the alarm should use NFC or not.
	 */
	@ColumnInfo(name = "should_use_nfc", defaultValue = "0")
	var shouldUseNfc: Boolean = false

	/**
	 * ID of the NFC tag that needs to be used to dismiss the alarm.
	 */
	@ColumnInfo(name = "nfc_tag_id", defaultValue = "")
	var nfcTagId: String = ""

	/**
	 * Order in which to dismiss NFC tags when multiple are selected.
	 */
	@ColumnInfo(name = "should_use_nfc_tag_dismiss_order", defaultValue = "0")
	var shouldUseNfcTagDismissOrder: Boolean = false

	/**
	 * Order in which to dismiss NFC tags when multiple are selected.
	 *
	 * 1 = RANDOM
	 * 2 = SEQUENTIAL
	 */
	@ColumnInfo(name = "nfc_tag_dismiss_order", defaultValue = "2")
	var nfcTagDismissOrder: Int = 2

	/**
	 * Whether the alarm should use the flashlight or not.
	 */
	@ColumnInfo(name = "should_use_flashlight", defaultValue = "0")
	var shouldUseFlashlight: Boolean = false

	/**
	 * Strength level of the flashlight. Only available on API >= 33.
	 */
	@ColumnInfo(name = "flashlight_strength_level", defaultValue = "0")
	var flashlightStrengthLevel: Int = 0

	/**
	 * Amount of time, in seconds, to wait before gradually increasing the flashlight
	 * strength level another step.
	 */
	@ColumnInfo(name = "gradually_increase_flashlight_strength_level_wait_time", defaultValue = "5")
	var graduallyIncreaseFlashlightStrengthLevelWaitTime: Int = 5

	/**
	 * Whether the flashlight should be blinked or not.
	 */
	@ColumnInfo(name = "should_blink_flashlight", defaultValue = "0")
	var shouldBlinkFlashlight: Boolean = false

	/**
	 * Number of seconds to turn on the flashlight.
	 */
	@ColumnInfo(name = "flashlight_on_duration", defaultValue = "1.0")
	var flashlightOnDuration: String = "1.0"

	/**
	 * Number of seconds to turn off the flashlight.
	 */
	@ColumnInfo(name = "flashlight_off_duration", defaultValue = "1.0")
	var flashlightOffDuration: String = "1.0"

	/**
	 * Path to the media that will play when the alarm is run.
	 */
	@ColumnInfo(name = "media_path", defaultValue = "")
	var mediaPath: String = ""

	/**
	 * Artist of the media that will play when the alarm is run.
	 */
	@ColumnInfo(name = "media_artist", defaultValue = "")
	var mediaArtist: String = ""

	/**
	 * Title of the media that will play when the alarm is run.
	 */
	@ColumnInfo(name = "media_title", defaultValue = "")
	var mediaTitle: String = ""

	/**
	 * Type of media.
	 */
	@ColumnInfo(name = "media_type", defaultValue = "0")
	var mediaType: Int = 0

	/**
	 * Local path, internal to the app, to the media that will play when the alarm is run.
	 *
	 * This will only be used if an alarm goes off in direct boot mode.
	 */
	@ColumnInfo(name = "local_media_path", defaultValue = "")
	var localMediaPath: String = ""

	/**
	 * Whether to shuffle the media.
	 *
	 * Note: This is only applicable if playing a directory, otherwise it will
	 *       be ignored.
	 */
	@ColumnInfo(name = "should_shuffle_media", defaultValue = "0")
	var shouldShuffleMedia: Boolean = false

	/**
	 * Whether to recursively play the media in a directory.
	 *
	 * Note: This is only applicable if playing a directory, otherwise it will
	 *       be ignored.
	 */
	@ColumnInfo(name = "should_recursively_play_media", defaultValue = "0")
	var shouldRecursivelyPlayMedia: Boolean = false

	/**
	 * Volume level to set when the alarm is run.
	 */
	@ColumnInfo(name = "volume", defaultValue = "0")
	var volume: Int = 0

	/**
	 * Audio source to use for the media that will play when the alarm is run.
	 */
	@ColumnInfo(name = "audio_source", defaultValue = "")
	var audioSource: String = ""

	/**
	 * Name of the alarm.
	 */
	@ColumnInfo(name = "name", defaultValue = "")
	var name: String = ""

	/**
	 * Whether to say the current time via text-to-speech when the alarm goes off.
	 */
	@ColumnInfo(name = "should_say_current_time", defaultValue = "0")
	var shouldSayCurrentTime: Boolean = false

	/**
	 * Whether to say the alarm name via text-to-speech when the alarm goes off.
	 */
	@ColumnInfo(name = "should_say_name", defaultValue = "0")
	var shouldSayName: Boolean = false

	/**
	 * Frequency at which to play text-to-speech, in units of minutes.
	 */
	@ColumnInfo(name = "tts_frequency", defaultValue = "0")
	var ttsFrequency: Int = 0

	/**
	 * Speech rate to use for text-to-speech.
	 */
	@ColumnInfo(name = "tts_speech_rate", defaultValue = "0.7")
	var ttsSpeechRate : Float = 0.7f

	/**
	 * Name of the voice to use for text-to-speech.
	 */
	@ColumnInfo(name = "tts_voice", defaultValue = "")
	var ttsVoice: String = ""

	/**
	 * Whether to gradually increase the volume or not, when an alarm is active.
	 */
	@ColumnInfo(name = "should_gradually_increase_volume", defaultValue = "0")
	var shouldGraduallyIncreaseVolume: Boolean = false

	/**
	 * Amount of time, in seconds, to wait before gradually increasing the
	 * volume another step.
	 */
	@ColumnInfo(name = "gradually_increase_volume_wait_time", defaultValue = "5")
	var graduallyIncreaseVolumeWaitTime: Int = 5

	/**
	 * Whether to restrict changing the volume or not, when an alarm is active.
	 */
	@ColumnInfo(name = "should_restrict_volume", defaultValue = "0")
	var shouldRestrictVolume: Boolean = false

	/**
	 * Whether to auto dismiss or not.
	 */
	@ColumnInfo(name = "should_auto_dismiss", defaultValue = "1")
	var shouldAutoDismiss: Boolean = true

	/**
	 * Time in which to auto dismiss the alarm (seconds).
	 */
	@ColumnInfo(name = "auto_dismiss_time", defaultValue = "900")
	var autoDismissTime: Int = 900

	/**
	 * Whether the alarm can be dismissed early or not.
	 */
	@ColumnInfo(name = "can_dismiss_early", defaultValue = "0")
	var canDismissEarly: Boolean = false

	/**
	 * Amount of time to allow a user to dismiss early by (minutes).
	 */
	@ColumnInfo(name = "dismiss_early_time", defaultValue = "30")
	var dismissEarlyTime: Int = 30

	/**
	 * Time of alarm that would have been next but was dismissed early.
	 */
	@ColumnInfo(name = "time_of_dismiss_early_alarm", defaultValue = "0")
	var timeOfDismissEarlyAlarm: Long = 0

	/**
	 * Whether to show a notification for dismiss early or not.
	 */
	@ColumnInfo(name = "should_show_dismiss_early_notification", defaultValue = "0")
	var shouldShowDismissEarlyNotification: Boolean = false

	/**
	 * Whether to delete the alarm after it is dismissed or not.
	 */
	@ColumnInfo(name = "should_delete_after_dismissed", defaultValue = "0")
	var shouldDeleteAfterDismissed: Boolean = false

	/**
	 * Whether to auto snooze or not.
	 */
	@ColumnInfo(name = "should_auto_snooze", defaultValue = "0")
	var shouldAutoSnooze: Boolean = false

	/**
	 * Time in which to auto snooze the alarm (seconds).
	 */
	@ColumnInfo(name = "auto_snooze_time", defaultValue = "300")
	var autoSnoozeTime: Int = 300

	/**
	 * Max number of snoozes.
	 */
	@ColumnInfo(name = "max_snooze", defaultValue = "-1")
	var maxSnooze: Int = -1

	/**
	 * Snooze duration (seconds).
	 */
	@ColumnInfo(name = "snooze_duration", defaultValue = "300")
	var snoozeDuration: Int = 300

	/**
	 * Whether snoozing is easy or not.
	 */
	@ColumnInfo(name = "should_easy_snooze", defaultValue = "0")
	var shouldEasySnooze: Boolean = false

	/**
	 * Whether snoozing with a volume button is allowed or not.
	 */
	@ColumnInfo(name = "should_volume_snooze", defaultValue = "0")
	var shouldVolumeSnooze: Boolean = false

	/**
	 * Whether to show a reminder or not.
	 */
	@ColumnInfo(name = "should_show_reminder", defaultValue = "0")
	var shouldShowReminder: Boolean = false

	/**
	 * The time to start showing a reminder (minutes).
	 */
	@ColumnInfo(name = "time_to_show_reminder", defaultValue = "5")
	var timeToShowReminder: Int = 5

	/**
	 * Frequency at which to show the reminder, in units of minutes.
	 */
	@ColumnInfo(name = "reminder_frequency", defaultValue = "0")
	var reminderFrequency: Int = 0

	/**
	 * Whether to use text-to-speech for the reminder.
	 */
	@ColumnInfo(name = "should_use_tts_for_reminder", defaultValue = "0")
	var shouldUseTtsForReminder: Boolean = false

	/**
	 * Whether to skip the next alarm or not.
	 */
	@ColumnInfo(name = "should_skip_next_alarm", defaultValue = "0")
	var shouldSkipNextAlarm: Boolean = false

	/**
	 * Check if the alarm has a sound that will be played when it goes off.
	 */
	val hasMedia: Boolean
		get() = mediaPath.isNotEmpty()

	/**
	 * Check if the alarm can be snoozed.
	 *
	 * @return True if the alarm can be snoozed, and False otherwise.
	 */
	val canSnooze: Boolean
		get() = (snoozeCount < maxSnooze) || (maxSnooze < 0)

	/**
	 * Check if the alarm is snoozed.
	 */
	val isSnoozed: Boolean
		get() = snoozeCount > 0

	/**
	 * Check if the alarm is being used, by being active or snoozed.
	 */
	val isInUse: Boolean
		get() = isActive || isSnoozed

	/**
	 * Check if the next alarm will be skipped, and there is only one day that the alarm,
	 * will run on is being used, and repeat is disabled.
	 *
	 * This is a unique case where the next alarm is effectively the final alarm, so
	 * there are basically no alarms after this.
	 */
	val isNextSkippedAndFinal: Boolean
		get() = shouldSkipNextAlarm && (days.size <= 1) && !shouldRepeat

	/**
	 * The normalized alarm name (with newlines replaced with spaces).
	 */
	val nameNormalized: String
		get()
		{
			return name.normalizeName()
		}

	/**
	 * ID of the NFC tag that needs to be used to dismiss the alarm.
	 */
	val nfcTagIdList: List<String>
		get()
		{
			// Create the regex
			val regex = Regex(" \\|\\| ")

			return if (nfcTagId.isEmpty())
			{
				// No NFC ID
				emptyList()
			}
			else
			{
				// Try to split the NFC IDs
				nfcTagId.split(regex)
			}
		}

	/**
	 * Check if should use TTS or not.
	 */
	val shouldUseTts: Boolean
		get() = shouldSayCurrentTime || shouldSayName

	/**
	 * Populate values with input parcel.
	 */
	private constructor(input: Parcel) : this()
	{
		// ID
		id = input.readLong()

		// Active alarm flags
		isActive = input.readInt() != 0
		timeActive = input.readLong()
		snoozeCount = input.readInt()

		// Normal stuff
		isEnabled = input.readInt() != 0
		hour = input.readInt()
		minute = input.readInt()
		setDays(input.readInt())
		date = input.readString() ?: ""

		// Repeat
		shouldRepeat = input.readInt() != 0
		repeatFrequency = input.readInt()
		repeatFrequencyUnits = input.readInt()
		repeatFrequencyDaysToRunBeforeStarting = input.readInt().toDays()

		// Vibrate
		shouldVibrate = input.readInt() != 0
		vibrateDuration = input.readLong()
		vibrateWaitTime = input.readLong()
		shouldVibratePattern = input.readInt() != 0
		vibrateRepeatPattern= input.readInt()
		vibrateWaitTimeAfterPattern= input.readLong()

		// NFC
		shouldUseNfc = input.readInt() != 0
		nfcTagId = input.readString() ?: ""
		shouldUseNfcTagDismissOrder = input.readInt() != 0
		nfcTagDismissOrder = input.readInt()

		// Flashlight
		shouldUseFlashlight = input.readInt() != 0
		flashlightStrengthLevel = input.readInt()
		graduallyIncreaseFlashlightStrengthLevelWaitTime = input.readInt()
		shouldBlinkFlashlight = input.readInt() != 0
		flashlightOnDuration = input.readString() ?: "1.0"
		flashlightOffDuration = input.readString() ?: "1.0"

		// Media
		mediaPath = input.readString() ?: ""
		mediaArtist = input.readString() ?: ""
		mediaTitle = input.readString() ?: ""
		mediaType = input.readInt()
		localMediaPath = input.readString() ?: ""
		shouldShuffleMedia = input.readInt() != 0
		shouldRecursivelyPlayMedia = input.readInt() != 0

		// Volume and audio source
		volume = input.readInt()
		shouldGraduallyIncreaseVolume = input.readInt() != 0
		graduallyIncreaseVolumeWaitTime = input.readInt()
		shouldRestrictVolume = input.readInt() != 0
		audioSource = input.readString() ?: ""

		// Name
		name = input.readString() ?: ""

		// Text-to-speech
		shouldSayCurrentTime = input.readInt() != 0
		shouldSayName = input.readInt() != 0
		ttsFrequency = input.readInt()
		ttsSpeechRate = input.readFloat()
		ttsVoice = input.readString() ?: ""

		// Dismiss
		shouldAutoDismiss = input.readInt() != 0
		autoDismissTime = input.readInt()
		canDismissEarly = input.readInt() != 0
		dismissEarlyTime = input.readInt()
		timeOfDismissEarlyAlarm = input.readLong()
		shouldShowDismissEarlyNotification = input.readInt() != 0
		shouldDeleteAfterDismissed = input.readInt() != 0

		// Snooze
		shouldAutoSnooze = input.readInt() != 0
		autoSnoozeTime = input.readInt()
		maxSnooze = input.readInt()
		snoozeDuration = input.readInt()
		shouldEasySnooze = input.readInt() != 0
		shouldVolumeSnooze = input.readInt() != 0

		// Reminder
		shouldShowReminder = input.readInt() != 0
		timeToShowReminder = input.readInt()
		reminderFrequency = input.readInt()
		shouldUseTtsForReminder = input.readInt() != 0

		// Skip next alarm
		shouldSkipNextAlarm = input.readInt() != 0
	}

	/**
	 * Add the days repeat frequency to the alarm time.
	 */
	fun addRepeatFrequencyDaysToTime(alarmCal: Calendar)
	{
		// Every 1 day
		if (repeatFrequency == 1)
		{
			return
		}
		// Every 2-7 days
		else if (repeatFrequency <= 7)
		{
			// Remove today if it is in the days
			days.removeToday()

			// Get the new day of week
			val calDay = alarmCal.get(Calendar.DAY_OF_WEEK)
			val newDay = calDay.toDay()

			// Add the new day to the days
			days.add(newDay)
		}
		// Every 8+ days
		else
		{
			// Get the new day of week
			val year = alarmCal.get(Calendar.YEAR)
			val month = alarmCal.get(Calendar.MONTH)
			val day = alarmCal.get(Calendar.DAY_OF_MONTH)

			// Use a date to keep track of when the alarm should run
			date = "$year-${month+1}-$day"
		}
	}

	/**
	 * Add the hours repeat frequency to the alarm time.
	 */
	fun addRepeatFrequencyHoursToTime(alarmCal: Calendar)
	{
		// Update the alarm hour
		hour = alarmCal.get(Calendar.HOUR_OF_DAY)

		// Every 25+ hours
		if (repeatFrequency > 24)
		{
			// Remove all the days
			days.clear()

			// Get the new day of week
			val calDay = alarmCal.get(Calendar.DAY_OF_WEEK)
			val newDay = calDay.toDay()

			// Add the new day to the days
			days.add(newDay)
		}
	}

	/**
	 * Add the minutes repeat frequency to the alarm time.
	 */
	fun addRepeatFrequencyMinutesToTime(alarmCal: Calendar)
	{
		// Update the alarm hour and minute
		hour = alarmCal.get(Calendar.HOUR_OF_DAY)
		minute = alarmCal.get(Calendar.MINUTE)
	}

	/**
	 * Add the months repeat frequency to the alarm time.
	 */
	fun addRepeatFrequencyMonthsToTime(alarmCal: Calendar)
	{
		// Get the new day of week
		val year = alarmCal.get(Calendar.YEAR)
		val month = alarmCal.get(Calendar.MONTH)
		val day = alarmCal.get(Calendar.DAY_OF_MONTH)

		// Set the new date
		date = "$year-${month+1}-$day"
	}

	/**
	 * Add the weeks repeat frequency to the alarm time.
	 */
	fun addRepeatFrequencyWeeksToTime()
	{
		// Get the next alarm day
		val nextCal = NacCalendar.getNextAlarmDay(this)!!
		println("NacAlarm: Next cal : ${calendarToString(nextCal, "EEE MMM dd HH:mm:ss z yyyy")}")

		// Ensure that the days to run before starting does not have any extra
		// days selected. These would be days that do not match the alarm days.
		// Remove any of the extra days and do a final compare if the days to run
		// before starting should be marked as empty or not
		if (repeatFrequencyDaysToRunBeforeStarting.isNotEmpty())
		{
			// Get any overlap between the days to run before starting and the days selected for the alarm
			val daysToRunOverlap = EnumSet.copyOf(repeatFrequencyDaysToRunBeforeStarting)
			daysToRunOverlap.retainAll(days)
			println("Quick check of days : $days | $repeatFrequencyDaysToRunBeforeStarting | $daysToRunOverlap")

			// There is no overlap between the days to run before starting and the days selected for the alarm
			if (daysToRunOverlap.isEmpty())
			{
				println("No days overlap! Clear the days to run before starting")
				// Clear the days to run as it probably had extra days that were not needed
				repeatFrequencyDaysToRunBeforeStarting = EnumSet.noneOf(Day::class.java)
			}
		}

		// No more days to run before starting the repeat frequency
		if (repeatFrequencyDaysToRunBeforeStarting.isEmpty())
		{
			// Get the new day of week
			val year = nextCal.get(Calendar.YEAR)
			val month = nextCal.get(Calendar.MONTH)
			val day = nextCal.get(Calendar.DAY_OF_MONTH)

			// Set the new date
			date = "$year-${month+1}-$day"

			// Re-add the days to the days to run before starting. This way,
			// when the date is run and the next alarm is recalculated, the
			// repeat frequency is not added to the calendar
			repeatFrequencyDaysToRunBeforeStarting = days
		}
	}

	/**
	 * Add the repeat frequency to the alarm time, if it should be added.
	 */
	fun addRepeatFrequencyToTime()
	{
		// Alarm will not repeat or weekly repeat frequency
		if (!shouldRepeat || ((repeatFrequencyUnits == 4) && (repeatFrequency == 1)))
		{
			return
		}

		// Create a calendar from the alarm
		val alarmCal = NacCalendar.alarmToCalendar(this)

		// Add the repeat frequency to the calendar
		alarmCal.add(repeatFrequencyUnits.toCalendarField(), repeatFrequency)

		// Check repeat frequency units
		when (repeatFrequencyUnits)
		{
			// Minutes (max: 4 hr)
			1 -> addRepeatFrequencyMinutesToTime(alarmCal)

			// Hours (max: 1 week)
			2 -> addRepeatFrequencyHoursToTime(alarmCal)

			// Days (max: 1 year)
			3 -> addRepeatFrequencyDaysToTime(alarmCal)

			// Weekly
			4 -> addRepeatFrequencyWeeksToTime()

			// Month
			5 -> addRepeatFrequencyMonthsToTime(alarmCal)
		}

	}

	/**
	 * Compare the next day this alarm will run with another alarm.
	 *
	 * @param alarm An alarm.
	 *
	 * @return A negative integer, zero, or a positive integer as this object is
	 * less than, equal to, or greater than the specified object.
	 */
	private fun compareDay(alarm: NacAlarm): Int
	{
		val cal = NacCalendar.getNextAlarmDay(this)
		val otherCal = NacCalendar.getNextAlarmDay(alarm)

		// Both alarms do not have next calendar days so they are technically equal
		return if ((cal == null) && (otherCal == null))
		{
			0
		}
		// This alarm does not have a next calendar day so prioritize the other alarm
		else if (cal == null)
		{
			1
		}
		// The other alarm does not have a next calendar day so prioritize this alarm
		else if (otherCal == null)
		{
			-1
		}
		// Do a normal comparison
		else
		{
			cal.compareTo(otherCal)
		}
	}

	/**
	 * Compare the in use value in this alarm with another alarm.
	 *
	 *
	 * At least one alarm should be in use, otherwise the comparison is
	 * meaningless.
	 *
	 * @param alarm An alarm.
	 *
	 * @return A negative integer, zero, or a positive integer as this object is
	 * less than, equal to, or greater than the specified object.
	 */
	private fun compareInUseValue(alarm: NacAlarm): Int
	{
		val value = computeInUseValue()
		val otherValue = alarm.computeInUseValue()

		return if (otherValue < 0)
		{
			-1
		} else if (value < 0)
		{
			1
		} else if (value == otherValue)
		{
			0
		} else
		{
			if (value < otherValue) -1 else 1
		}
	}

	/**
	 * Compare the time of this alarm with another alarm.
	 *
	 * @param alarm An alarm.
	 *
	 * @return A negative integer, zero, or a positive integer as this object is
	 * less than, equal to, or greater than the specified object.
	 */
	private fun compareTime(alarm: NacAlarm): Int
	{
		val locale = Locale.getDefault()
		val format = "%1$02d:%2$02d"
		val time = String.format(locale, format, hour, minute)
		val otherTime = String.format(locale, format, alarm.hour, alarm.minute)

		return time.compareTo(otherTime)
	}

	/**
	 * Compare this alarm with another alarm.
	 *
	 * @param other An alarm.
	 *
	 * @return A negative integer, zero, or a positive integer as this object is
	 * less than, equal to, or greater than the specified object.
	 */
	override fun compareTo(other: NacAlarm): Int
	{
		return if (this == other)
		{
			0
		} else if (isInUse || other.isInUse)
		{
			val value = compareInUseValue(other)
			if (value == 0)
			{
				compareTime(other)
			} else
			{
				value
			}
		} else if (isEnabled xor other.isEnabled)
		{
			if (isEnabled) -1 else 1
		} else
		{
			compareDay(other)
		}
	}

	/**
	 * Get value corresponding to how in use an alarm is. This is used as a
	 * means to easily compare two alarms that are both in use.
	 *
	 * If an alarm is NOT IN USE, return -1.
	 *
	 * If an alarm is ACTIVE AND NOT SNOOZED, return 0.
	 *
	 * If an alarm is ACTIVE AND SNOOZED, return snooze count.
	 *
	 * If an alarm is NOT ACTIVE AND SNOOZED, return 1000 * snooze count.
	 *
	 * @return A value corresponding to how in use an alarm is.
	 */
	private fun computeInUseValue(): Int
	{
		// Check if alarm is not in use
		if (!isInUse)
		{
			return -1
		}

		// Scale the in use value
		val scale = if (isActive) 1 else 1000

		// Compute value by how many times the alarm has been snoozed
		return scale * snoozeCount
	}

	/**
	 * Create a copy of this alarm.
	 *
	 * The ID of the new alarm will be set to 0.
	 *
	 * @return A copy of this alarm.
	 */
	open fun copy(): NacAlarm
	{
		val alarm = build()

		// ID
		alarm.id = 0

		// Normal stuff
		alarm.isEnabled = isEnabled
		alarm.hour = hour
		alarm.minute = minute
		alarm.days = days
		alarm.date = date

		// Repeat
		alarm.shouldRepeat = shouldRepeat
		alarm.repeatFrequency = repeatFrequency
		alarm.repeatFrequencyUnits = repeatFrequencyUnits
		alarm.repeatFrequencyDaysToRunBeforeStarting = repeatFrequencyDaysToRunBeforeStarting

		// Vibrate
		alarm.shouldVibrate = shouldVibrate
		alarm.vibrateDuration = vibrateDuration
		alarm.vibrateWaitTime = vibrateWaitTime
		alarm.shouldVibratePattern = shouldVibratePattern
		alarm.vibrateRepeatPattern = vibrateRepeatPattern
		alarm.vibrateWaitTimeAfterPattern = vibrateWaitTimeAfterPattern

		// NFC
		alarm.shouldUseNfc = shouldUseNfc
		alarm.nfcTagId = nfcTagId
		alarm.shouldUseNfcTagDismissOrder = shouldUseNfcTagDismissOrder
		alarm.nfcTagDismissOrder = nfcTagDismissOrder

		// Flashlight
		alarm.shouldUseFlashlight = shouldUseFlashlight
		alarm.flashlightStrengthLevel = flashlightStrengthLevel
		alarm.graduallyIncreaseFlashlightStrengthLevelWaitTime = graduallyIncreaseFlashlightStrengthLevelWaitTime
		alarm.shouldBlinkFlashlight = shouldBlinkFlashlight
		alarm.flashlightOnDuration = flashlightOnDuration
		alarm.flashlightOffDuration = flashlightOffDuration

		// Media
		alarm.mediaPath = mediaPath
		alarm.mediaArtist = mediaArtist
		alarm.mediaTitle = mediaTitle
		alarm.mediaType = mediaType
		alarm.localMediaPath = localMediaPath
		alarm.shouldShuffleMedia = shouldShuffleMedia
		alarm.shouldRecursivelyPlayMedia = shouldRecursivelyPlayMedia

		// Volume and audio source
		alarm.volume = volume
		alarm.shouldGraduallyIncreaseVolume = shouldGraduallyIncreaseVolume
		alarm.graduallyIncreaseVolumeWaitTime = graduallyIncreaseVolumeWaitTime
		alarm.shouldRestrictVolume = shouldRestrictVolume
		alarm.audioSource = audioSource

		// Name
		alarm.name = name

		// Text-to-speech
		alarm.shouldSayCurrentTime = shouldSayCurrentTime
		alarm.shouldSayName = shouldSayName
		alarm.ttsFrequency = ttsFrequency
		alarm.ttsSpeechRate = ttsSpeechRate
		alarm.ttsVoice = ttsVoice

		// Dismiss
		alarm.shouldAutoDismiss = shouldAutoDismiss
		alarm.autoDismissTime = autoDismissTime
		alarm.canDismissEarly = canDismissEarly
		alarm.dismissEarlyTime = dismissEarlyTime
		alarm.timeOfDismissEarlyAlarm = timeOfDismissEarlyAlarm
		alarm.shouldShowDismissEarlyNotification = shouldShowDismissEarlyNotification
		alarm.shouldDeleteAfterDismissed = shouldDeleteAfterDismissed

		// Snooze
		alarm.shouldAutoSnooze = shouldAutoSnooze
		alarm.autoSnoozeTime = autoSnoozeTime
		alarm.maxSnooze = maxSnooze
		alarm.snoozeDuration = snoozeDuration
		alarm.shouldEasySnooze = shouldEasySnooze
		alarm.shouldVolumeSnooze = shouldVolumeSnooze

		// Reminder
		alarm.shouldShowReminder = shouldShowReminder
		alarm.timeToShowReminder = timeToShowReminder
		alarm.reminderFrequency = reminderFrequency
		alarm.shouldUseTtsForReminder = shouldUseTtsForReminder

		// Skip next alarm
		alarm.shouldSkipNextAlarm = shouldSkipNextAlarm

		return alarm
	}

	/**
	 * Describe contents (required for Parcelable).
	 */
	override fun describeContents(): Int
	{
		return 0
	}

	/**
	 * Dismiss an alarm.
	 *
	 * This will not update the database, or schedule the next alarm. That
	 * still needs to be done after calling this method.
	 */
	fun dismiss()
	{
		isActive = false
		shouldSkipNextAlarm = false
		timeActive = 0
		snoozeCount = 0
		timeOfDismissEarlyAlarm = 0

		// Date is set
		if (date.isNotEmpty())
		{
			// Clear the date
			date = ""
		}

		// Alarm will repeat
		if (shouldRepeat)
		{
			// Repeat frequency starting days needs to be cleaned up
			if ((repeatFrequencyUnits == 4) && (repeatFrequency > 1) && repeatFrequencyDaysToRunBeforeStarting.isNotEmpty())
			{
				// Remove today if it is in the days
				repeatFrequencyDaysToRunBeforeStarting.removeToday()
			}

			// Add the repeat frequency to the alarm time, if it should be added
			addRepeatFrequencyToTime()
		}
		// Alarm will not repeat
		else
		{
			// Toggle the alarm
			toggleAlarm()
		}
	}

	/**
	 * Dismiss an alarm early.
	 */
	fun dismissEarly()
	{
		// Do not dismiss early if the next alarm will be skipped
		if (shouldSkipNextAlarm)
		{
			return
		}

		// Alarm will repeat
		val time = if (shouldRepeat)
		{
			// Get the next alarm time
			val cal = NacCalendar.getNextAlarmDay(this)!!
			cal.timeInMillis
		}
		// Alarm will not repeat. No need to get the next alarm time
		else
		{
			null
		}

		// Dismiss the alarm
		dismiss()

		// Set the dismiss early time if it was set
		timeOfDismissEarlyAlarm = time ?: return
	}

	/**
	 * Check if this alarm equals another item.
	 *
	 * @param other An item to compare.
	 *
	 * @return True if both alarms are the same, and false otherwise.
	 */
	override fun equals(other: Any?): Boolean
	{
		return (other != null)
			&& (other is NacAlarm)
			&& (this.equalsId(other))
			&& (isActive == other.isActive)
			&& (timeActive == other.timeActive)
			&& (snoozeCount == other.snoozeCount)
			&& (localMediaPath == other.localMediaPath)
			&& (timeOfDismissEarlyAlarm == other.timeOfDismissEarlyAlarm)
			&& fuzzyEquals(other)
	}

	/**
	 * Check if this alarm has the same ID as another alarm.
	 *
	 * @param alarm An alarm.
	 *
	 * @return True if both alarms are the same, and false otherwise.
	 */
	fun equalsId(alarm: NacAlarm): Boolean
	{
		return id == alarm.id
	}

	/**
	 * Fuzzy equals to compare most of the important alarm attributes, but not all.
	 */
	fun fuzzyEquals(alarm: NacAlarm): Boolean
	{
		return (isEnabled == alarm.isEnabled)
			&& (hour == alarm.hour)
			&& (minute == alarm.minute)
			&& (days == alarm.days)
			&& (date == alarm.date)
			&& (shouldRepeat == alarm.shouldRepeat)
			&& (repeatFrequency == alarm.repeatFrequency)
			&& (repeatFrequencyUnits == alarm.repeatFrequencyUnits)
			&& (repeatFrequencyDaysToRunBeforeStarting == alarm.repeatFrequencyDaysToRunBeforeStarting)
			&& (shouldVibrate == alarm.shouldVibrate)
			&& (vibrateDuration == alarm.vibrateDuration)
			&& (vibrateWaitTime == alarm.vibrateWaitTime)
			&& (shouldVibratePattern == alarm.shouldVibratePattern)
			&& (vibrateRepeatPattern == alarm.vibrateRepeatPattern)
			&& (vibrateWaitTimeAfterPattern == alarm.vibrateWaitTimeAfterPattern)
			&& (shouldUseNfc == alarm.shouldUseNfc)
			&& (nfcTagId == alarm.nfcTagId)
			&& (shouldUseNfcTagDismissOrder == alarm.shouldUseNfcTagDismissOrder)
			&& (nfcTagDismissOrder == alarm.nfcTagDismissOrder)
			&& (shouldUseFlashlight == alarm.shouldUseFlashlight)
			&& (flashlightStrengthLevel == alarm.flashlightStrengthLevel)
			&& (graduallyIncreaseFlashlightStrengthLevelWaitTime == alarm.graduallyIncreaseFlashlightStrengthLevelWaitTime)
			&& (shouldBlinkFlashlight == alarm.shouldBlinkFlashlight)
			&& (flashlightOnDuration == alarm.flashlightOnDuration)
			&& (flashlightOffDuration == alarm.flashlightOffDuration)
			&& (mediaPath == alarm.mediaPath)
			&& (mediaArtist == alarm.mediaArtist)
			&& (mediaTitle == alarm.mediaTitle)
			&& (mediaType == alarm.mediaType)
			&& (shouldShuffleMedia == alarm.shouldShuffleMedia)
			&& (shouldRecursivelyPlayMedia == alarm.shouldRecursivelyPlayMedia)
			&& (volume == alarm.volume)
			&& (audioSource == alarm.audioSource)
			&& (name == alarm.name)
			&& (shouldSayCurrentTime == alarm.shouldSayCurrentTime)
			&& (shouldSayName == alarm.shouldSayName)
			&& (ttsFrequency == alarm.ttsFrequency)
			&& (ttsSpeechRate == alarm.ttsSpeechRate)
			&& (ttsVoice == alarm.ttsVoice)
			&& (shouldGraduallyIncreaseVolume == alarm.shouldGraduallyIncreaseVolume)
			&& (graduallyIncreaseVolumeWaitTime == alarm.graduallyIncreaseVolumeWaitTime)
			&& (shouldRestrictVolume == alarm.shouldRestrictVolume)
			&& (shouldAutoDismiss == alarm.shouldAutoDismiss)
			&& (autoDismissTime == alarm.autoDismissTime)
			&& (canDismissEarly == alarm.canDismissEarly)
			&& (dismissEarlyTime == alarm.dismissEarlyTime)
			&& (shouldShowDismissEarlyNotification == alarm.shouldShowDismissEarlyNotification)
			&& (shouldDeleteAfterDismissed == alarm.shouldDeleteAfterDismissed)
			&& (shouldAutoSnooze == alarm.shouldAutoSnooze)
			&& (autoSnoozeTime == alarm.autoSnoozeTime)
			&& (maxSnooze == alarm.maxSnooze)
			&& (snoozeDuration == alarm.snoozeDuration)
			&& (shouldEasySnooze == alarm.shouldEasySnooze)
			&& (shouldVolumeSnooze == alarm.shouldVolumeSnooze)
			&& (shouldShowReminder == alarm.shouldShowReminder)
			&& (timeToShowReminder == alarm.timeToShowReminder)
			&& (reminderFrequency == alarm.reminderFrequency)
			&& (shouldUseTtsForReminder == alarm.shouldUseTtsForReminder)
			&& (shouldSkipNextAlarm == alarm.shouldSkipNextAlarm)
	}

	/**
	 * @return The time string.
	 */
	fun getClockTime(context: Context): String
	{
		return NacCalendar.getClockTime(context, hour, minute)
	}

	/**
	 * Get the meridian (AM or PM).
	 *
	 * @return The meridian (AM or PM).
	 */
	fun getMeridian(context: Context): String
	{
		return NacCalendar.getMeridian(context, hour)
	}

	/**
	 * @see .getNameNormalized
	 */
	fun getNameNormalizedForMessage(max: Int): String
	{
		val name = nameNormalized

		return if (name.length > max)
		{
			"${name.substring(0, max-3)}..."
		}
		else
		{
			name
		}
	}

	/**
	 * Compute hash code of object.
	 */
	override fun hashCode(): Int
	{
		return 31 * (
			+ id.hashCode()
			+ isActive.hashCode()
			+ timeActive.hashCode()
			+ snoozeCount
			+ isEnabled.hashCode()
			+ hour
			+ minute
			+ shouldRepeat.hashCode()
			+ repeatFrequency
			+ repeatFrequencyUnits
			+ shouldVibrate.hashCode()
			+ vibrateDuration.hashCode()
			+ vibrateWaitTime.hashCode()
			+ shouldVibratePattern.hashCode()
			+ vibrateRepeatPattern
			+ vibrateWaitTimeAfterPattern.hashCode()
			+ shouldUseNfc.hashCode()
			+ shouldUseNfcTagDismissOrder.hashCode()
			+ nfcTagDismissOrder
			+ shouldUseFlashlight.hashCode()
			+ flashlightStrengthLevel
			+ graduallyIncreaseFlashlightStrengthLevelWaitTime
			+ shouldBlinkFlashlight.hashCode()
			+ mediaType
			+ shouldShuffleMedia.hashCode()
			+ shouldRecursivelyPlayMedia.hashCode()
			+ volume
			+ shouldSayCurrentTime.hashCode()
			+ shouldSayName.hashCode()
			+ ttsFrequency
			+ ttsSpeechRate.hashCode()
			+ shouldGraduallyIncreaseVolume.hashCode()
			+ graduallyIncreaseVolumeWaitTime
			+ shouldRestrictVolume.hashCode()
			+ shouldAutoDismiss.hashCode()
			+ autoDismissTime
			+ canDismissEarly.hashCode()
			+ dismissEarlyTime
			+ timeOfDismissEarlyAlarm.hashCode()
			+ shouldShowDismissEarlyNotification.hashCode()
			+ shouldDeleteAfterDismissed.hashCode()
			+ shouldAutoSnooze.hashCode()
			+ autoSnoozeTime
			+ maxSnooze
			+ snoozeDuration
			+ shouldEasySnooze.hashCode()
			+ shouldVolumeSnooze.hashCode()
			+ shouldShowReminder.hashCode()
			+ timeToShowReminder
			+ reminderFrequency
			+ shouldUseTtsForReminder.hashCode()
			+ shouldSkipNextAlarm.hashCode()
			+ days.hashCode()
			+ date.hashCode()
			+ repeatFrequencyDaysToRunBeforeStarting.hashCode()
			+ nfcTagId.hashCode()
			+ flashlightOnDuration.hashCode()
			+ flashlightOffDuration.hashCode()
			+ mediaPath.hashCode()
			+ mediaArtist.hashCode()
			+ mediaTitle.hashCode()
			+ localMediaPath.hashCode()
			+ audioSource.hashCode()
			+ name.hashCode()
			+ ttsVoice.hashCode()
			+ hasMedia.hashCode()
			+ canSnooze.hashCode()
			+ isSnoozed.hashCode()
			+ isInUse.hashCode()
			+ isNextSkippedAndFinal.hashCode()
			+ shouldUseTts.hashCode()
			+ nameNormalized.hashCode()
			+ nfcTagIdList.hashCode())
	}

	/**
	 * Increment the snooze count by 1.
	 */
	private fun incrementSnoozeCount()
	{
		snoozeCount += 1
	}

	/**
	 * Print all values in the alarm object.
	 */
	@Suppress("unused")
	open fun print()
	{
		println("Alarm Information")
		println("Id                    : $id")
		println("Is Active             : $isActive")
		println("Time Active           : $timeActive")
		println("Snooze Count          : $snoozeCount")
		println("Is Enabled            : $isEnabled")
		println("Hour                  : $hour")
		println("Minute                : $minute")
		println("Days                  : $days")
		println("Date                  : $date")
		println("Repeat                : $shouldRepeat")
		println("Repeat Freq           : $repeatFrequency")
		println("Repeat Freq Units     : $repeatFrequencyUnits")
		println("Repeat Freq Days 2 Run: $repeatFrequencyDaysToRunBeforeStarting")
		println("Vibrate               : $shouldVibrate")
		println("Vibrate duration      : $vibrateDuration")
		println("Vibrate wait time     : $vibrateWaitTime")
		println("Vibrate pattern       : $shouldVibratePattern")
		println("Vibrate repeat pat.   : $vibrateRepeatPattern")
		println("Vibrate wait after    : $vibrateWaitTimeAfterPattern")
		println("Use NFC               : $shouldUseNfc")
		println("Nfc Tag Id            : $nfcTagId")
		println("Should NFC Dismiss Ord: $shouldUseNfcTagDismissOrder")
		println("Nfc Tag Dismiss Order : $nfcTagDismissOrder")
		println("Use Flashlight        : $shouldUseFlashlight")
		println("Flashlight Strength   : $flashlightStrengthLevel")
		println("Grad Inc Flash        : $graduallyIncreaseFlashlightStrengthLevelWaitTime")
		println("Should Blink Flash    : $shouldBlinkFlashlight")
		println("Flashlight On         : $flashlightOnDuration")
		println("Flashlight Off        : $flashlightOffDuration")
		println("Media Path            : $mediaPath")
		println("Media Artist          : $mediaArtist")
		println("Media Name            : $mediaTitle")
		println("Media Type            : $mediaType")
		println("Local media Path      : $localMediaPath")
		println("Shuffle media         : $shouldShuffleMedia")
		println("Recusively Play       : $shouldRecursivelyPlayMedia")
		println("Volume                : $volume")
		println("Audio Source          : $audioSource")
		println("Name                  : $name")
		println("Tts say time          : $shouldSayCurrentTime")
		println("Tts say name          : $shouldSayName")
		println("Tts Freq              : $ttsFrequency")
		println("Tts Speech Rate       : $ttsSpeechRate")
		println("Tts Voice             : $ttsVoice")
		println("Grad Inc Vol          : $shouldGraduallyIncreaseVolume")
		println("Grad Inc Vol Wait T   : $graduallyIncreaseVolumeWaitTime")
		println("Restrict Vol          : $shouldRestrictVolume")
		println("Should auto dismiss   : $shouldAutoDismiss")
		println("Auto Dismiss          : $autoDismissTime")
		println("Use Dismiss Early     : $canDismissEarly")
		println("Dismiss Early         : $dismissEarlyTime")
		println("Time of Early Alarm   : $timeOfDismissEarlyAlarm")
		println("Should Dismiss Early N: $shouldShowDismissEarlyNotification")
		println("Should delete after   : $shouldDeleteAfterDismissed")
		println("Should auto snooze    : $shouldAutoSnooze")
		println("Auto Snooze           : $autoSnoozeTime")
		println("Max Snooze            : $maxSnooze")
		println("Snooze Duration       : $snoozeDuration")
		println("Should Easy Snooze    : $shouldEasySnooze")
		println("Should Volume Snooze  : $shouldVolumeSnooze")
		println("Show Reminder         : $shouldShowReminder")
		println("Time to show remind   : $timeToShowReminder")
		println("Reminder freq         : $reminderFrequency")
		println("Use Tts 4 Reminder    : $shouldUseTtsForReminder")
		println("Should skip next      : $shouldSkipNextAlarm")
	}

	/**
	 * @see .setDays
	 */
	fun setDays(value: Int)
	{
		days = value.toDays()
	}

	/**
	 * Skip an alarm.
	 */
	fun skipAlarm()
	{
		// Check if the alarm does not need to be skipped
		if (!shouldSkipNextAlarm)
		{
			return
		}

		// Get the next time this alarm will run
		val nextCal = NacCalendar.getNextAlarmDay(this)

		// There is no next alarm
		if (nextCal == null)
		{
			// Disable the alarm since there is no next time this will run
			isEnabled = false
		}
		// Normal dismiss logic of an alarm
		else
		{
			dismiss()
		}
	}

	/**
	 * Snooze the alarm.
	 *
	 * @return Calendar instance of when the snoozed alarm will go off.
	 */
	fun snooze(): Calendar
	{
		// Reset the active flag
		isActive = false

		// Add the snooze duration value to the current time
		val cal = Calendar.getInstance()
		cal.add(Calendar.SECOND, snoozeDuration)

		// Increment the snooze count. The "isSnoozed" variable checks the
		// snooze count so this basically makes "isSnoozed" true
		incrementSnoozeCount()

		return cal
	}

	/**
	 * Toast the flashlight message.
	 */
	fun toastFlashlight(context: Context)
	{
		// Determine which message to show
		val messageId = if (shouldUseFlashlight)
		{
			R.string.message_flashlight_enabled
		}
		else
		{
			R.string.message_flashlight_disabled
		}

		// Toast the message
		quickToast(context, messageId)
	}

	/**
	 * Toast the NFC message.
	 */
	fun toastNfc(context: Context, allNfcTags: List<NacNfcTag>)
	{
		// Determine which message to show
		val message = if (shouldUseNfc)
		{
			// NFC ID
			if (nfcTagId.isNotEmpty())
			{
				// Find a matching NFC tag
				val tag = allNfcTags.firstOrNull { it.nfcId == nfcTagId }

				// Saved and named
				if (tag != null)
				{
					context.getString(R.string.message_nfc_required_saved, tag.name)
				}
				// Unsaved and no name
				else
				{
					context.getString(R.string.message_nfc_required_unsaved)
				}
			}
			// Use any
			else
			{
				context.getString(R.string.message_nfc_required_use_any)
			}
		}
		// Normal
		else
		{
			context.getString(R.string.message_nfc_optional)
		}

		// Toast the message
		quickToast(context, message)
	}

	/**
	 * Toast the NFC ID message.
	 */
	fun toastNfcId(context: Context)
	{
		// Determine which message to show
		val message = if (nfcTagId.isNotEmpty())
		{
			// Get the string to show a specific NFC tag
			val nfcId = context.getString(R.string.message_show_nfc_tag_id)

			"$nfcId: $nfcTagId"
		}
		else
		{
			// Get the string to show any NFC tag
			context.getString(R.string.message_nfc_required_use_any)
		}

		// Toast the message
		quickToast(context, message)
	}

	/**
	 * Toast the repeat message.
	 */
	open fun toastRepeat(context: Context)
	{
		// Determine which message to show
		val messageId = if (shouldRepeat)
		{
			R.string.message_alarm_repeat_enabled
		}
		else
		{
			R.string.message_alarm_repeat_disabled
		}

		// Toast the message
		quickToast(context, messageId)
	}

	/**
	 * Toast the vibrate message.
	 */
	fun toastVibrate(context: Context)
	{
		// Determine which message to show
		val messageId = if (shouldVibrate)
		{
			R.string.message_vibrate_enabled
		}
		else
		{
			R.string.message_vibrate_disabled
		}


		// Toast the message
		quickToast(context, messageId)
	}

	/**
	 * Toggle the the current day/enabled attribute of the alarm.
	 *
	 * An alarm can only be toggled if repeat is not enabled.
	 */
	private fun toggleAlarm()
	{
		// Check if the alarm should be repeated
		if (shouldRepeat)
		{
			return
		}

		// Check if there are any days selected
		if (days.isNotEmpty())
		{
			toggleToday()
		}

		// Check if days are NOT selected because if the toggle deselected the last day
		// then the alarm should be disabled.
		//
		// If it did not deselect the last day, then there is no harm in checking again
		// (this use to be the "else" part of the above "if") to disable the alarm
		if (days.isEmpty())
		{
			isEnabled = false
		}
	}

	/**
	 * Toggle a day.
	 */
	fun toggleDay(day: Day)
	{
		// Check if day is contained in the list of alarm days
		if (days.contains(day))
		{
			days.remove(day)
		}
		// Day is not present in alarm days
		else
		{
			days.add(day)
		}
	}

	/**
	 * Toggle repeat.
	 */
	fun toggleRepeat()
	{
		shouldRepeat = !shouldRepeat
	}

	/**
	 * Toggle today.
	 */
	fun toggleToday()
	{
		// Get today's day
		val day = Day.TODAY

		// Toggle today
		toggleDay(day)
	}

	/**
	 * Toggle use the flashlight.
	 */
	fun toggleUseFlashlight()
	{
		shouldUseFlashlight = !shouldUseFlashlight
	}

	/**
	 * Toggle use NFC.
	 */
	fun toggleUseNfc()
	{
		shouldUseNfc = !shouldUseNfc
	}

	/**
	 * Toggle vibrate.
	 */
	fun toggleVibrate()
	{
		shouldVibrate = !shouldVibrate
	}

	/** Whether or not the alarm will alarm soon.
	 * <p>
	 * "Soon" is determined by the dismiss early time. If it will alarm within
	 * that time, then it is soon.
	 */
	fun willAlarmSoon(): Boolean
	{
		// Alarm is disabled or unable to use dismiss early
		if (!isEnabled || !canDismissEarly || (dismissEarlyTime == 0) || shouldSkipNextAlarm)
		{
			return false
		}

		// Determine the difference in time between the next alarm and today
		val today = Calendar.getInstance()
		val cal = NacCalendar.getNextAlarmDay(this)!!
		val diff = (cal.timeInMillis - today.timeInMillis) / 1000 / 60

		// Compare the two amounts of time
		return (diff < dismissEarlyTime)
	}

	/**
	 * Write data into parcel (required for Parcelable).
	 *
	 * Update this when adding/removing an element.
	 */
	override fun writeToParcel(output: Parcel, flags: Int)
	{
		// ID
		output.writeLong(id)

		// Active alarm flags
		output.writeInt(if (isActive) 1 else 0)
		output.writeLong(timeActive)
		output.writeInt(snoozeCount)

		// Normal stuff
		output.writeInt(if (isEnabled) 1 else 0)
		output.writeInt(hour)
		output.writeInt(minute)
		output.writeInt(days.daysToValue())
		output.writeString(date)

		// Repeat
		output.writeInt(if (shouldRepeat) 1 else 0)
		output.writeInt(repeatFrequency)
		output.writeInt(repeatFrequencyUnits)
		output.writeInt(repeatFrequencyDaysToRunBeforeStarting.daysToValue())

		// Vibrate
		output.writeInt(if (shouldVibrate) 1 else 0)
		output.writeLong(vibrateDuration)
		output.writeLong(vibrateWaitTime)
		output.writeInt(if (shouldVibratePattern) 1 else 0)
		output.writeInt(vibrateRepeatPattern)
		output.writeLong(vibrateWaitTimeAfterPattern)

		// NFC
		output.writeInt(if (shouldUseNfc) 1 else 0)
		output.writeString(nfcTagId)
		output.writeInt(if (shouldUseNfcTagDismissOrder) 1 else 0)
		output.writeInt(nfcTagDismissOrder)

		// Flashlight
		output.writeInt(if (shouldUseFlashlight) 1 else 0)
		output.writeInt(flashlightStrengthLevel)
		output.writeInt(graduallyIncreaseFlashlightStrengthLevelWaitTime)
		output.writeInt(if (shouldBlinkFlashlight) 1 else 0)
		output.writeString(flashlightOnDuration)
		output.writeString(flashlightOffDuration)

		// Media
		output.writeString(mediaPath)
		output.writeString(mediaArtist)
		output.writeString(mediaTitle)
		output.writeInt(mediaType)
		output.writeString(localMediaPath)
		output.writeInt(if (shouldShuffleMedia) 1 else 0)
		output.writeInt(if (shouldRecursivelyPlayMedia) 1 else 0)

		// Volume and audio source
		output.writeInt(volume)
		output.writeInt(if (shouldGraduallyIncreaseVolume) 1 else 0)
		output.writeInt(graduallyIncreaseVolumeWaitTime)
		output.writeInt(if (shouldRestrictVolume) 1 else 0)
		output.writeString(audioSource)

		// Name
		output.writeString(name)

		// Text-to-speech
		output.writeInt(if (shouldSayCurrentTime) 1 else 0)
		output.writeInt(if (shouldSayName) 1 else 0)
		output.writeInt(ttsFrequency)
		output.writeFloat(ttsSpeechRate)
		output.writeString(ttsVoice)

		// Dismiss
		output.writeInt(if (shouldAutoDismiss) 1 else 0)
		output.writeInt(autoDismissTime)
		output.writeInt(if (canDismissEarly) 1 else 0)
		output.writeInt(dismissEarlyTime)
		output.writeLong(timeOfDismissEarlyAlarm)
		output.writeInt(if (shouldShowDismissEarlyNotification) 1 else 0)
		output.writeInt(if (shouldDeleteAfterDismissed) 1 else 0)

		// Snooze
		output.writeInt(if (shouldAutoSnooze) 1 else 0)
		output.writeInt(autoSnoozeTime)
		output.writeInt(maxSnooze)
		output.writeInt(snoozeDuration)
		output.writeInt(if (shouldEasySnooze) 1 else 0)
		output.writeInt(if (shouldVolumeSnooze) 1 else 0)

		// Reminder
		output.writeInt(if (shouldShowReminder) 1 else 0)
		output.writeInt(timeToShowReminder)
		output.writeInt(reminderFrequency)
		output.writeInt(if (shouldUseTtsForReminder) 1 else 0)

		// Skip next alarm
		output.writeInt(if (shouldSkipNextAlarm) 1 else 0)
	}

	companion object
	{

		/**
		 * Generate parcel (required for Parcelable).
		 */
		@Suppress("unused")
		@JvmField
		val CREATOR: Parcelable.Creator<NacAlarm> = object : Parcelable.Creator<NacAlarm>
		{
			override fun createFromParcel(input: Parcel): NacAlarm
			{
				return NacAlarm(input)
			}

			override fun newArray(size: Int): Array<NacAlarm?>
			{
				return arrayOfNulls(size)
			}
		}

		/**
		 * Build an alarm.
		 */
		fun build(shared: NacSharedPreferences? = null): NacAlarm
		{
			// Create an alarm and get calendar instance
			val alarm = NacAlarm()
			val calendar = Calendar.getInstance()

			// Normal stuff
			alarm.isEnabled = true
			alarm.hour = calendar[Calendar.HOUR_OF_DAY]
			alarm.minute = calendar[Calendar.MINUTE]

			// Defaults that probably do not need it because they are already set this way
			alarm.timeOfDismissEarlyAlarm = 0
			alarm.shouldSkipNextAlarm = false

			// Unable to access defaults in shared preferences because null
			if (shared == null)
			{
				return alarm
			}

			// Days
			alarm.days = shared.days.toDays()

			// Repeat
			alarm.shouldRepeat = shared.shouldRepeat
			alarm.repeatFrequency = shared.repeatFrequency
			alarm.repeatFrequencyUnits = shared.repeatFrequencyUnits
			alarm.repeatFrequencyDaysToRunBeforeStarting = shared.repeatFrequencyDaysToRunBeforeStarting.toDays()

			// Vibrate
			alarm.shouldVibrate = shared.shouldVibrate
			alarm.vibrateDuration = shared.vibrateDuration
			alarm.vibrateWaitTime = shared.vibrateWaitTime
			alarm.shouldVibratePattern = shared.shouldVibratePattern
			alarm.vibrateRepeatPattern = shared.vibrateRepeatPattern
			alarm.vibrateWaitTimeAfterPattern = shared.vibrateWaitTimeAfterPattern

			// NFC
			alarm.shouldUseNfc = shared.shouldUseNfc
			alarm.nfcTagId = shared.nfcTagId
			alarm.shouldUseNfcTagDismissOrder = shared.shouldUseNfcTagDismissOrder
			alarm.nfcTagDismissOrder = shared.nfcTagDismissOrder

			// Flashlight
			alarm.shouldUseFlashlight = shared.shouldUseFlashlight
			alarm.flashlightStrengthLevel = shared.flashlightStrengthLevel
			alarm.graduallyIncreaseFlashlightStrengthLevelWaitTime = shared.graduallyIncreaseFlashlightStrengthLevelWaitTime
			alarm.shouldBlinkFlashlight = shared.shouldBlinkFlashlight
			alarm.flashlightOnDuration = shared.flashlightOnDuration
			alarm.flashlightOffDuration = shared.flashlightOffDuration

			// Media
			alarm.mediaPath = shared.mediaPath
			alarm.mediaArtist = shared.mediaArtist
			alarm.mediaTitle = shared.mediaTitle
			alarm.mediaType = shared.mediaType
			alarm.localMediaPath = shared.localMediaPath
			alarm.shouldShuffleMedia = shared.shouldShuffleMedia
			alarm.shouldRecursivelyPlayMedia = shared.recursivelyPlayMedia

			// Volume and audio source
			alarm.volume = shared.volume
			alarm.shouldGraduallyIncreaseVolume = shared.shouldGraduallyIncreaseVolume
			alarm.graduallyIncreaseVolumeWaitTime = shared.graduallyIncreaseVolumeWaitTime
			alarm.shouldRestrictVolume = shared.shouldRestrictVolume
			alarm.audioSource = shared.audioSource

			// Name
			alarm.name = shared.name

			// Text-to-speech
			alarm.shouldSayCurrentTime = shared.shouldSayCurrentTime
			alarm.shouldSayName = shared.shouldSayAlarmName
			alarm.ttsFrequency = shared.ttsFrequency
			alarm.ttsSpeechRate = shared.ttsSpeechRate
			alarm.ttsVoice = shared.ttsVoice

			// Dismiss
			alarm.shouldAutoDismiss = shared.shouldAutoDismiss
			alarm.autoDismissTime = shared.autoDismissTime
			alarm.canDismissEarly = shared.canDismissEarly
			alarm.dismissEarlyTime = shared.dismissEarlyTime
			alarm.shouldShowDismissEarlyNotification = shared.shouldShowDismissEarlyNotification
			alarm.shouldDeleteAfterDismissed = shared.shouldDeleteAfterDismissed

			// Snooze
			alarm.shouldAutoSnooze = shared.shouldAutoSnooze
			alarm.autoSnoozeTime = shared.autoSnoozeTime
			alarm.maxSnooze = shared.maxSnooze
			alarm.snoozeDuration = shared.snoozeDuration
			alarm.shouldEasySnooze = shared.shouldEasySnooze
			alarm.shouldVolumeSnooze = shared.shouldVolumeSnooze

			// Reminder
			alarm.shouldShowReminder = shared.shouldShowReminder
			alarm.timeToShowReminder = shared.timeToShowReminder
			alarm.reminderFrequency = shared.reminderFrequency
			alarm.shouldUseTtsForReminder = shared.shouldUseTtsForReminder

			return alarm
		}

		/**
		 * Calculate the auto dismiss time from a minutes index.
		 */
		fun calcAutoDismissFromMinutesIndex(index: Int): Int
		{
			return index * 60
		}

		/**
		 * Calculate the auto dismiss time from a seconds index.
		 */
		fun calcAutoDismissFromSecondsIndex(index: Int): Int
		{
			return index
		}

		/**
		 * Calculate the auto dismiss index from a value.
		 */
		fun calcAutoDismissIndex(time: Int): Pair<Int, Int>
		{
			// Get the minute and seconds components from the time
			var (minutes, seconds) = calcMinutesAndSecondsFromTime(time)

			// Check if minutes and seconds are both 0. This is no good, and so should
			// default minutes to a better index
			if ((minutes == 0) && (seconds == 0))
			{
				minutes = 15
			}

			// Check if the number is way too big
			if (minutes > 60)
			{
				minutes /= 60
			}

			return Pair(minutes, seconds)
		}

		/**
		 * Calculate the auto snooze time from a minutes index.
		 */
		fun calcAutoSnoozeFromMinutesIndex(index: Int): Int
		{
			return index * 60
		}

		/**
		 * Calculate the auto snooze time from a seconds index.
		 */
		fun calcAutoSnoozeFromSecondsIndex(index: Int): Int
		{
			return index
		}

		/**
		 * Calculate the auto snooze index from a value.
		 */
		fun calcAutoSnoozeIndex(time: Int): Pair<Int, Int>
		{
			// Get the minute and seconds components from the time
			var (minutes, seconds) = calcMinutesAndSecondsFromTime(time)

			// Check if minutes and seconds are both 0. This is no good, and so should
			// default minutes to the 1st index
			if ((minutes == 0) && (seconds == 0))
			{
				minutes = 5
			}

			// Check if the number is way too big
			if (minutes > 30)
			{
				minutes /= 60
			}

			// Calculate the index
			return Pair(minutes, seconds)
		}

		/**
		 * Calculate the dismiss early index from a time.
		 */
		fun calcDismissEarlyIndex(time: Int): Int
		{
			return if (time == 0)
			{
				// Time should never be 0 min, but if it is, use the default of 30 min
				30 / 5 + 3
			}
			else if (time <= 5)
			{
				// 1 to 5 min
				time - 1
			}
			else
			{
				// 10+ min
				time / 5 + 3
			}
		}

		/**
		 * Calculate the dismiss early time from an index.
		 */
		fun calcDismissEarlyTime(index: Int): Int
		{
			return if (index < 5)
			{
				index + 1
			}
			else
			{
				(index - 3) * 5
			}
		}

		/**
		 * Calculate the flashlight on/off duration.
		 */
		fun calcFlashlightOnOffDuration(index: Int): String
		{
			return when (index)
			{
				0    -> "0.5"
				1    -> "1.0"
				2    -> "1.5"
				3    -> "2.0"
				4    -> "2.5"
				5    -> "3.0"
				6    -> "3.5"
				7    -> "4.0"
				8    -> "4.5"
				9    -> "5.0"
				10   -> "5.5"
				11   -> "6.0"
				12   -> "6.5"
				13   -> "7.0"
				14   -> "7.5"
				15   -> "8.0"
				16   -> "8.5"
				17   -> "9.0"
				18   -> "9.5"
				19   -> "10.0"
				else -> "1.0"
			}
		}

		/**
		 * Calculate the flashlight on/off duration index.
		 */
		fun calcFlashlightOnOffDurationIndex(duration: String): Int
		{
			return when (duration)
			{
				"0.5"  -> 0
				"1.0"  -> 1
				"1.5"  -> 2
				"2.0"  -> 3
				"2.5"  -> 4
				"3.0"  -> 5
				"3.5"  -> 6
				"4.0"  -> 7
				"4.5"  -> 8
				"5.0"  -> 9
				"5.5"  -> 10
				"6.0"  -> 11
				"6.5"  -> 12
				"7.0"  -> 13
				"7.5"  -> 14
				"8.0"  -> 15
				"8.5"  -> 16
				"9.0"  -> 17
				"9.5"  -> 18
				"10.0" -> 19
				else   -> 1
			}
		}

		/**
		 * Calculate the gradually increase volume index from a time.
		 */
		fun calcGraduallyIncreaseVolumeIndex(time: Int): Int
		{
			return time - 1
		}

		/**
		 * Calculate the gradually increase volume wait time from an index.
		 */
		fun calcGraduallyIncreaseVolumeWaitTime(index: Int): Int
		{
			return index + 1
		}

		/**
		 * Calculate the max snooze value from an index.
		 */
		fun calcMaxSnooze(index: Int): Int
		{
			return if (index == 11)
			{
				-1
			}
			else
			{
				index
			}
		}

		/**
		 * Calculate the max snooze index from a value.
		 */
		fun calcMaxSnoozeIndex(value: Int): Int
		{
			return if (value == -1)
			{
				11
			}
			else
			{
				value
			}
		}

		/**
		 * Calculate the minutes and seconds components from a time.
		 */
		private fun calcMinutesAndSecondsFromTime(time: Int): Pair<Int, Int>
		{
			val minutes = time / 60
			val seconds = time % 60

			return Pair(minutes, seconds)
		}

		/**
		 * Calculate the NFC tag dismiss order from an index.
		 */
		fun calcNfcTagDismissOrderFromIndex(index: Int): Int
		{
			return when (index)
			{
				0 ->
				{
					NacNfcTagDismissOrder.RANDOM
				}
				1 ->
				{
					NacNfcTagDismissOrder.SEQUENTIAL
				}
				else ->
				{
					NacNfcTagDismissOrder.SEQUENTIAL
				}
			}
		}

		/**
		 * Calculate the NFC tag dismiss order index.
		 */
		fun calcNfcTagDismissOrderIndex(dismissOrder: Int): Int
		{
			return when (dismissOrder)
			{
				NacNfcTagDismissOrder.RANDOM ->
				{
					0
				}
				NacNfcTagDismissOrder.SEQUENTIAL ->
				{
					1
				}
				else ->
				{
					1
				}
			}
		}

		/**
		 * Calculate the repeat frequency units from an index.
		 */
		fun calcRepeatFrequencyUnitsFromIndex(index: Int): Int
		{
			return when (index)
			{
				0 -> 1
				1 -> 2
				2 -> 3
				3 -> 4
				4 -> 5
				else -> 4
			}
		}

		/**
		 * Calculate the repeat frequency units index.
		 *
		 * 1 = Minutes
		 * 2 = Hours
		 * 3 = Days
		 * 4 = Weeks
		 * 5 = Months
		 */
		fun calcRepeatFrequencyUnitsIndex(units: Int): Int
		{
			return when (units)
			{
				1 -> 0
				2 -> 1
				3 -> 2
				4 -> 3
				5 -> 4
				else -> 3
			}
		}

		/**
		 * Calculate the repeat frequency index.
		 */
		fun calcRepeatFrequencyIndex(value: Int, units: Int): Int
		{
			// Determine the index
			val index = when (units)
			{
				1 -> value-15
				else -> value-1
			}

			// Make sure to return a proper index
			return if (index >= 0) index else 0
		}

		/**
		 * Calculate the snooze duration from a minutes index.
		 */
		fun calcSnoozeDurationFromMinutesIndex(index: Int): Int
		{
			return index * 60
		}

		/**
		 * Calculate the snooze duration time from a seconds index.
		 */
		fun calcSnoozeDurationFromSecondsIndex(index: Int): Int
		{
			return index
		}

		/**
		 * Calculate the snooze duration index from a value.
		 */
		fun calcSnoozeDurationIndex(value: Int): Pair<Int, Int>
		{
			// Get the minute and seconds components from the time
			var (minutes, seconds) = calcMinutesAndSecondsFromTime(value)

			// Check if minutes and seconds are both 0. This is no good, and so should
			// default minutes to the 1st index
			if ((minutes == 0) && (seconds == 0))
			{
				minutes = 5
			}

			// Check if the number is way too big
			if (minutes > 90)
			{
				minutes /= 60
			}

			// Calculate the index
			return Pair(minutes, seconds)
		}

		/**
		 * Calculate the upcoming reminder time to show index from a time.
		 */
		fun calcUpcomingReminderTimeToShowIndex(time: Int): Int
		{
			return if (time == 0)
			{
				4
			}
			else if (time <= 10)
			{
				time - 1
			}
			else
			{
				time/5 + 7
			}
		}

		/**
		 * Calculate the upcoming reminder time to show from an index.
		 */
		fun calcUpcomingReminderTimeToShow(index: Int): Int
		{
			return if (index < 10)
			{
				index + 1
			}
			else
			{
				(index-7) * 5
			}
		}

	}

}

/**
 * Hilt module to provide an instance of an alarm.
 */
@Suppress("unused")
@InstallIn(SingletonComponent::class)
@Module
class NacAlarmModule
{

	/**
	 * Provide an instance of an alarm.
	 */
	@Provides
	fun provideAlarm() : NacAlarm
	{
		return NacAlarm.build()
	}

}
