package com.nfcalarmclock.mediapicker.ringtone

import android.os.Bundle
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.widget.RadioButton
import android.widget.RadioGroup
import androidx.core.net.toUri
import androidx.media3.common.util.UnstableApi
import com.nfcalarmclock.R
import com.nfcalarmclock.alarm.db.NacAlarm
import com.nfcalarmclock.mediapicker.NacBaseChildMediaPickerFragment
import com.nfcalarmclock.shared.NacSharedPreferences
import com.nfcalarmclock.system.getDeviceProtectedStorageContext
import com.nfcalarmclock.system.media.NacMedia
import com.nfcalarmclock.system.media.buildLocalMediaPath
import com.nfcalarmclock.system.media.getMediaArtist
import com.nfcalarmclock.system.media.getMediaTitle
import com.nfcalarmclock.view.setupThemeColor

/**
 * Pick a ringtone.
 */
@UnstableApi
abstract class NacRingtonePickerFragment<T: NacAlarm>
	: NacBaseChildMediaPickerFragment<T>()
{

	/**
	 * Radio group.
	 */
	private var radioGroup: RadioGroup? = null

	/**
	 * Create a radio button.
	 */
	private fun createRadioButton(title: String, path: String): RadioButton
	{
		// Inflate the view
		val inflater = LayoutInflater.from(context)
		val view = inflater.inflate(R.layout.radio_button_ringtone, radioGroup, true)

		// Get the radio button
		val button = view.findViewById<RadioButton>(R.id.radio_button_ringtone)

		// Generate an ID for the radio button
		val id = View.generateViewId()

		// Setup the radio button
		button.id = id
		button.text = title
		button.tag = path

		// Set the on click listener
		button.setOnClickListener {

			// Get the URI
			val uri = path.toUri()

			// Play the media at the URI
			play(uri)

		}

		return button
	}

	/**
	 * Called when the Clear button is clicked.
	 */
	override fun onClearClicked()
	{
		// Super
		super.onClearClicked()

		// Clear the radio button that is checked
		radioGroup!!.clearCheck()
	}

	/**
	 * Called when the view will be created.
	 */
	override fun onCreateView(inflater: LayoutInflater, container: ViewGroup?,
							  savedInstanceState: Bundle?): View?
	{
		return inflater.inflate(R.layout.frg_ringtone, container, false)
	}

	/**
	 * Called when the Ok button is clicked.
	 */
	override fun onOkClicked()
	{
		// Get the activity and the device protected storage context
		val activity = requireActivity()
		val deviceContext = getDeviceProtectedStorageContext(activity)

		// Get the URI from the path
		val uri = mediaPath.toUri()

		// Set the media information
		mediaArtist = uri.getMediaArtist(deviceContext)
		mediaTitle = uri.getMediaTitle(deviceContext)
		mediaType = NacMedia.TYPE_RINGTONE
		localMediaPath = buildLocalMediaPath(deviceContext, mediaArtist, mediaTitle, mediaType)

		// Copy the file to device encrypted storage
		copyMediaToDeviceEncryptedStorage(deviceContext)

		// Super
		super.onOkClicked()
	}

	/**
	 * Called after the view is created.
	 */
	override fun onViewCreated(view: View, savedInstanceState: Bundle?)
	{
		// Set the radio group
		radioGroup = view.findViewById(R.id.radio_group)

		// Setup the action buttons
		setupActionButtons(view)

		// Setup the radio buttons
		setupRadioButtons()
	}

	/**
	 * Setup the radio buttons.
	 */
	private fun setupRadioButtons()
	{
		// Get all the ringtones
		val context = requireContext()
		val ringtones = NacMedia.getRingtones(context)
		val shared = NacSharedPreferences(context)

		// Iterate over each ringtone
		for ((title, path) in ringtones)
		{
			// Skip if path is empty
			if (path.isEmpty())
			{
				continue
			}

			// Create a button for each ringtone
			val button = createRadioButton(title, path)

			// Setup the radio button color
			button.setupThemeColor(shared)

			// Check if the paths match
			if (path == mediaPath)
			{
				// Set the radio button as selected
				button.isChecked = true
			}
		}
	}

}