package com.nfcalarmclock.system.permission

import android.annotation.SuppressLint
import android.content.Context
import android.os.Build
import androidx.appcompat.app.AppCompatActivity
import com.nfcalarmclock.shared.NacSharedPreferences
import com.nfcalarmclock.system.permission.NacPermissionRequestDialog.OnPermissionRequestListener
import com.nfcalarmclock.system.permission.postnotifications.NacPostNotificationsPermission
import com.nfcalarmclock.system.permission.postnotifications.NacPostNotificationsPermissionRequestDialog
import com.nfcalarmclock.system.permission.scheduleexactalarm.NacScheduleExactAlarmPermission
import com.nfcalarmclock.system.permission.scheduleexactalarm.NacScheduleExactAlarmPermissionRequestDialog
import com.nfcalarmclock.system.permission.systemalertwindow.NacSystemAlertWindowPermission
import com.nfcalarmclock.system.permission.systemalertwindow.NacSystemAlertWindowPermissionRequestDialog
import java.util.EnumSet

/**
 * Manage permission requests.
 */
class NacPermissionRequestManager(activity: AppCompatActivity)
{

	/**
	 * Enum of all permissions that could be requested.
	 */
	enum class Permission
	{
		//IGNORE_BATTERY_OPTIMIZATION,
		POST_NOTIFICATIONS,
		SCHEDULE_EXACT_ALARM,
		SYSTEM_ALERT_WINDOW
	}

	/**
	 * A set of all the permissions that need to be requested.
	 */
	private var permissionRequestSet: EnumSet<Permission> = EnumSet.noneOf(Permission::class.java)

	/**
	 * Current position number of the permission being requested.
	 */
	private var currentPosition = 0

	/**
	 * Total number of permissions that need to be requested.
	 */
	private var totalNumberOfPermissions = 0

	/**
	 * Check if analysis has been completed or not.
	 *
	 * This way it does not need to keep getting repeated if the app goes through
	 * its onStop/onResume lifecycle.
	 */
	private var isAnalyzed = false

	/**
	 * Constructor.
	 */
	init
	{
		analyze(activity)
	}

	/**
	 * Analyze the permissions that need to be requested.
	 */
	private fun analyze(context: Context)
	{
		val shared = NacSharedPreferences(context)
		val set = EnumSet.noneOf(Permission::class.java)

		// Check if analysis has already been completed or not
		if (isAnalyzed)
		{
			return
		}

		// Post notifications
		if (NacPostNotificationsPermission.shouldRequestPermission(context, shared))
		{
			set.add(Permission.POST_NOTIFICATIONS)
		}

		// Schedule exact alarms
		if (NacScheduleExactAlarmPermission.shouldRequestPermission(context, shared))
		{
			set.add(Permission.SCHEDULE_EXACT_ALARM)
		}

		// System alert window
		if (NacSystemAlertWindowPermission.shouldRequestPermission(context, shared))
		{
			set.add(Permission.SYSTEM_ALERT_WINDOW)
		}

		//// Ignore battery optimization
		//if (NacIgnoreBatteryOptimizationPermission.shouldRequestPermission(context, shared))
		//{
		//	set.add(Permission.IGNORE_BATTERY_OPTIMIZATION)
		//}

		// Set the permissions that need to be requested
		permissionRequestSet = set

		// Set the current position
		currentPosition = 0

		// Set the total number of permissions that need to be requested
		totalNumberOfPermissions = set.size

		// Set the flag indicating that analysis has been completed
		isAnalyzed = true
	}

	/**
	 * Get the number of permissions that need to be requested.
	 *
	 * @return The number of permissions that need to be requested.
	 */
	fun count(): Int
	{
		return totalNumberOfPermissions
	}

	/**
	 * Increment the current position.
	 */
	private fun incrementCurrentPosition()
	{
		currentPosition += 1
	}

	/**
	 * Request all permissions that need to be requested.
	 */
	fun requestPermissions(activity: AppCompatActivity, onDone: () -> Unit = {})
	{
		// Analyze which permissions need to be requested
		analyze(activity)

		// Show the first permission dialog that should be requested
		showNextPermissionRequestDialog(activity, onDone=onDone)
	}

	/**
	 * Reset all attributes.
	 */
	fun reset()
	{
		permissionRequestSet = EnumSet.noneOf(Permission::class.java)
		currentPosition = 0
		totalNumberOfPermissions = 0
		isAnalyzed = false
	}

	/**
	 * Setup dialog page information.
	 */
	private fun setupDialogPageInfo(dialog: NacPermissionRequestDialog)
	{
		// Set position
		dialog.position = currentPosition

		// Set total number of pages
		dialog.totalNumberOfPages = totalNumberOfPermissions
	}

	/**
	 * Show the next permission request dialog.
	 */
	@SuppressLint("NewApi")
	fun showNextPermissionRequestDialog(activity: AppCompatActivity, onDone: () -> Unit = {})
	{
		// Get the permission request set
		val permissionRequestSet = permissionRequestSet

		// Increment the current position
		incrementCurrentPosition()

		// Show the dialog to request the permission to...
		// Post notifications
		if (permissionRequestSet.contains(Permission.POST_NOTIFICATIONS))
		{
			showPostNotificationPermissionDialog(activity, onDone=onDone)
		}
		// Schedule exact alarm
		else if (permissionRequestSet.contains(Permission.SCHEDULE_EXACT_ALARM))
		{
			showScheduleExactAlarmPermissionDialog(activity, onDone=onDone)
		}
		// System alert window
		else if (permissionRequestSet.contains(Permission.SYSTEM_ALERT_WINDOW))
		{
			showSystemAlertWindowPermissionDialog(activity, onDone=onDone)
		}
		//// Ignore battery optimization
		//else if (permissionRequestSet.contains(Permission.IGNORE_BATTERY_OPTIMIZATION))
		//{
		//	showIgnoreBatteryOptimizationPermissionDialog(activity, onDone=onDone)
		//}
		// Reset and done
		else
		{
			reset()
			onDone()
		}
	}

	///**
	// * Show the dialog to ignore battery optimizations.
	// */
	//private fun showIgnoreBatteryOptimizationPermissionDialog(
	//	activity: AppCompatActivity, onDone: () -> Unit = {})
	//{
	//	// Create the dialog
	//	val dialog = NacIgnoreBatteryOptimizationPermissionRequestDialog()

	//	// Setup the current position and total number of pages in the dialog
	//	setupDialogPageInfo(dialog)

	//	// Handle the cases where the permission request is accepted/canceled
	//	dialog.onPermissionRequestListener = object : OnPermissionRequestListener
	//	{
	//		/**
	//		 * Called when the permission request is accepted.
	//		 */
	//		override fun onPermissionRequestAccepted(permission: String)
	//		{
	//			permissionRequestSet.remove(Permission.IGNORE_BATTERY_OPTIMIZATION)
	//			NacIgnoreBatteryOptimizationPermission.requestPermission(activity)
	//		}

	//		/**
	//		 * Called when the permission request was canceled.
	//		 */
	//		override fun onPermissionRequestCanceled(permission: String)
	//		{
	//			permissionRequestSet.remove(Permission.IGNORE_BATTERY_OPTIMIZATION)
	//			showNextPermissionRequestDialog(activity, onDone=onDone)
	//		}
	//	}

	//	// Show the dialog
	//	dialog.show(activity.supportFragmentManager,
	//		NacIgnoreBatteryOptimizationPermissionRequestDialog.TAG)
	//}

	/**
	 * Show the POST_NOTIFICATIONS permission dialog.
	 */
	fun showPostNotificationPermissionDialog(activity: AppCompatActivity, onDone: () -> Unit = {})
	{
		// Do nothing if an older version of Android is being used
		if (Build.VERSION.SDK_INT < Build.VERSION_CODES.TIRAMISU)
		{
			return
		}

		// Create the dialog
		val dialog = NacPostNotificationsPermissionRequestDialog()

		// Setup the current position and total number of pages in the dialog
		setupDialogPageInfo(dialog)

		// Handle the cases where the permission request is accepted/canceled
		dialog.onPermissionRequestListener = object : OnPermissionRequestListener
		{
			/**
			 * Called when the permission request is accepted.
			 */
			override fun onPermissionRequestAccepted(permission: String)
			{
				permissionRequestSet.remove(Permission.POST_NOTIFICATIONS)
				NacPostNotificationsPermission.requestPermission(activity, 69)
			}

			/**
			 * Called when the permission request is canceled.
			 */
			override fun onPermissionRequestCanceled(permission: String)
			{
				permissionRequestSet.remove(Permission.POST_NOTIFICATIONS)
				showNextPermissionRequestDialog(activity, onDone=onDone)
			}
		}

		// Show the dialog
		dialog.show(activity.supportFragmentManager,
			NacPostNotificationsPermissionRequestDialog.TAG)
	}

	/**
	 * Show the dialog to request the schedule exact alarm permission.
	 */
	fun showScheduleExactAlarmPermissionDialog(activity: AppCompatActivity, onDone: () -> Unit = {})
	{
		// Do nothing if an older version of Android is being used
		if (Build.VERSION.SDK_INT < Build.VERSION_CODES.S)
		{
			return
		}

		// Create the dialog
		val dialog = NacScheduleExactAlarmPermissionRequestDialog()

		// Setup the current position and total number of pages in the dialog
		setupDialogPageInfo(dialog)

		// Handle the cases where the permission request is accepted/canceled
		dialog.onPermissionRequestListener = object : OnPermissionRequestListener
		{
			/**
			 * Called when the permission request is accepted.
			 */
			override fun onPermissionRequestAccepted(permission: String)
			{
				permissionRequestSet.remove(Permission.SCHEDULE_EXACT_ALARM)
				NacScheduleExactAlarmPermission.requestPermission(activity)
			}

			/**
			 * Called when the permission request was canceled.
			 */
			override fun onPermissionRequestCanceled(permission: String)
			{
				permissionRequestSet.remove(Permission.SCHEDULE_EXACT_ALARM)
				showNextPermissionRequestDialog(activity, onDone=onDone)
			}
		}

		// Show the dialog
		dialog.show(activity.supportFragmentManager,
			NacScheduleExactAlarmPermissionRequestDialog.TAG)
	}

	/**
	 * Show the dialog to request the SYSTEM_ALERT_WINDOW permission.
	 */
	fun showSystemAlertWindowPermissionDialog(activity: AppCompatActivity, onDone: () -> Unit = {})
	{
		// Do nothing if an older version of Android is being used
		if (Build.VERSION.SDK_INT < Build.VERSION_CODES.TIRAMISU)
		{
			return
		}

		// Create the dialog
		val dialog = NacSystemAlertWindowPermissionRequestDialog()

		// Setup the current position and total number of pages in the dialog
		setupDialogPageInfo(dialog)

		// Handle the cases where the permission request is accepted/canceled
		dialog.onPermissionRequestListener = object : OnPermissionRequestListener
		{
			/**
			 * Called when the permission request is accepted.
			 */
			override fun onPermissionRequestAccepted(permission: String)
			{
				permissionRequestSet.remove(Permission.SYSTEM_ALERT_WINDOW)
				NacSystemAlertWindowPermission.requestPermission(activity)
			}

			/**
			 * Called when the permission request was canceled.
			 */
			override fun onPermissionRequestCanceled(permission: String)
			{
				permissionRequestSet.remove(Permission.SYSTEM_ALERT_WINDOW)
				showNextPermissionRequestDialog(activity, onDone=onDone)
			}
		}

		// Show the dialog
		dialog.show(activity.supportFragmentManager,
			NacScheduleExactAlarmPermissionRequestDialog.TAG)
	}

}