/*
 * Nextcloud - Android Client
 *
 * SPDX-FileCopyrightText: 2025 Jimly Asshiddiqy
 * SPDX-License-Identifier: AGPL-3.0-or-later
 */

import org.gradle.testing.jacoco.plugins.JacocoPlugin
import org.gradle.testing.jacoco.plugins.JacocoPluginExtension
import org.gradle.testing.jacoco.tasks.JacocoReport
import org.gradle.testing.jacoco.plugins.JacocoTaskExtension
import org.gradle.api.tasks.testing.Test
import org.gradle.kotlin.dsl.apply
import org.gradle.kotlin.dsl.configure
import org.gradle.kotlin.dsl.getByType
import org.gradle.kotlin.dsl.named
import org.gradle.kotlin.dsl.register


apply<JacocoPlugin>()

// Use version from version catalog
configure<JacocoPluginExtension> {
    toolVersion = extensions
        .getByType<VersionCatalogsExtension>()
        .named("libs")
        .findVersion("jacoco")
        .get()
        .requiredVersion
}

val flavor = "gplay"
val buildType = "Debug"
val variant = "${flavor.replaceFirstChar(Char::titlecase)}${buildType.replaceFirstChar(Char::titlecase)}"

val fileFilter = listOf(
    // data binding
    "**/databinding/*",
    "android/databinding/**/*.class",
    "**/android/databinding/*Binding.class",
    "**/android/databinding/*",
    "**/androidx/databinding/*",
    "**/BR.*",
    // android
    "**/R.class",
    "**/R\$*.class",
    "**/BuildConfig.*",
    "**/Manifest*.*",
    "**/*Test*.*",
    "android/**/*.*",
    // kotlin
    "**/*MapperImpl*.*",
    "**/*\$ViewInjector*.*",
    "**/*\$ViewBinder*.*",
    "**/*Component*.*",
    "**/*BR*.*",
    "**/Manifest*.*",
    "**/*\$Lambda\$*.*",
    "**/*Companion*.*",
    "**/*Module*.*",
    "**/*Dagger*.*",
    "**/*Hilt*.*",
    "**/*MembersInjector*.*",
    "**/*_MembersInjector.class",
    "**/*_Factory*.*",
    "**/*_Provide*Factory*.*",
    "**/*Extensions*.*",
    // sealed and data classes
    "**/*\$Result.*",
    "**/*\$Result\$*.*",
    // adapters generated by moshi
    "**/*JsonAdapter.*",
    // Hilt
    "**/*Module.kt",
    "**/di/**",
    "dagger.hilt.internal/*",
    "hilt_aggregated_deps/*",
    // Navigation component / inlined classes
    "**/*Args*.*",
    "**/*Directions*.*",
    "**/*inlined*.class",
    // Jetpack Compose generated
    "**/composables/**"
)

// Configure test tasks to include Jacoco instrumentation
tasks.withType<Test>().configureEach {
    extensions.configure(JacocoTaskExtension::class) {
        isIncludeNoLocationClasses = true
        excludes = listOf("jdk.internal.*")
    }
}

// Main Jacoco report task
val jacocoReport = tasks.register<JacocoReport>("jacocoTest${variant}UnitTestReport") {
    dependsOn("test${variant}UnitTest")

    reports {
        xml.required.set(true)
        html.required.set(true)
        csv.required.set(false)

        // Output locations
        html.outputLocation.set(layout.buildDirectory.dir("reports/jacoco/${name}"))
        xml.outputLocation.set(layout.buildDirectory.file("reports/jacoco/${name}/jacoco.xml"))
    }

    // Directories
    val kotlinClasses =  fileTree(layout.buildDirectory.dir("tmp/kotlin-classes/${variant}"))
    val javacClasses = fileTree(layout.buildDirectory.dir("intermediates/javac/${variant}/classes"))

    classDirectories.setFrom(files(kotlinClasses, javacClasses))
    sourceDirectories.setFrom(
        files(
            "src/main/java",
            "src/$flavor/java",
            "src/${buildType.lowercase()}/java"
        )
    )
    executionData.setFrom(
        fileTree(layout.buildDirectory.asFile) {
            include(
                "jacoco/test${variant}UnitTest.exec",
                "outputs/unit_test_code_coverage/${variant}UnitTest/test${variant}UnitTest.exec"
            )
        }
    )
}

// Backward compatibility alias
tasks.register("jacocoTest${variant}UnitTest") {
    dependsOn("test${variant}UnitTest", jacocoReport)
}
