package com.monuk7735.nope.remote.ui.theme

import android.app.Activity
import android.content.SharedPreferences
import android.content.res.Configuration
import android.os.Build
import androidx.compose.foundation.isSystemInDarkTheme
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.darkColorScheme
import androidx.compose.material3.dynamicDarkColorScheme
import androidx.compose.material3.dynamicLightColorScheme
import androidx.compose.material3.lightColorScheme
import androidx.compose.runtime.Composable
import androidx.compose.runtime.SideEffect
import androidx.compose.ui.graphics.toArgb
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.platform.LocalView
import androidx.compose.runtime.remember
import androidx.compose.ui.platform.LocalContext
import android.content.Context
import androidx.core.view.WindowCompat
import com.monuk7735.nope.remote.R
import com.monuk7735.nope.remote.utils.SystemBarHelper
import androidx.compose.runtime.DisposableEffect
import androidx.compose.runtime.mutableStateOf

data class ThemeSettings(
    val useDarkTheme: Boolean,
    val useDynamicColors: Boolean
)

@Composable
fun rememberThemeSettings(): ThemeSettings {
    val context = LocalContext.current
    val sharedPrefName = androidx.compose.ui.res.stringResource(R.string.shared_pref_app_settings)
    val sharedPrefs = remember {
        context.getSharedPreferences(
            sharedPrefName,
            Context.MODE_PRIVATE
        )
    }

    val darkModeKey = androidx.compose.ui.res.stringResource(R.string.pref_settings_dark_mode)
    val dynamicColorKey = androidx.compose.ui.res.stringResource(R.string.pref_settings_dynamic_color)
    val systemInDarkTheme = isSystemInDarkTheme()

    val (settings, setSettings) = remember {
        mutableStateOf(
            ThemeSettings(
                useDarkTheme = when (sharedPrefs.getInt(darkModeKey, 0)) {
                    1 -> false
                    2 -> true
                    else -> systemInDarkTheme // Initial state using captured value
                },
                useDynamicColors = sharedPrefs.getBoolean(dynamicColorKey, true)
            )
        )
    }

    DisposableEffect(sharedPrefs) {
        val listener = SharedPreferences.OnSharedPreferenceChangeListener { prefs, key ->
            if (key == darkModeKey || key == dynamicColorKey) {
                setSettings(
                    ThemeSettings(
                        useDarkTheme = when (prefs.getInt(darkModeKey, 0)) {
                            1 -> false
                            2 -> true
                            else -> false // Value ignored as it is overridden by finalDarkMode logic
                        },
                        useDynamicColors = prefs.getBoolean(dynamicColorKey, true)
                    )
                )
            }
        }
        sharedPrefs.registerOnSharedPreferenceChangeListener(listener)
        onDispose {
            sharedPrefs.unregisterOnSharedPreferenceChangeListener(listener)
        }
    }
    
    val finalDarkMode = remember(settings, systemInDarkTheme) {
        val prefMode = sharedPrefs.getInt(darkModeKey, 0)
        when (prefMode) {
             1 -> false
             2 -> true
             else -> systemInDarkTheme
        }
    }

    return settings.copy(useDarkTheme = finalDarkMode)
}

private val LightThemeColorScheme = lightColorScheme(

    primary = md_theme_light_primary,
    onPrimary = md_theme_light_onPrimary,
    primaryContainer = md_theme_light_primaryContainer,
    onPrimaryContainer = md_theme_light_onPrimaryContainer,
    secondary = md_theme_light_secondary,
    onSecondary = md_theme_light_onSecondary,
    secondaryContainer = md_theme_light_secondaryContainer,
    onSecondaryContainer = md_theme_light_onSecondaryContainer,
    tertiary = md_theme_light_tertiary,
    onTertiary = md_theme_light_onTertiary,
    tertiaryContainer = md_theme_light_tertiaryContainer,
    onTertiaryContainer = md_theme_light_onTertiaryContainer,
    error = md_theme_light_error,
    errorContainer = md_theme_light_errorContainer,
    onError = md_theme_light_onError,
    onErrorContainer = md_theme_light_onErrorContainer,
    background = md_theme_light_background,
    onBackground = md_theme_light_onBackground,
    surface = md_theme_light_surface,
    onSurface = md_theme_light_onSurface,
    surfaceVariant = md_theme_light_surfaceVariant,
    onSurfaceVariant = md_theme_light_onSurfaceVariant,
    outline = md_theme_light_outline,
    inverseOnSurface = md_theme_light_inverseOnSurface,
    inverseSurface = md_theme_light_inverseSurface,
)
private val DarkThemeColorScheme = darkColorScheme(

    primary = md_theme_dark_primary,
    onPrimary = md_theme_dark_onPrimary,
    primaryContainer = md_theme_dark_primaryContainer,
    onPrimaryContainer = md_theme_dark_onPrimaryContainer,
    secondary = md_theme_dark_secondary,
    onSecondary = md_theme_dark_onSecondary,
    secondaryContainer = md_theme_dark_secondaryContainer,
    onSecondaryContainer = md_theme_dark_onSecondaryContainer,
    tertiary = md_theme_dark_tertiary,
    onTertiary = md_theme_dark_onTertiary,
    tertiaryContainer = md_theme_dark_tertiaryContainer,
    onTertiaryContainer = md_theme_dark_onTertiaryContainer,
    error = md_theme_dark_error,
    errorContainer = md_theme_dark_errorContainer,
    onError = md_theme_dark_onError,
    onErrorContainer = md_theme_dark_onErrorContainer,
    background = md_theme_dark_background,
    onBackground = md_theme_dark_onBackground,
    surface = md_theme_dark_surface,
    onSurface = md_theme_dark_onSurface,
    surfaceVariant = md_theme_dark_surfaceVariant,
    onSurfaceVariant = md_theme_dark_onSurfaceVariant,
    outline = md_theme_dark_outline,
    inverseOnSurface = md_theme_dark_inverseOnSurface,
    inverseSurface = md_theme_dark_inverseSurface,
)

@Composable
fun NopeRemoteTheme(
    useDarkTheme: Boolean = isSystemInDarkTheme(),
    useDynamicColors: Boolean = true,
    content: @Composable() () -> Unit,
) {
    val colorScheme = when {
        useDynamicColors && Build.VERSION.SDK_INT >= Build.VERSION_CODES.S -> {
            val context = LocalContext.current
            if (useDarkTheme) dynamicDarkColorScheme(context) else dynamicLightColorScheme(context)
        }
        useDarkTheme -> DarkThemeColorScheme
        else -> LightThemeColorScheme
    }

    val view = LocalView.current
    if (!view.isInEditMode) {
        SideEffect {
            val window = (view.context as Activity).window
            SystemBarHelper.setTransparentSystemBars(window)
            
            val insetsController = WindowCompat.getInsetsController(window, view)
            insetsController.isAppearanceLightStatusBars = !useDarkTheme
            insetsController.isAppearanceLightNavigationBars = !useDarkTheme
        }
    }

    MaterialTheme(
        colorScheme = colorScheme,
        typography = AppTypography,
        content = content
    )

}