/*
 * Copyright (C) 2012-2018 The Android Money Manager Ex Project Team
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package com.money.manager.ex.reports;

import android.content.ContentValues;
import android.content.Intent;
import android.database.Cursor;
import android.database.DatabaseUtils;
import android.database.sqlite.SQLiteQueryBuilder;
import android.graphics.Typeface;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import android.text.TextUtils;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.View;
import android.widget.LinearLayout;
import android.widget.ListView;
import android.widget.TextView;

import com.money.manager.ex.R;
import com.money.manager.ex.account.AccountTypes;
import com.money.manager.ex.core.TransactionTypes;
import com.money.manager.ex.core.UIHelper;
import com.money.manager.ex.currency.CurrencyService;
import com.money.manager.ex.database.QueryMobileData;
import com.money.manager.ex.search.CategorySub;
import com.money.manager.ex.search.SearchActivity;
import com.money.manager.ex.search.SearchParameters;

import org.parceler.Parcels;

import java.util.ArrayList;

import androidx.annotation.NonNull;
import androidx.core.view.MenuHost;
import androidx.core.view.MenuProvider;
import androidx.fragment.app.FragmentManager;
import androidx.fragment.app.FragmentTransaction;
import androidx.lifecycle.Lifecycle;
import androidx.loader.content.Loader;
import info.javaperformance.money.Money;
import info.javaperformance.money.MoneyFactory;
import timber.log.Timber;

/**
 * Categories report fragment.
 */
public class CategoriesReportFragment
    extends BaseReportFragment {

    private LinearLayout mListViewFooter;

    @Override
    public void onActivityCreated(Bundle savedInstanceState) {
        setListAdapter(null);
        setSearchMenuVisible(true);

        // disable fab issue #2504
        setFabVisible(false);

        //create header view
        LinearLayout mListViewHeader = (LinearLayout) addListViewHeaderFooter(R.layout.item_generic_report_2_columns);
        TextView txtColumn1 = mListViewHeader.findViewById(R.id.textViewColumn1);
        TextView txtColumn2 = mListViewHeader.findViewById(R.id.textViewColumn2);
        //set header
        txtColumn1.setText(R.string.category);
        txtColumn1.setTypeface(null, Typeface.BOLD);
        txtColumn2.setText(R.string.amount);
        txtColumn2.setTypeface(null, Typeface.BOLD);
        //add to list view
        getListView().addHeaderView(mListViewHeader);

        //create footer view
        mListViewFooter = (LinearLayout) addListViewHeaderFooter(R.layout.item_generic_report_2_columns);
        txtColumn1 = mListViewFooter.findViewById(R.id.textViewColumn1);
        txtColumn2 = mListViewFooter.findViewById(R.id.textViewColumn2);
        //set footer
        txtColumn1.setText(R.string.total);
        txtColumn1.setTypeface(null, Typeface.BOLD_ITALIC);
        txtColumn2.setText(R.string.total);
        txtColumn2.setTypeface(null, Typeface.BOLD_ITALIC);

        //add to list view --> move to load finished
        //getListView().addFooterView(mListViewFooter);

        //set adapter
        CategoriesReportAdapter adapter = new CategoriesReportAdapter(getActivity(), null);
        setListAdapter(adapter);
        //call super method
        super.onActivityCreated(savedInstanceState);
    }

    @Override
    protected void addCustomMenuProviders(MenuHost menuHost) {
        super.addCustomMenuProviders(menuHost);
        // add menu
        menuHost.addMenuProvider(new MenuProvider() {
            @Override
            public void onCreateMenu(@NonNull Menu menu, @NonNull MenuInflater menuInflater) {
                old_onCreateOptionsMenu(menu, menuInflater);
            }

            @Override
            public boolean onMenuItemSelected(@NonNull MenuItem menuItem) {
                return old_onOptionsItemSelected(menuItem);
            }
        }, getViewLifecycleOwner(), Lifecycle.State.RESUMED);
    }

    public void old_onCreateOptionsMenu(@NonNull Menu menu, @NonNull MenuInflater inflater) {
        super.old_onCreateOptionsMenu(menu, inflater);
        // pie chart
        MenuItem itemChart = menu.findItem(R.id.menu_chart);
        if (itemChart != null) {
            itemChart.setVisible(!(((CategoriesReportActivity) getActivity()).mIsDualPanel));
            UIHelper uiHelper = new UIHelper(getActivity());
            itemChart.setIcon(uiHelper.resolveAttribute(R.attr.ic_action_pie_chart));
        }
    }


    public boolean old_onOptionsItemSelected(@NonNull MenuItem item) {
        if (item.getItemId() == R.id.menu_chart) {
            showChart();
        }

        if (item.getItemId() < 0) {
            // category
            String whereClause = getWhereClause();
            if (!TextUtils.isEmpty(whereClause))
                whereClause += " AND ";
            else
                whereClause = "";
            whereClause += " " + QueryMobileData.CATEGID + "=" + Math.abs(item.getItemId());
            //create arguments
            Bundle args = new Bundle();
            args.putString(KEY_WHERE_CLAUSE, whereClause);
            //starts loader
            startLoader(args);
        }
        return super.old_onOptionsItemSelected(item);
    }


    // Loader

    @Override
    public void onLoadFinished(Loader<Cursor> loader, Cursor data) {
        super.onLoadFinished(loader, data);

        if (loader.getId() == ID_LOADER) {//parse cursor for calculate total
            if (data == null) return;

            // move to first record #1539
            data.moveToPosition(-1);

            CurrencyService currencyService = new CurrencyService(getActivity().getApplicationContext());

            Money totalAmount = MoneyFactory.fromString("0");
            while (data.moveToNext()) {
                String totalRow = data.getString(data.getColumnIndexOrThrow("TOTAL"));
                if (!TextUtils.isEmpty(totalRow)) {
                    totalAmount = totalAmount.add(MoneyFactory.fromString(totalRow));
                } else {
                    new UIHelper(getActivity()).showToast("reading total");
                }
            }
            TextView txtColumn2 = mListViewFooter.findViewById(R.id.textViewColumn2);
            txtColumn2.setText(currencyService.getBaseCurrencyFormatted(totalAmount));

            // solved bug chart
            if (data.getCount() > 0) {
                getListView().removeFooterView(mListViewFooter);
                getListView().addFooterView(mListViewFooter);
            }

            if (((CategoriesReportActivity) getActivity()).mIsDualPanel) {
                Handler handler = new Handler(Looper.getMainLooper());
                handler.postDelayed(() -> showChart(), 1000);
            }
        }
    }

    @Override
    protected boolean onQueryTextChange(String newText) {
        //recall last where clause
        String whereClause = getWhereClause();
        if (whereClause == null) whereClause = "";

        int start = whereClause.indexOf("/** */");
        if (start > 0) {
            int end = whereClause.indexOf("/** */", start + 1) + "/** */".length();
            whereClause = whereClause.substring(0, start) + whereClause.substring(end);
            // trim some space
            whereClause = whereClause.trim();
        }

        if (!TextUtils.isEmpty(whereClause)) {
            whereClause += " /** */AND ";
        } else {
            whereClause = "/** */";
        }
        // use token to replace criteria
        whereClause += "(" + QueryMobileData.Category + " Like '%" + newText + "%')";

        //create arguments
        Bundle args = new Bundle();
        args.putString(KEY_WHERE_CLAUSE, whereClause);
        //starts loader
        startLoader(args);
        return super.onQueryTextChange(newText);
    }

    @Override
    protected String prepareQuery(String whereClause) {
        SQLiteQueryBuilder builder = new SQLiteQueryBuilder();
        QueryMobileData mobileData = new QueryMobileData(getContext());

        //data to compose builder
        String[] projectionIn = new String[]{
            "ID AS _id", // this does not fetch anything, unfortunately.
            QueryMobileData.CATEGID, QueryMobileData.Category,
            "SUM(" + QueryMobileData.AmountBaseConvRate + ") AS TOTAL"
        };

        String selection = QueryMobileData.Status + "<>'V' AND " +
            QueryMobileData.TransactionType + " IN ('Withdrawal', 'Deposit')" +
// ignore Stock Movement
                " AND " +
                QueryMobileData.ACCOUNTTYPE + "<>'"+ AccountTypes.SHARES.toString()  +"' AND " +
                QueryMobileData.TOACCOUNTTYPE + "<>'"+ AccountTypes.SHARES.toString() +"'" ;
        if (!TextUtils.isEmpty(whereClause)) {
            selection += " AND " + whereClause;
        }


        String groupBy = QueryMobileData.CATEGID + ", " + QueryMobileData.Category;

        String having = null;
        if (!TextUtils.isEmpty(((CategoriesReportActivity) getActivity()).mFilter)) {
            String filter = ((CategoriesReportActivity) getActivity()).mFilter;
            if (TransactionTypes.valueOf(filter).equals(TransactionTypes.Withdrawal)) {
                having = "SUM(" + QueryMobileData.AmountBaseConvRate + ") < 0";
            } else {
                having = "SUM(" + QueryMobileData.AmountBaseConvRate + ") > 0";
            }
        }

        // String sortOrder = QueryMobileData.Category + ", " + QueryMobileData.Subcategory;
        String sortOrder = QueryMobileData.Category;

        //compose builder
        builder.setTables(mobileData.getSource());

        //return query
        return builder.buildQuery(projectionIn, selection, groupBy, having, sortOrder, null);
    }

    @Override
    public String getSubTitle() {
        return null;
    }

    /**
     * List item clicked. Show the transaction list for the category.
     * @param l        The ListView where the click happened
     * @param v        The view that was clicked within the ListView
     * @param position The position of the view in the list
     * @param id       The row id of the item that was clicked
     */
    @Override
    public void onListItemClick(ListView l, View v, int position, long id) {
        CategorySub category = null;
        try {
            category = getCategoryFromSelectedItem(l, position);
        } catch (Exception e) {
            Timber.e(e, "getting category from selected item");
        }
        if (category == null) return;

        // now list the transactions for the given category/subcategory combination,
        // in the selected time period.

//        showTransactionsFragment(values);

        // Show search activity with the results.
        SearchParameters parameters = new SearchParameters();
        parameters.category = category;
        parameters.dateFrom = mDateFrom;
        parameters.dateTo = mDateTo;

        showSearchActivityFor(parameters);
    }

    public void showChart() {
        CategoriesReportAdapter adapter = (CategoriesReportAdapter) getListAdapter();
        if (adapter == null) return;
        Cursor cursor = adapter.getCursor();
        if (cursor == null) return;
        if (cursor.getCount() <= 0) return;

        ArrayList<ValuePieEntry> arrayList = new ArrayList<>();
        CurrencyService currencyService = new CurrencyService(getActivity().getApplicationContext());

        // Reset cursor to initial position.
        cursor.moveToPosition(-1);
        // process cursor
        while (cursor.moveToNext()) {
            ValuePieEntry item = new ValuePieEntry();
            String category = cursor.getString(cursor.getColumnIndexOrThrow(QueryMobileData.Category));
            // total
            double total = Math.abs(cursor.getDouble(cursor.getColumnIndexOrThrow("TOTAL")));
            // check if category is empty
            if (TextUtils.isEmpty(category)) {
                category = getString(R.string.empty_category);
            }

            item.setText(category);
            item.setValue(total);
            item.setValueFormatted(currencyService.getCurrencyFormatted(currencyService.getBaseCurrencyId(),
                    MoneyFactory.fromDouble(total)));
            // add element
            arrayList.add(item);
        }

        Bundle args = new Bundle();
        args.putSerializable(PieChartFragment.KEY_CATEGORIES_VALUES, arrayList);
        //get fragment manager
        FragmentManager fragmentManager = getActivity().getSupportFragmentManager();
        if (fragmentManager != null) {
            PieChartFragment fragment;
            fragment = (PieChartFragment) fragmentManager.findFragmentByTag(IncomeVsExpensesChartFragment.class.getSimpleName());
            if (fragment == null) {
                fragment = new PieChartFragment();
            }
            fragment.setChartArguments(args);
            fragment.setDisplayHomeAsUpEnabled(true);

            if (fragment.isVisible()) fragment.onResume();

            FragmentTransaction fragmentTransaction = fragmentManager.beginTransaction();
            if (((CategoriesReportActivity) getActivity()).mIsDualPanel) {
                fragmentTransaction.replace(R.id.fragmentChart, fragment, PieChartFragment.class.getSimpleName());
            } else {
                fragmentTransaction.replace(R.id.fragmentMain, fragment, PieChartFragment.class.getSimpleName());
                fragmentTransaction.addToBackStack(null);
            }
            fragmentTransaction.commit();
        }
    }

    // Private

    private CategorySub getCategoryFromSelectedItem(ListView l, int position) {
        // Reading item from the list view, not adapter!
        Object item = l.getItemAtPosition(position);
        if (item == null) return null;

        Cursor cursor = (Cursor) item;

        ContentValues values = new ContentValues();
        DatabaseUtils.cursorLongToContentValues(cursor, QueryMobileData.CATEGID, values);
        DatabaseUtils.cursorStringToContentValues(cursor, QueryMobileData.Category, values);

        CategorySub result = new CategorySub();
        result.categId = values.getAsLong(QueryMobileData.CATEGID);
        result.categName = values.getAsString(QueryMobileData.Category);
        return result;
    }

    private void showSearchActivityFor(SearchParameters parameters) {
        Intent intent = new Intent(getActivity(), SearchActivity.class);
        intent.putExtra(SearchActivity.EXTRA_SEARCH_PARAMETERS, Parcels.wrap(parameters));
        intent.setAction(Intent.ACTION_INSERT);
        startActivity(intent);
    }

}
