package com.mhss.app.presentation

import androidx.compose.material3.SnackbarHostState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.setValue
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import com.mhss.app.domain.model.Task
import com.mhss.app.domain.use_case.CanScheduleAlarmsUseCase
import com.mhss.app.domain.use_case.DeleteTaskUseCase
import com.mhss.app.domain.use_case.GetTaskByIdUseCase
import com.mhss.app.domain.use_case.UpsertTaskUseCase
import com.mhss.app.ui.R
import com.mhss.app.ui.snackbar.showSnackbar
import com.mhss.app.util.date.now
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.launch
import org.koin.android.annotation.KoinViewModel
import org.koin.core.annotation.Named

@KoinViewModel
class TaskDetailsViewModel(
    private val getTask: GetTaskByIdUseCase,
    private val upsertTask: UpsertTaskUseCase,
    private val deleteTask: DeleteTaskUseCase,
    private val canScheduleAlarms: CanScheduleAlarmsUseCase,
    @Named("applicationScope") private val applicationScope: CoroutineScope,
    taskId: String
) : ViewModel() {

    var taskDetailsUiState by mutableStateOf(TaskDetailsUiState())
        private set

    init {
        viewModelScope.launch {
            val task = getTask(taskId)
            if (taskId.isNotBlank() && task == null) {
                taskDetailsUiState.snackbarHostState.showSnackbar(R.string.error_item_not_found)
            }
            taskDetailsUiState = taskDetailsUiState.copy(
               task = task
            )
        }
    }

    fun onEvent(event: TaskDetailsEvent) {
        when (event) {

            TaskDetailsEvent.ErrorDisplayed -> {
                taskDetailsUiState = taskDetailsUiState.copy(alarmError = false)
            }
            // Using applicationScope to avoid cancelling when the user exits the screen
            // and the view model is cleared before the job finishes
            is TaskDetailsEvent.ScreenOnStop -> applicationScope.launch {
                if (!taskDetailsUiState.navigateUp) {
                    if (taskChanged(taskDetailsUiState.task!!, event.task)) {
                        val newTask = taskDetailsUiState.task!!.copy(
                            title = event.task.title.ifBlank { "Untitled" },
                            description = event.task.description,
                            dueDate = event.task.dueDate,
                            priority = event.task.priority,
                            subTasks = event.task.subTasks,
                            recurring = event.task.recurring,
                            frequency = event.task.frequency,
                            frequencyAmount = event.task.frequencyAmount,
                            isCompleted = event.task.isCompleted,
                            updatedDate = now()
                        )
                        upsertTask(
                            task = newTask,
                            previousTask = taskDetailsUiState.task
                        )
                        taskDetailsUiState = taskDetailsUiState.copy(task = newTask)
                    }
                }
            }

            is TaskDetailsEvent.DeleteTask -> viewModelScope.launch {
                deleteTask(taskDetailsUiState.task!!)
                taskDetailsUiState = taskDetailsUiState.copy(navigateUp = true)
            }

            TaskDetailsEvent.DueDateEnabled -> {
                if (!canScheduleAlarms()) {
                    taskDetailsUiState = taskDetailsUiState.copy(alarmError = true)
                }
            }
        }
    }

    data class TaskDetailsUiState(
        val task: Task? = null,
        val navigateUp: Boolean = false,
        val alarmError: Boolean = false,
        val snackbarHostState: SnackbarHostState = SnackbarHostState()
    )

    private fun taskChanged(
        task: Task,
        newTask: Task
    ): Boolean {
        return task.title != newTask.title ||
                task.description != newTask.description ||
                task.dueDate != newTask.dueDate ||
                task.isCompleted != newTask.isCompleted ||
                task.priority != newTask.priority ||
                task.subTasks != newTask.subTasks ||
                task.recurring != newTask.recurring ||
                task.frequency != newTask.frequency ||
                task.frequencyAmount != newTask.frequencyAmount
    }

}