package com.mhss.app.presentation

import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateListOf
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.setValue
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import com.mhss.app.domain.model.AiMessage
import com.mhss.app.domain.model.AiMessageAttachment
import com.mhss.app.domain.model.AiRepositoryException
import com.mhss.app.domain.model.AssistantResult
import com.mhss.app.domain.model.CalendarEvent
import com.mhss.app.domain.model.Note
import com.mhss.app.domain.model.Task
import com.mhss.app.domain.use_case.GetAllEventsUseCase
import com.mhss.app.domain.use_case.GetNoteUseCase
import com.mhss.app.domain.use_case.GetTaskByIdUseCase
import com.mhss.app.domain.use_case.SearchNotesUseCase
import com.mhss.app.domain.use_case.SearchTasksUseCase
import com.mhss.app.domain.use_case.SendAiMessageUseCase
import com.mhss.app.preferences.PrefsConstants
import com.mhss.app.preferences.domain.model.AiProvider
import com.mhss.app.preferences.domain.model.intPreferencesKey
import com.mhss.app.preferences.domain.model.stringSetPreferencesKey
import com.mhss.app.preferences.domain.model.toAiProvider
import com.mhss.app.preferences.domain.use_case.GetPreferenceUseCase
import com.mhss.app.ui.ItemView
import com.mhss.app.ui.toIntList
import com.mhss.app.ui.toNotesView
import com.mhss.app.util.date.formatDate
import com.mhss.app.util.date.formatDateForMapping
import com.mhss.app.util.date.now
import com.mhss.app.util.date.todayPlusDays
import kotlinx.coroutines.Job
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.catch
import kotlinx.coroutines.flow.collect
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.flow.map
import kotlinx.coroutines.flow.onCompletion
import kotlinx.coroutines.flow.onEach
import kotlinx.coroutines.launch
import kotlinx.serialization.json.Json
import org.koin.android.annotation.KoinViewModel
import kotlin.uuid.Uuid

@KoinViewModel
class AssistantViewModel(
    private val sendAiMessage: SendAiMessageUseCase,
    private val getPreference: GetPreferenceUseCase,
    private val searchNotes: SearchNotesUseCase,
    private val searchTasks: SearchTasksUseCase,
    private val getCalendarEvents: GetAllEventsUseCase,
    private val getNoteById: GetNoteUseCase,
    private val getTaskById: GetTaskByIdUseCase
) : ViewModel() {

    private val _messages = mutableStateListOf<AiMessage>()
    val messages: List<AiMessage> = _messages
    val attachments = mutableStateListOf<AiMessageAttachment>()

    var uiState by mutableStateOf(UiState())
        private set

    var aiEnabled by mutableStateOf(false)
        private set

    private var searchNotesJob: Job? = null
    private var searchTasksJob: Job? = null
    private var sendMessageJob: Job? = null

    init {
        viewModelScope.launch {
            getPreference(
                intPreferencesKey(PrefsConstants.NOTE_VIEW_KEY),
                ItemView.LIST.value
            ).onEach {
                uiState = uiState.copy(noteView = it.toNotesView())
            }.collect()
        }
        viewModelScope.launch {
            getPreference(intPreferencesKey(PrefsConstants.AI_PROVIDER_KEY), AiProvider.None.id)
                .map { it.toAiProvider() }
                .collect { provider ->
                    aiEnabled = provider != AiProvider.None
                }
        }
    }

    fun onEvent(event: AssistantEvent) {
        when (event) {
            is AssistantEvent.SendMessage -> {
                sendMessageJob?.cancel()
                sendMessageJob = viewModelScope.launch {
                    val message = AiMessage.UserMessage(
                        content = event.content,
                        attachments = event.attachments,
                        attachmentsText = getAttachmentText(event.attachments),
                        time = now(),
                        uuid = Uuid.random().toString()
                    )

                    _messages.add(0, message)
                    attachments.clear()

                    uiState = uiState.copy(
                        loading = true,
                        error = null
                    )
                    
                    sendAiMessage(_messages.reversed())
                        .catch { e ->
                            delay(300)
                            
                            val error = if (e is AiRepositoryException) {
                                e.failure
                            } else {
                                AssistantResult.OtherError(e.message)
                            }

                            if (error !is AssistantResult.ToolCallLimitExceeded) {
                                _messages.removeAt(0)
                            }

                            uiState = uiState.copy(
                                loading = false,
                                error = error
                            )
                        }
                        .onCompletion {
                            uiState = uiState.copy(loading = false)
                        }
                        .collect { msg ->
                            _messages.add(0, msg)
                        }
                }
            }

            is AssistantEvent.SearchNotes -> {
                searchNotesJob?.cancel()
                searchNotesJob = viewModelScope.launch {
                    delay(300)
                    searchNotes(event.query).let {
                        uiState = uiState.copy(searchNotes = it)
                    }
                }
            }

            is AssistantEvent.SearchTasks -> {
                searchTasksJob?.cancel()
                searchTasksJob = viewModelScope.launch {
                    delay(300)
                    searchTasks(event.query).first().let {
                        uiState = uiState.copy(searchTasks = it)
                    }
                }
            }

            AssistantEvent.AddAttachmentEvents -> {
                attachments.add(AiMessageAttachment.CalenderEvents)
            }

            is AssistantEvent.AddAttachmentNote -> viewModelScope.launch {
                val note = getNoteById(event.id) ?: return@launch
                attachments.add(
                    AiMessageAttachment.Note(
                        note.copy(
                            title = note.title.ifBlank { "Untitled Note" }
                        )
                    )
                )
            }

            is AssistantEvent.AddAttachmentTask -> viewModelScope.launch {
                attachments.add(AiMessageAttachment.Task(getTaskById(event.id) ?: return@launch ))
            }

            is AssistantEvent.RemoveAttachment -> {
                attachments.removeAt(event.index)
            }

            AssistantEvent.CancelMessage -> {
                sendMessageJob?.cancel()
                if (messages.firstOrNull() is AiMessage.UserMessage) {
                    _messages.removeAt(0)
                }
                uiState = uiState.copy(loading = false)
            }
        }
    }

    private suspend fun getAttachmentText(attachments: List<AiMessageAttachment>): String {
        val builder = StringBuilder()
        if (attachments.isEmpty()) return ""
        builder.appendLine()
        builder.appendLine("Attached content from the user:")
        for (attachment in attachments) {
            when (attachment) {
                is AiMessageAttachment.Note -> {
                    builder.appendLine("Attached Note:")
                    builder.appendLine(Json.encodeToString(attachment.note))
                }

                is AiMessageAttachment.Task -> {
                    builder.appendLine("Attached Task:")
                    builder.appendLine(Json.encodeToString(attachment.task))
                }

                is AiMessageAttachment.CalenderEvents -> {
                    builder.appendLine("Next 7 days events:")
                    builder.appendLine(Json.encodeToString(getEventsForNext7Days()))
                    builder.appendLine("(Today's date: ${now().formatDate()})")
                }
            }
        }
        return builder.toString()
    }

    private suspend fun getEventsForNext7Days(): Map<String, List<CalendarEvent>> {
        val excluded = getPreference(
            stringSetPreferencesKey(PrefsConstants.EXCLUDED_CALENDARS_KEY),
            emptySet()
        ).first()
        return getCalendarEvents(excluded.toIntList(), todayPlusDays(7)) {
            it.start.formatDateForMapping()
        }
    }


    data class UiState(
        val loading: Boolean = false,
        val error: AssistantResult.Failure? = null,
        val noteView: ItemView = ItemView.LIST,
        val searchNotes: List<Note> = emptyList(),
        val searchTasks: List<Task> = emptyList()
    )
}