/*
 * Copyright 2025 Blocker
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.merxury.blocker.feature.applist

import android.content.pm.PackageInfo
import android.content.pm.PackageManager
import android.content.pm.PackageManager.PackageInfoFlags
import androidx.lifecycle.SavedStateHandle
import app.cash.turbine.test
import com.merxury.blocker.core.data.util.PermissionStatus.SHELL_USER
import com.merxury.blocker.core.domain.InitializeDatabaseUseCase
import com.merxury.blocker.core.domain.applist.SearchAppListUseCase
import com.merxury.blocker.core.domain.controller.GetAppControllerUseCase
import com.merxury.blocker.core.domain.controller.GetServiceControllerUseCase
import com.merxury.blocker.core.model.ComponentType.ACTIVITY
import com.merxury.blocker.core.model.ComponentType.PROVIDER
import com.merxury.blocker.core.model.ComponentType.RECEIVER
import com.merxury.blocker.core.model.ComponentType.SERVICE
import com.merxury.blocker.core.model.data.ComponentInfo
import com.merxury.blocker.core.model.data.InstalledApp
import com.merxury.blocker.core.model.preference.AppPropertiesData
import com.merxury.blocker.core.testing.controller.FakeAppController
import com.merxury.blocker.core.testing.controller.FakeServiceController
import com.merxury.blocker.core.testing.data.TestAppStateCache
import com.merxury.blocker.core.testing.repository.TestAppPropertiesRepository
import com.merxury.blocker.core.testing.repository.TestAppRepository
import com.merxury.blocker.core.testing.repository.TestComponentRepository
import com.merxury.blocker.core.testing.repository.TestUserDataRepository
import com.merxury.blocker.core.testing.repository.defaultUserData
import com.merxury.blocker.core.testing.util.MainDispatcherRule
import com.merxury.blocker.core.testing.util.TestAnalyticsHelper
import com.merxury.blocker.core.testing.util.TestPermissionMonitor
import com.merxury.blocker.feature.applist.AppListUiState.Initializing
import com.merxury.blocker.feature.applist.AppListUiState.Success
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.test.runTest
import org.junit.Before
import org.junit.Rule
import org.junit.Test
import org.mockito.kotlin.any
import org.mockito.kotlin.doReturn
import org.mockito.kotlin.mock
import kotlin.test.assertEquals
import kotlin.test.assertIs

class AppListViewModelTest {
    @get:Rule
    val mainDispatcherRule = MainDispatcherRule()

    private val analyticsHelper = TestAnalyticsHelper()
    private val userDataRepository = TestUserDataRepository()
    private val appRepository = TestAppRepository()
    private val componentRepository = TestComponentRepository()
    private val appPropertiesRepository = TestAppPropertiesRepository()
    private val appStateCache = TestAppStateCache()
    private val serviceController = FakeServiceController()
    private val dispatcher: CoroutineDispatcher = mainDispatcherRule.testDispatcher
    private val permissionMonitor = TestPermissionMonitor()
    private val appController = FakeAppController()
    private val packageInfo = mock<PackageInfo> {
        on { toString() } doReturn "MockedPackageInfo"
    }
    private val pm = mock<PackageManager> {
        on { getPackageInfo(any<String>(), any<Int>()) } doReturn packageInfo
        on {
            getPackageInfo(
                any<String>(),
                any<PackageInfoFlags>(),
            )
        } doReturn packageInfo
    }
    private val savedStateHandle = SavedStateHandle()
    private lateinit var viewModel: AppListViewModel

    @Before
    fun setup() {
        val initializeDatabaseUseCase = InitializeDatabaseUseCase(
            appRepository = appRepository,
            componentRepository = componentRepository,
            appPropertiesRepository = appPropertiesRepository,
        )
        val getAppController = GetAppControllerUseCase(
            userDataRepository = userDataRepository,
            rootAppController = appController,
            shizukuAppController = appController,
        )
        val getServiceController = GetServiceControllerUseCase(
            userDataRepository = userDataRepository,
            rootServiceController = serviceController,
            shizukuServiceController = serviceController,
        )
        val searchAppListUseCase = SearchAppListUseCase(
            pm = pm,
            userDataRepository = userDataRepository,
            appRepository = appRepository,
            appStateCache = appStateCache,
            getAppController = getAppController,
            getServiceController = getServiceController,
            cpuDispatcher = dispatcher,
        )
        viewModel = AppListViewModel(
            userDataRepository = userDataRepository,
            analyticsHelper = analyticsHelper,
            permissionMonitor = permissionMonitor,
            pm = pm,
            initializeDatabase = initializeDatabaseUseCase,
            searchAppList = searchAppListUseCase,
            getAppController = getAppController,
            ioDispatcher = dispatcher,
            cpuDispatcher = dispatcher,
            mainDispatcher = dispatcher,
            appRepository = appRepository,
            savedStateHandle = savedStateHandle,
        )
    }

    @Test
    fun appListUiState_whenInitial_thenShowDefault() = runTest {
        assertIs<Initializing>(viewModel.uiState.value)
    }

    @Test
    fun appListUiState_whenInitializingApp_thenShowInitializingApp() = runTest {
        viewModel.uiState.test {
            // Cause sending of app list will trigger the initialization of the database
            // The condition will be initialized in the `test` block
            userDataRepository.sendUserData(defaultUserData)
            appPropertiesRepository.sendAppProperties(AppPropertiesData())
            componentRepository.sendComponentList(sampleComponentList)
            appRepository.sendAppList(sampleAppList)

            // Initial state
            assertEquals(Initializing(), awaitItem())
            // Actual app list
            sampleAppList.forEach {
                assertEquals(Initializing(it.label), awaitItem())
            }
            // Going to search the component list
            // It will emit an Initializing state and then a Success state
            assertEquals(Initializing(), awaitItem())
            assertIs<Success>(awaitItem())
        }
    }

    @Test
    fun appListUiState_whenShowSystemAppChange_thenRefreshing() = runTest {
        viewModel.uiState.test {
            userDataRepository.sendUserData(defaultUserData)
            appPropertiesRepository.sendAppProperties(AppPropertiesData(componentDatabaseInitialized = true))
            componentRepository.sendComponentList(sampleComponentList)
            appRepository.sendAppList(sampleAppList)
            assertEquals(Initializing(), awaitItem())
            assertEquals(Success(), awaitItem())
            userDataRepository.setShowSystemApps(true)
            assertEquals(Success(isRefreshing = true), awaitItem())
            assertEquals(Success(), awaitItem())
        }
    }

    @Test
    fun appListUiState_whenPermissionChange_thenRefreshing() = runTest {
        viewModel.uiState.test {
            userDataRepository.sendUserData(defaultUserData)
            appPropertiesRepository.sendAppProperties(AppPropertiesData(componentDatabaseInitialized = true))
            componentRepository.sendComponentList(sampleComponentList)
            appRepository.sendAppList(sampleAppList)
            assertEquals(Initializing(), awaitItem())
            assertEquals(Success(), awaitItem())
            permissionMonitor.setPermission(SHELL_USER)
            assertEquals(Success(isRefreshing = true), awaitItem())
            assertEquals(Success(), awaitItem())
        }
    }
}

private val sampleAppList = listOf(
    InstalledApp(
        label = "App1",
        packageName = "com.merxury.test1",
    ),
    InstalledApp(
        label = "App2",
        packageName = "com.merxury.test2",
    ),
    InstalledApp(
        label = "App3",
        packageName = "com.merxury.test3",
        isSystem = true,
    ),
)

private val sampleComponentList = listOf(
    ComponentInfo(
        simpleName = "Activity1",
        name = "com.merxury.blocker.test.activity1",
        packageName = "com.merxury.test1",
        type = ACTIVITY,
        description = "An example activity",
    ),
    ComponentInfo(
        simpleName = "Service1",
        name = "com.merxury.blocker.test.service1",
        packageName = "com.merxury.test1",
        type = SERVICE,
        description = "An example service",
        pmBlocked = true,
    ),
    ComponentInfo(
        simpleName = "Receiver1",
        name = "com.merxury.blocker.test.receiver1",
        packageName = "com.merxury.test1",
        type = RECEIVER,
        description = "An example receiver",
    ),
    ComponentInfo(
        simpleName = "Provider1",
        name = "com.merxury.blocker.test.provider1",
        packageName = "com.merxury.test1",
        type = PROVIDER,
        description = "An example provider",
    ),
)
