package com.mcsnowflake.worktimer.state

import com.mcsnowflake.worktimer.state.SessionData.Type.LONG_BREAK
import com.mcsnowflake.worktimer.state.SessionData.Type.SHORT_BREAK
import com.mcsnowflake.worktimer.state.SessionData.Type.WORK_SESSION
import io.kotest.core.spec.style.BehaviorSpec
import io.kotest.datatest.withData
import io.kotest.matchers.shouldBe
import kotlin.time.Clock
import kotlin.time.Duration.Companion.minutes
import kotlin.time.ExperimentalTime

@OptIn(ExperimentalTime::class)
class SessionDataTest : BehaviorSpec() {

    init {
        given("SessionData") {
            `when`("creating sessions") {
                data class SessionCreationTest(
                    val type: SessionData.Type,
                    val repetition: Int
                ) {
                    override fun toString() = "${type.name} with repetition $repetition"
                }

                withData(
                    SessionCreationTest(WORK_SESSION, 0),
                    SessionCreationTest(SHORT_BREAK, 1),
                    SessionCreationTest(LONG_BREAK, 2),
                    SessionCreationTest(WORK_SESSION, 5)
                ) { testCase ->
                    val start = Clock.System.now()
                    val end = start + 25.minutes

                    val session = SessionData(testCase.type, start, end, testCase.repetition)

                    session.type shouldBe testCase.type
                    session.start shouldBe start
                    session.end shouldBe end
                    session.repetition shouldBe testCase.repetition
                }
            }

            `when`("prolonging sessions") {
                data class ProlongTest(
                    val type: SessionData.Type,
                    val prolongDuration: kotlin.time.Duration
                ) {
                    override fun toString() = "${type.name} prolonged by ${prolongDuration.inWholeMinutes}min"
                }

                withData(
                    ProlongTest(WORK_SESSION, 5.minutes),
                    ProlongTest(SHORT_BREAK, 2.minutes),
                    ProlongTest(LONG_BREAK, 10.minutes),
                    ProlongTest(WORK_SESSION, 15.minutes)
                ) { testCase ->
                    val start = Clock.System.now()
                    val originalEnd = start + 25.minutes
                    val session = SessionData(testCase.type, start, originalEnd, 1)

                    val prolongedSession = session.prolongedBy(testCase.prolongDuration)

                    prolongedSession.type shouldBe session.type
                    prolongedSession.start shouldBe session.start
                    prolongedSession.repetition shouldBe session.repetition
                    prolongedSession.end shouldBe (originalEnd + testCase.prolongDuration)
                }
            }

            `when`("using companion object") {
                then("should create new work session with default values") {
                    val start = Clock.System.now()
                    val end = start + 30.minutes

                    val session = SessionData.newSession(start, end)

                    session.type shouldBe WORK_SESSION
                    session.start shouldBe start
                    session.end shouldBe end
                    session.repetition shouldBe 0
                }
            }

            `when`("testing session types") {
                withData(
                    WORK_SESSION,
                    SHORT_BREAK,
                    LONG_BREAK
                ) { sessionType ->
                    val start = Clock.System.now()
                    val end = start + 20.minutes

                    val session = SessionData(sessionType, start, end, 0)

                    session.type shouldBe sessionType
                }
            }
        }
    }
}
