package com.mcsnowflake.worktimer.ui.views.credits

import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.DesignServices
import androidx.compose.material.icons.filled.Extension
import androidx.compose.material.icons.outlined.Lightbulb
import androidx.compose.material.icons.outlined.MusicNote
import androidx.compose.material3.HorizontalDivider
import androidx.compose.material3.Icon
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.PrimaryTabRow
import androidx.compose.material3.Surface
import androidx.compose.material3.Tab
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableIntStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.vector.ImageVector
import androidx.compose.ui.platform.LocalUriHandler
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.ExperimentalTextApi
import androidx.compose.ui.text.LinkAnnotation
import androidx.compose.ui.text.SpanStyle
import androidx.compose.ui.text.TextLinkStyles
import androidx.compose.ui.text.TextStyle
import androidx.compose.ui.text.buildAnnotatedString
import androidx.compose.ui.text.font.FontFamily
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.text.style.TextDecoration
import androidx.compose.ui.text.withLink
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import com.mcsnowflake.worktimer.R
import com.mcsnowflake.worktimer.ui.components.appbar.SimpleAppBar
import com.mikepenz.aboutlibraries.ui.compose.android.produceLibraries
import com.mikepenz.aboutlibraries.ui.compose.m3.LibrariesContainer

const val CREDITS = "credits"

@Preview
@Composable
fun Credits(navigateBack: () -> Unit = {}) {

    var tabIndex by remember { mutableIntStateOf(0) }
    val tabs = listOf(stringResource(id = R.string.credits_tab_creators), stringResource(id = R.string.credits_tab_libraries))
    val libs by produceLibraries(R.raw.aboutlibraries)

    Column(modifier = Modifier.fillMaxWidth()) {
        SimpleAppBar(stringResource(id = R.string.credits), navigateBack)
        PrimaryTabRow(selectedTabIndex = tabIndex) {
            tabs.forEachIndexed { index, title ->
                Tab(
                    text = { Text(title) },
                    selected = tabIndex == index,
                    onClick = { tabIndex = index },
                    icon = {
                        when (index) {
                            0 -> Icon(imageVector = Icons.Default.DesignServices, contentDescription = null)
                            1 -> Icon(imageVector = Icons.Default.Extension, contentDescription = null)
                        }
                    }
                )
            }
        }
        when (tabIndex) {
            0 -> Creators()
            1 -> LibrariesContainer(libs, Modifier.fillMaxSize())
        }
    }
}

@Composable
fun Creators() = Surface {
    Column(
        modifier = Modifier.fillMaxSize(),
        verticalArrangement = Arrangement.Top,
        horizontalAlignment = Alignment.CenterHorizontally
    ) {
        CreditCard(
            stringResource(id = R.string.credits_pomodoro_title),
            Icons.Outlined.Lightbulb,
            stringResource(id = R.string.credits_pomodoro_description),
            "https://www.pomodorotechnique.com/",
            stringResource(id = R.string.credits_pomodoro_affiliation)
        )
        HorizontalDivider()
        CreditCard(
            stringResource(id = R.string.credits_sound_title),
            Icons.Outlined.MusicNote,
            stringResource(id = R.string.credits_sound_description),
            "https://lex-filmton.de"
        )
        HorizontalDivider()
    }
}

@OptIn(ExperimentalTextApi::class)
@Composable
fun CreditCard(
    title: String,
    icon: ImageVector,
    description: String,
    linkText: String,
    subText: String = ""
) = Column(
    modifier = Modifier
        .fillMaxWidth()
        .padding(16.dp),
    verticalArrangement = Arrangement.spacedBy(8.dp, Alignment.CenterVertically),
    horizontalAlignment = Alignment.CenterHorizontally
) {
    val uriHandler = LocalUriHandler.current

    Row(
        horizontalArrangement = Arrangement.spacedBy(8.dp, Alignment.CenterHorizontally),
        verticalAlignment = Alignment.CenterVertically
    ) {
        Icon(icon, title, Modifier.size(32.dp))
        Text(
            text = title,
            color = MaterialTheme.colorScheme.tertiary,
            fontFamily = FontFamily.Serif,
            style = MaterialTheme.typography.headlineSmall
        )
    }
    Spacer(modifier = Modifier.height(4.dp))
    Text(text = description, textAlign = TextAlign.Center)
    val linkStyle = SpanStyle(
        color = MaterialTheme.colorScheme.tertiary,
        textDecoration = TextDecoration.Underline,
        fontSize = 15.sp
    )

    val link =
        LinkAnnotation.Url(
            linkText,
            TextLinkStyles(linkStyle),
        ) { uriHandler.openUri((it as LinkAnnotation.Url).url) }
    val annotatedString1 = buildAnnotatedString {
        withLink(link) { append(linkText) }
    }

    Text(
        text = annotatedString1,
        modifier = Modifier.fillMaxWidth(),
        style = TextStyle(fontSize = 12.sp, textAlign = TextAlign.Center),
    )
    if (subText.isNotBlank()) Text(
        text = subText,
        style = MaterialTheme.typography.bodySmall.copy(textAlign = TextAlign.Center),
    )
}
