package com.mcsnowflake.worktimer.notifications

import android.app.NotificationManager
import com.mcsnowflake.worktimer.configuration.ConfigurationStore
import com.mcsnowflake.worktimer.configuration.Preference
import com.mcsnowflake.worktimer.notifications.NotificationBuilder.Companion.HYDRATION_NOTIFICATION_ID
import com.mcsnowflake.worktimer.notifications.NotificationBuilder.Companion.TIMER_NOTIFICATION_ID
import com.mcsnowflake.worktimer.state.TimerState
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.drop
import kotlinx.coroutines.launch

class NotificationHandler(
    private val builder: NotificationBuilder,
    private val notificationManager: NotificationManager,
    configuration: ConfigurationStore,
    private val timerState: StateFlow<TimerState>,
    coroutineScope: CoroutineScope
) {

    private val areNotificationsOn = configuration.getAsStateFlow(Preference.NOTIFICATIONS_ON, coroutineScope)

    init {
        coroutineScope.launch {
            timerState.drop(1).collect {
                if (areNotificationsOn.value) updateNotifications(it)
            }
        }
    }

    private suspend fun updateNotifications(event: TimerState) = when (event) {
        is TimerState.Session.Running -> updateRunningNotification(event)
        is TimerState.Session.Finished -> sendAlarmNotification(event)
        is TimerState.Stopped -> clearNotifications()
    }

    fun sendHydrationNotification() = notificationManager.notify(
        HYDRATION_NOTIFICATION_ID,
        builder.buildHydrateNotification()
    )

    private fun updateRunningNotification(update: TimerState.Session.Running) = notificationManager.notify(
        TIMER_NOTIFICATION_ID,
        builder.buildRunningNotification(update)
    )

    private suspend fun sendAlarmNotification(event: TimerState.Session.Finished) = notificationManager.notify(
        TIMER_NOTIFICATION_ID,
        builder.buildAlarmNotification(event)
    )

    private fun clearNotifications() = notificationManager.cancelAll()
}
