@file:OptIn(ExperimentalTime::class)

package com.mcsnowflake.worktimer.ui.components.timer

import android.text.format.DateUtils.formatElapsedTime
import androidx.compose.animation.AnimatedContent
import androidx.compose.animation.AnimatedContentTransitionScope.SlideDirection.Companion.Down
import androidx.compose.animation.AnimatedContentTransitionScope.SlideDirection.Companion.Left
import androidx.compose.animation.AnimatedContentTransitionScope.SlideDirection.Companion.Right
import androidx.compose.animation.AnimatedContentTransitionScope.SlideDirection.Companion.Up
import androidx.compose.animation.core.FastOutLinearInEasing
import androidx.compose.animation.core.tween
import androidx.compose.animation.fadeIn
import androidx.compose.animation.fadeOut
import androidx.compose.animation.togetherWith
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.State
import androidx.compose.runtime.derivedStateOf
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalConfiguration
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import androidx.core.os.ConfigurationCompat
import com.mcsnowflake.worktimer.R
import com.mcsnowflake.worktimer.configuration.TimerInput
import com.mcsnowflake.worktimer.state.SessionData
import com.mcsnowflake.worktimer.state.SessionData.Type.LONG_BREAK
import com.mcsnowflake.worktimer.state.SessionData.Type.SHORT_BREAK
import com.mcsnowflake.worktimer.state.SessionData.Type.WORK_SESSION
import com.mcsnowflake.worktimer.state.TimerState
import com.mcsnowflake.worktimer.state.TimerState.Session
import com.mcsnowflake.worktimer.state.TimerState.Stopped
import com.mcsnowflake.worktimer.ui.views.dashboard.SetupViewModel
import com.mcsnowflake.worktimer.ui.views.dashboard.SuffixTransformation
import java.util.Locale
import kotlin.math.ceil
import kotlin.time.Clock
import kotlin.time.Duration
import kotlin.time.Duration.Companion.seconds
import kotlin.time.ExperimentalTime
import kotlinx.coroutines.delay
import org.koin.compose.viewmodel.koinActivityViewModel

@Composable
fun LowerBar(
    modifier: Modifier = Modifier,
    timerState: State<TimerState> = mutableStateOf(Stopped)
) = AnimatedContent(
    targetState = timerState.value,
    modifier = modifier.padding(top = 6.dp),
    contentAlignment = Alignment.TopEnd,
    transitionSpec = {
        when {

            initialState is Session.Running && targetState is Stopped ->
                slideIntoContainer(Down, animationSpec = tween(500, 500)) togetherWith
                    slideOutOfContainer(Right, animationSpec = tween(500, easing = FastOutLinearInEasing))

            initialState is Stopped && targetState is Session.Running ->
                slideIntoContainer(Left, animationSpec = tween(500, 300)) togetherWith
                    slideOutOfContainer(Up, animationSpec = tween(300, easing = FastOutLinearInEasing))

            else -> fadeIn() togetherWith fadeOut()
        }
    }) { state ->
    when (state) {
        is Session.Finished -> Spacer(Modifier.fillMaxWidth())
        is Session.Running -> RemainingTimeDisplay(state.session)
        is Stopped -> LowerInputs()
    }
}

@Composable
fun LowerInputs(
    viewModel: SetupViewModel = koinActivityViewModel()
) {
    Row(
        horizontalArrangement = Arrangement.spacedBy(32.dp),
        verticalAlignment = Alignment.CenterVertically
    ) {
        NumberInput(
            value = viewModel[TimerInput.SHORT_BREAK_DURATION],
            onValueChange = { viewModel[TimerInput.SHORT_BREAK_DURATION] = it.trim() },
            modifier = Modifier.weight(1F),
            label = { Text(text = stringResource(id = R.string.short_break_duration_label)) },
            visualTransformation = SuffixTransformation(stringResource(id = R.string.unit_text)),
            accentColor = MaterialTheme.colorScheme.tertiary
        )
        NumberInput(
            value = viewModel[TimerInput.LONG_BREAK_DURATION],
            onValueChange = { viewModel[TimerInput.LONG_BREAK_DURATION] = it.trim() },
            modifier = Modifier.weight(1F),
            label = { Text(text = stringResource(id = R.string.long_break_duration_label)) },
            visualTransformation = SuffixTransformation(stringResource(id = R.string.unit_text)),
            accentColor = MaterialTheme.colorScheme.tertiary
        )
    }
}

@Composable
private fun RemainingTimeDisplay(session: SessionData) {
    val primaryColor = MaterialTheme.colorScheme.primary
    val tertiaryColor = MaterialTheme.colorScheme.tertiary

    val infoTextColor by remember {
        derivedStateOf {
            when (session.type) {
                WORK_SESSION -> primaryColor
                SHORT_BREAK, LONG_BREAK -> tertiaryColor
            }
        }
    }

    val configuration = LocalConfiguration.current
    val locale = remember {
        ConfigurationCompat.getLocales(configuration).get(0) ?: Locale.getDefault()
    }
    var remainingTime by remember {
        mutableStateOf(getFormattedTime(session, locale))
    }

    LaunchedEffect(session) {
        while (true) {
            remainingTime = getFormattedTime(session, locale)
            delay(1.seconds)
        }
    }

    Text(
        text = remainingTime,
        modifier = Modifier.fillMaxWidth(),
        color = infoTextColor,
        textAlign = TextAlign.Right,
        fontSize = 20.sp
    )
}

private fun getFormattedTime(session: SessionData, locale: Locale): String =
    (session.end - Clock.System.now()).format(locale)

private fun Duration.format(locale: Locale): String {
    val seconds = ceil(inWholeSeconds.toFloat()).toInt()
    return when {
        inWholeHours >= 1 -> "${formatElapsedTime(inWholeSeconds)} h"
        inWholeMinutes < 1 -> String.format(locale, "%d sec", seconds)
        else -> String.format(locale, "%d min %d sec", seconds / 60, (seconds % 60))
    }
}
