package com.mcsnowflake.worktimer.ui.components.buttons

import androidx.compose.animation.core.Spring
import androidx.compose.animation.core.VisibilityThreshold
import androidx.compose.animation.core.spring
import androidx.compose.animation.core.tween
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.material3.Button
import androidx.compose.material3.ButtonDefaults
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.unit.IntOffset
import androidx.compose.ui.unit.dp
import com.mcsnowflake.worktimer.state.SessionData.Type.WORK_SESSION
import com.mcsnowflake.worktimer.state.TimerEvent
import com.mcsnowflake.worktimer.ui.views.dashboard.DashBoardViewModel
import com.mcsnowflake.worktimer.ui.views.dashboard.SetupViewModel
import kotlinx.coroutines.flow.MutableSharedFlow
import org.koin.compose.koinInject
import org.koin.compose.viewmodel.koinActivityViewModel

@Composable
fun ActionButtons(
    modifier: Modifier = Modifier,
    dashBoardViewModel: DashBoardViewModel,
    events: MutableSharedFlow<TimerEvent> = koinInject(),
    viewModel: SetupViewModel = koinActivityViewModel()
) {

    val nextLabel by dashBoardViewModel.nextButtonLabel.collectAsState()
    val moreLabel by dashBoardViewModel.moreButtonLabel.collectAsState()
    val timerState by dashBoardViewModel.currentSessionData.collectAsState()
    val isRunning by dashBoardViewModel.isRunning.collectAsState()
    val showNextButton by dashBoardViewModel.showNextButton.collectAsState()
    val showMoreButton by dashBoardViewModel.showMoreButton.collectAsState()
    val workSessionColors = ButtonDefaults.buttonColors(MaterialTheme.colorScheme.primary, MaterialTheme.colorScheme.onPrimary)
    val breakColors = ButtonDefaults.buttonColors(MaterialTheme.colorScheme.tertiary, MaterialTheme.colorScheme.onTertiary)

    LazyColumn(
        modifier = modifier
            .fillMaxWidth()
            .height(200.dp),
        verticalArrangement = Arrangement.SpaceEvenly,
        horizontalAlignment = Alignment.CenterHorizontally
    ) {

        if (showNextButton) item(1) {
            Button(
                { events.tryEmit(TimerEvent.NEXT) },
                Modifier
                    .width(200.dp)
                    .animateItem(
                        fadeInSpec = tween(delayMillis = 500),
                        fadeOutSpec = spring(stiffness = Spring.StiffnessHigh),
                    ),
                true,
                colors = if (timerState.type != WORK_SESSION) workSessionColors else breakColors
            ) {
                Text(stringResource(nextLabel))
            }
        }
        if (showMoreButton) item(2) {
            Button(
                { events.tryEmit(TimerEvent.MORE) },
                Modifier
                    .width(200.dp)
                    .animateItem(
                        fadeInSpec = tween(delayMillis = 500),
                        fadeOutSpec = spring(stiffness = Spring.StiffnessHigh),
                    ),
                true,
                colors = if (timerState.type == WORK_SESSION) workSessionColors else breakColors
            ) { Text(stringResource(moreLabel)) }
        }
        item(3) {
            StartStopButton(
                modifier = Modifier.animateItem(
                    placementSpec = spring(
                        dampingRatio = Spring.DampingRatioLowBouncy,
                        stiffness = Spring.StiffnessMediumLow,
                        visibilityThreshold = IntOffset.VisibilityThreshold,
                    ),
                ),
                isTimerRunning = isRunning,
                onClick = {
                    if (isRunning) events.tryEmit(TimerEvent.STOP)
                    else events.tryEmit(TimerEvent.START)
                },
                enabled = viewModel.isInputValid.collectAsState()
            )
        }
    }
}
