@file:OptIn(ExperimentalTime::class)

package com.mcsnowflake.worktimer.state

import android.util.Log
import com.mcsnowflake.worktimer.configuration.ConfigurationStore
import com.mcsnowflake.worktimer.configuration.LateAlarmStatus
import com.mcsnowflake.worktimer.configuration.Preference
import com.mcsnowflake.worktimer.configuration.StateStore
import com.mcsnowflake.worktimer.configuration.StateStore.Companion.LATE_ALARMS
import com.mcsnowflake.worktimer.configuration.StateStore.Companion.TIMER_STATE
import com.mcsnowflake.worktimer.configuration.TimerInput
import com.mcsnowflake.worktimer.configuration.TimerInput.OVERTIME_DURATION
import com.mcsnowflake.worktimer.state.SessionData.Type.LONG_BREAK
import com.mcsnowflake.worktimer.state.SessionData.Type.SHORT_BREAK
import com.mcsnowflake.worktimer.state.SessionData.Type.WORK_SESSION
import com.mcsnowflake.worktimer.state.TimerState.Session
import com.mcsnowflake.worktimer.state.TimerState.Session.Running
import com.mcsnowflake.worktimer.state.TimerState.Stopped
import kotlin.time.Clock
import kotlin.time.Duration.Companion.seconds
import kotlin.time.ExperimentalTime
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.launch
import kotlinx.serialization.json.Json

/**
 * Lifecycle-aware timer state machine that
 * - exposes a StateFlow<TimerState> for UI
 * - persists current state (start/end/session/repetitions)
 * - updates countdown notification and schedules alarms
 * - can operate without a foreground service
 */
class TimerStateMachine(
    private val configuration: ConfigurationStore,
    private val stateStore: StateStore,
    startState: TimerState,
    val events: Flow<TimerEvent>,
    private val scope: CoroutineScope = CoroutineScope(SupervisorJob() + Dispatchers.Default)
) {

    private val _state = MutableStateFlow(startState)
    val state: StateFlow<TimerState> = _state.asStateFlow()

    init {
        scope.launch {
            events.collect { event ->
                process(event, state.value)
            }
        }
    }

    private fun process(event: TimerEvent, state: TimerState) = when (state) {
        is Session if event == TimerEvent.NEXT -> state.startNextSession()
        is Session if event == TimerEvent.MORE -> state.prolongSession()
        is Session if event == TimerEvent.STOP -> state.stopSession()
        is Stopped if event == TimerEvent.START -> state.startNewSession()
        is Running if event == TimerEvent.FINISH -> state.finishSession()
        else -> Log.e("TimerStateMachine", "unexpected transition: $event in $state")
    }

    private fun Running.finishSession() {
        scope.launch {
            handleLateAlarms()
            if (configuration[Preference.AUTO_SWITCH_MODE]) startNextSession()
            else setState(Session.Finished(session))
        }
    }

    private suspend fun Running.handleLateAlarms() {
        if (stateStore[LATE_ALARMS] != LateAlarmStatus.LATE_ALARMS_DISMISSED.name && (session.end - Clock.System.now()).absoluteValue > 5.seconds)
            stateStore[LATE_ALARMS] = LateAlarmStatus.LATE_ALARMS_REGISTERED.name
    }

    private fun Stopped.startNewSession() {
        scope.launch {
            setState(Running(createSession(WORK_SESSION, 0)))
        }
    }

    private fun Session.stopSession() {
        scope.launch { setState(Stopped) }
    }

    fun Session.startNextSession() {
        scope.launch {
            val nextSession = when (session.type) {
                WORK_SESSION -> {
                    val shortReps = configuration[TimerInput.SHORT_BREAK_REPETITIONS]
                    if (session.repetition >= shortReps) createSession(LONG_BREAK, 0)
                    else createSession(SHORT_BREAK, session.repetition + 1)
                }

                SHORT_BREAK, LONG_BREAK -> createSession(WORK_SESSION, session.repetition + 1)
            }
            setState(Running(nextSession))
        }
    }

    private fun Session.prolongSession() {
        scope.launch {
            val updatedState = Running(session.prolongedBy(configuration.getDuration(OVERTIME_DURATION)))
            setState(updatedState)
        }
    }

    private suspend fun createSession(type: SessionData.Type, repetition: Int): SessionData {
        val duration = when (type) {
            WORK_SESSION -> configuration.getDuration(TimerInput.WORK_SESSION_DURATION)
            SHORT_BREAK -> configuration.getDuration(TimerInput.SHORT_BREAK_DURATION)
            LONG_BREAK -> configuration.getDuration(TimerInput.LONG_BREAK_DURATION)
        }
        return SessionData(type, Clock.System.now(), Clock.System.now() + duration, repetition)
    }

    private suspend fun setState(state: TimerState) {
        _state.emit(state)
        stateStore[TIMER_STATE] = Json.encodeToString(state).encodeToByteArray()
    }
}


