@file:OptIn(ExperimentalTime::class)

package com.mcsnowflake.worktimer.notifications

import android.app.Notification
import android.app.NotificationChannel
import android.app.NotificationManager
import android.app.PendingIntent
import android.content.ContentResolver
import android.content.Context
import android.content.Intent
import android.media.AudioAttributes
import androidx.core.app.NotificationCompat
import androidx.core.app.NotificationManagerCompat
import androidx.core.net.toUri
import com.mcsnowflake.worktimer.MainActivity
import com.mcsnowflake.worktimer.R
import com.mcsnowflake.worktimer.TextHandler
import com.mcsnowflake.worktimer.configuration.ConfigurationStore
import com.mcsnowflake.worktimer.configuration.Preference.OVERTIME_MODE
import com.mcsnowflake.worktimer.state.SessionData.Type.LONG_BREAK
import com.mcsnowflake.worktimer.state.SessionData.Type.SHORT_BREAK
import com.mcsnowflake.worktimer.state.SessionData.Type.WORK_SESSION
import com.mcsnowflake.worktimer.state.TimerState
import kotlin.time.Duration.Companion.seconds
import kotlin.time.ExperimentalTime

class NotificationBuilder(
    private val applicationContext: Context,
    private val configuration: ConfigurationStore
) {

    val pendingIntent: PendingIntent = PendingIntent.getActivity(
        applicationContext,
        PENDING_REQUEST_CODE,
        Intent(applicationContext, MainActivity::class.java)
            .addCategory(Intent.CATEGORY_LAUNCHER)
            .setAction("android.intent.action.MAIN"),
        PendingIntent.FLAG_IMMUTABLE
    )

    internal fun buildRunningNotification(update: TimerState.Session): Notification =
        NotificationCompat.Builder(applicationContext, ALARM_CHANNEL_ID)
            .setContentIntent(pendingIntent)
            .setSmallIcon(R.drawable.ic_logo_simple)
            .setSound((ContentResolver.SCHEME_ANDROID_RESOURCE + "://" + applicationContext.packageName + "/" + R.raw.gong1).toUri())
            .setCategory(Notification.CATEGORY_PROGRESS)
            .setContentTitle(applicationContext.getString(TextHandler.getTitle(update)))
            .setWhen(update.session.end.toEpochMilliseconds())
            .setShowWhen(true)
            .setUsesChronometer(true)
            .setChronometerCountDown(true)
            .setOngoing(true)
            .addAction(buildNotificationAction(NotificationAction.STOP))
            .addAction(
                when (update.session.type) {
                    WORK_SESSION -> buildNotificationAction(NotificationAction.PAUSE)
                    SHORT_BREAK, LONG_BREAK -> buildNotificationAction(NotificationAction.WORK)
                }
            ).build()

    internal suspend fun buildAlarmNotification(update: TimerState.Session): Notification =
        NotificationCompat.Builder(applicationContext, ALARM_CHANNEL_ID)
            .setContentIntent(pendingIntent)
            .setSmallIcon(R.drawable.ic_logo_simple)
            .setSound((ContentResolver.SCHEME_ANDROID_RESOURCE + "://" + applicationContext.packageName + "/" + R.raw.gong2).toUri())
            .setAllowSystemGeneratedContextualActions(false)
            .setCategory(Notification.CATEGORY_ALARM)
            .setContentTitle(applicationContext.getString(TextHandler.getTitle(update)))
            .setContentText(applicationContext.getString(TextHandler.getInfoText(update.session.type)))
            .setWhen(update.session.end.toEpochMilliseconds())
            .setShowWhen(true)
            .setUsesChronometer(true)
            .setOngoing(true)
            .addMoreAction()
            .addAction(buildNotificationAction(NotificationAction.STOP))
            .addAction(buildNotificationAction(NotificationAction.RESUME))
            .build()

    private suspend fun NotificationCompat.Builder.addMoreAction() = if (configuration[OVERTIME_MODE]) {
        addAction(buildNotificationAction(NotificationAction.MORE))
    } else this

    internal fun buildHydrateNotification() =
        NotificationCompat.Builder(applicationContext, HYDRATE_CHANNEL_ID)
            .setAutoCancel(true)
            .setTimeoutAfter(15.seconds.inWholeMilliseconds)
            .setSmallIcon(R.drawable.ic_drop)
            .setCategory(Notification.CATEGORY_ALARM)
            .setContentTitle(applicationContext.getString(R.string.hydration_notification_title))
            .build()

    private fun buildNotificationAction(action: NotificationAction)
        : NotificationCompat.Action = NotificationCompat.Action.Builder(
        action.getActionIcon(),
        applicationContext.getString(action.getActionTitle()),
        PendingIntent.getBroadcast(
            applicationContext,
            PENDING_REQUEST_CODE + action.ordinal,
            Intent(applicationContext, ActionReceiver::class.java).apply {
                setAction(action.name)
            },
            PendingIntent.FLAG_UPDATE_CURRENT or PendingIntent.FLAG_IMMUTABLE
        )
    ).build()

    companion object {
        const val PENDING_REQUEST_CODE = 52334
        const val TIMER_NOTIFICATION_ID = 2345
        const val HYDRATION_NOTIFICATION_ID = 5346
        const val ALARM_CHANNEL_ID = "47657"
        const val HYDRATE_CHANNEL_ID = "5253432"

        fun createChannelsWith(applicationContext: Context) {
            NotificationManagerCompat.from(applicationContext).createNotificationChannel(
                NotificationChannel(
                    ALARM_CHANNEL_ID,
                    applicationContext.getString(R.string.notification_channel_name),
                    NotificationManager.IMPORTANCE_HIGH
                ).apply {
                    lockscreenVisibility = NotificationCompat.VISIBILITY_PUBLIC
                    setShowBadge(true)
                    setSound(
                        (ContentResolver.SCHEME_ANDROID_RESOURCE + "://" + applicationContext.packageName + "/" + R.raw.gong2).toUri(),
                        AudioAttributes.Builder().setUsage(AudioAttributes.USAGE_NOTIFICATION).build()
                    )
                }
            )
            NotificationManagerCompat.from(applicationContext).createNotificationChannel(
                NotificationChannel(
                    HYDRATE_CHANNEL_ID,
                    applicationContext.getString(R.string.hydration_channel_name),
                    NotificationManager.IMPORTANCE_HIGH
                ).apply {
                    lockscreenVisibility = NotificationCompat.VISIBILITY_PUBLIC
                    setShowBadge(false)
                    setSound(
                        (ContentResolver.SCHEME_ANDROID_RESOURCE + "://" + applicationContext.packageName + "/" + R.raw.drop).toUri(),
                        AudioAttributes.Builder().setUsage(AudioAttributes.USAGE_NOTIFICATION).build()
                    )
                }
            )
        }
    }
}
