package com.mcsnowflake.worktimer.notifications

import android.app.NotificationManager
import android.util.Log
import com.mcsnowflake.worktimer.configuration.ConfigurationStore
import com.mcsnowflake.worktimer.configuration.Preference
import com.mcsnowflake.worktimer.state.SessionData.Type.WORK_SESSION
import com.mcsnowflake.worktimer.state.TimerState
import com.mcsnowflake.worktimer.state.TimerState.Session.Running
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.drop
import kotlinx.coroutines.launch

class DndModeHandler(
    private val notificationManager: NotificationManager,
    configuration: ConfigurationStore,
    private val timerState: StateFlow<TimerState>,
    coroutineScope: CoroutineScope = CoroutineScope(Dispatchers.IO + SupervisorJob())
) {

    private var previousSetting = NotificationManager.INTERRUPTION_FILTER_UNKNOWN
    private val isDndModeOn = configuration.getAsStateFlow(Preference.DND_MODE, coroutineScope)

    init {

        coroutineScope.launch {
            isDndModeOn.collect {
                when (timerState.value) {
                    is Running -> if (it) activateDnd() else deactivateDnd()
                    else -> {}
                }
            }
        }
        coroutineScope.launch {
            timerState.drop(1).collect {
                if (isDndModeOn.value) process(it)
            }
        }
    }

    private fun process(event: TimerState) {
        Log.d(javaClass.simpleName, event.toString())
        when (event) {
            is Running if event.session.type == WORK_SESSION -> activateDnd()
            is Running if event.session.type != WORK_SESSION -> deactivateDnd()
            is TimerState.Session.Finished if event.session.type == WORK_SESSION -> deactivateDnd()
            is TimerState.Stopped -> deactivateDnd()
            else -> {}
        }
    }

    private fun activateDnd() {
        previousSetting = notificationManager.currentInterruptionFilter
        notificationManager.setInterruptionFilter(NotificationManager.INTERRUPTION_FILTER_PRIORITY)
        Log.d(javaClass.simpleName, "DND enabled")
    }

    private fun deactivateDnd() {
        if (previousSetting != NotificationManager.INTERRUPTION_FILTER_UNKNOWN) {
            notificationManager.setInterruptionFilter(previousSetting)
            previousSetting = NotificationManager.INTERRUPTION_FILTER_UNKNOWN
            Log.d(javaClass.simpleName, "InterruptionFilter reset")
        } else Log.d(javaClass.simpleName, "InterruptionFilter not reset, NotificationManager.INTERRUPTION_FILTER_UNKNOWN")
    }

    // TODO check for permission first -> warning toast if no permission + DND mode reset
    //    fun recheckDND() {
//        Log.d(javaClass.simpleName, "recheck dnd")
//        if (preferences.getBoolean("wants_dnd", false) && !internalStates[DND_MODE_KEY]!!.value && notificationManager.isNotificationPolicyAccessGranted) {
//            set(DND_MODE_KEY, true)
//            with(preferences.edit()) {
//                remove("wants_dnd")
//                commit()
//            }
//        }
//    }
}
