package com.mcsnowflake.worktimer

import android.content.Context
import android.os.Bundle
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.core.view.WindowCompat
import androidx.lifecycle.coroutineScope
import androidx.navigation.NavType
import androidx.navigation.compose.NavHost
import androidx.navigation.compose.composable
import androidx.navigation.compose.rememberNavController
import androidx.navigation.navArgument
import com.mcsnowflake.worktimer.configuration.ConfigurationStore
import com.mcsnowflake.worktimer.configuration.Preference
import com.mcsnowflake.worktimer.configuration.StateStore
import com.mcsnowflake.worktimer.configuration.StateStore.Companion.FIRST_START
import com.mcsnowflake.worktimer.configuration.StateStore.Companion.VERSION
import com.mcsnowflake.worktimer.ui.components.SetInsetsColor
import com.mcsnowflake.worktimer.ui.components.screenLock.KeepScreenOn
import com.mcsnowflake.worktimer.ui.dialogs.DndPermissionDialog
import com.mcsnowflake.worktimer.ui.dialogs.LateAlarmsDialog
import com.mcsnowflake.worktimer.ui.theme.Theme
import com.mcsnowflake.worktimer.ui.views.STATS_VIEW
import com.mcsnowflake.worktimer.ui.views.StatsView
import com.mcsnowflake.worktimer.ui.views.about.ABOUT_SCREEN
import com.mcsnowflake.worktimer.ui.views.about.AboutScreen
import com.mcsnowflake.worktimer.ui.views.about.SHOW_APP_BAR
import com.mcsnowflake.worktimer.ui.views.about.showAppBar
import com.mcsnowflake.worktimer.ui.views.credits.CREDITS
import com.mcsnowflake.worktimer.ui.views.credits.Credits
import com.mcsnowflake.worktimer.ui.views.dashboard.DASHBOARD
import com.mcsnowflake.worktimer.ui.views.dashboard.Dashboard
import com.mcsnowflake.worktimer.ui.views.settings.PREFERENCES_SCREEN
import com.mcsnowflake.worktimer.ui.views.settings.Settings
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.runBlocking
import org.koin.android.ext.android.inject
import org.koin.compose.koinInject

class MainActivity : ComponentActivity() {
    private val configurationStore: ConfigurationStore by inject<ConfigurationStore>()
    private val stateStore: StateStore by inject()

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        WindowCompat.enableEdgeToEdge(window)
        WindowCompat.setDecorFitsSystemWindows(window, false)

        checkForOldSettingsAndConvertToDataStore()

        val getAppVersion = applicationContext.getAppVersion()
        val startScreen = runBlocking { selectStartScreen(getAppVersion) }

        setContent {
            KeepScreenOn()
            SetInsetsColor(window)
            val configuration: ConfigurationStore = koinInject<ConfigurationStore>()
            val dynamicColorOn by configuration.getAsFlow(Preference.DYNAMIC_COLOR_ON).collectAsState(false)
            Theme(dynamicColor = dynamicColorOn) {
                DndPermissionDialog()
                LateAlarmsDialog()

                val navController = rememberNavController()
                NavHost(navController, startScreen) {
                    composable(
                        ABOUT_SCREEN,
                        listOf(navArgument(SHOW_APP_BAR) { type = NavType.BoolType })
                    ) { AboutScreen(navController, it.arguments?.getBoolean(SHOW_APP_BAR) ?: false) }
                    composable(CREDITS) { Credits { navController.popBackStack() } }
                    composable(STATS_VIEW) { StatsView { navController.popBackStack() } }
                    composable(DASHBOARD) { Dashboard(navigateTo = { navController.navigate(it) }) }
                    composable(PREFERENCES_SCREEN) { Settings(navigateBack = { navController.popBackStack() }) }
                }
            }
        }
    }

    private suspend fun selectStartScreen(currentVersion: String): String {
        val firstStart = stateStore[FIRST_START] ?: true
        val storedVersion = stateStore[VERSION] ?: ""
        if (storedVersion != currentVersion) stateStore[VERSION] = currentVersion

        return if (firstStart) {
            stateStore[FIRST_START] = false
            ABOUT_SCREEN.showAppBar(false)
        } else {
            DASHBOARD
        }
    }

    private fun checkForOldSettingsAndConvertToDataStore() = lifecycle.coroutineScope.launch(Dispatchers.IO) {
        configurationStore.recover(applicationContext.getSharedPreferences(applicationContext.packageName + "_preferences", MODE_PRIVATE))
    }

    private fun Context.getAppVersion(): String = runCatching {
        packageManager.getPackageInfo(packageName, 0).versionName?.substringBefore('-')
            ?: "N/A"
    }.getOrElse {
        "Unknown Version"
    }
}
