package com.mcsnowflake.worktimer.ui.components.timer

import androidx.compose.animation.AnimatedContent
import androidx.compose.animation.AnimatedContentTransitionScope.SlideDirection.Companion.Down
import androidx.compose.animation.AnimatedContentTransitionScope.SlideDirection.Companion.Left
import androidx.compose.animation.AnimatedContentTransitionScope.SlideDirection.Companion.Right
import androidx.compose.animation.AnimatedContentTransitionScope.SlideDirection.Companion.Up
import androidx.compose.animation.core.FastOutLinearInEasing
import androidx.compose.animation.core.tween
import androidx.compose.animation.fadeIn
import androidx.compose.animation.fadeOut
import androidx.compose.animation.togetherWith
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.wrapContentHeight
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.State
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import com.mcsnowflake.worktimer.R
import com.mcsnowflake.worktimer.configuration.TimerInput
import com.mcsnowflake.worktimer.state.TimerState
import com.mcsnowflake.worktimer.state.TimerState.Session
import com.mcsnowflake.worktimer.state.TimerState.Stopped
import com.mcsnowflake.worktimer.ui.views.dashboard.SetupViewModel
import com.mcsnowflake.worktimer.ui.views.dashboard.SuffixTransformation
import kotlin.time.Duration.Companion.seconds
import kotlinx.coroutines.delay
import org.koin.compose.viewmodel.koinActivityViewModel

@Composable
fun UpperBar(
    modifier: Modifier = Modifier,
    timerState: State<TimerState> = mutableStateOf(Stopped)
) = AnimatedContent(
    targetState = timerState.value,
    modifier = modifier.padding(bottom = 8.dp),
    label = "UpperBar",
    contentAlignment = Alignment.BottomStart,
    transitionSpec = {
        when {

            initialState is Session.Running && targetState is Stopped ->
                slideIntoContainer(Up, animationSpec = tween(500, 500)) togetherWith
                    slideOutOfContainer(Left, animationSpec = tween(300, easing = FastOutLinearInEasing))

            initialState is Stopped && targetState is Session.Running ->
                slideIntoContainer(Right, animationSpec = tween(500, 300)) togetherWith
                    slideOutOfContainer(Down, animationSpec = tween(300, easing = FastOutLinearInEasing))

            else -> fadeIn() togetherWith fadeOut()
        }
    }

) { state ->

    when (state) {
        is Session.Finished -> Spacer(Modifier.fillMaxWidth())
        Stopped -> UpperInputs()
        is Session.Running -> RemainingTimeHeader()
    }

}

@Composable
private fun RemainingTimeHeader(modifier: Modifier = Modifier) {
    Text(
        modifier = modifier.fillMaxWidth(),
        text = stringResource(id = R.string.remaining_time_header),
        textAlign = TextAlign.Left,
        fontSize = 20.sp,
    )
}

@Composable
private fun UpperInputs(
    modifier: Modifier = Modifier,
    viewModel: SetupViewModel = koinActivityViewModel()
) = Column(
    modifier = modifier
        .fillMaxWidth()
        .wrapContentHeight(),
    verticalArrangement = Arrangement.spacedBy(8.dp)
) {
    var isFirstAppearance by remember { mutableStateOf(true) }
    LaunchedEffect(true) {
        delay(1.seconds)
        isFirstAppearance = false
    }

    Row(
        horizontalArrangement = Arrangement.spacedBy(32.dp),
        verticalAlignment = Alignment.CenterVertically
    ) {
        NumberInput(
            value = viewModel[TimerInput.WORK_SESSION_DURATION],
            onValueChange = { viewModel[TimerInput.WORK_SESSION_DURATION] = it.trim() },
            modifier = Modifier.weight(1F),
            label = { Text(text = stringResource(id = R.string.work_session_duration_label)) },
            visualTransformation = SuffixTransformation(stringResource(id = R.string.unit_text)),
            accentColor = MaterialTheme.colorScheme.secondary
        )
        NumberInput(
            value = viewModel[TimerInput.SHORT_BREAK_REPETITIONS],
            onValueChange = { viewModel[TimerInput.SHORT_BREAK_REPETITIONS] = it.trim() },
            modifier = Modifier.weight(1F),
            label = { Text(text = stringResource(id = R.string.break_ratio_label)) },
            visualTransformation = SuffixTransformation(stringResource(id = R.string.break_ratio_suffix)),
            accentColor = MaterialTheme.colorScheme.secondary
        )
    }
}
