@file:OptIn(ExperimentalTime::class)

package com.mcsnowflake.worktimer.alarms

import android.app.AlarmManager
import android.app.PendingIntent
import android.content.Context
import android.content.Intent
import android.net.Uri
import android.os.Build
import android.util.Log
import com.mcsnowflake.worktimer.configuration.ConfigurationStore
import com.mcsnowflake.worktimer.configuration.Preference.HYDRATION_MODE
import com.mcsnowflake.worktimer.state.SessionData
import com.mcsnowflake.worktimer.state.TimerState
import com.mcsnowflake.worktimer.state.TimerState.Session.Finished
import com.mcsnowflake.worktimer.state.TimerState.Session.Running
import com.mcsnowflake.worktimer.state.TimerState.Stopped
import kotlin.time.Clock.System.now
import kotlin.time.Duration.Companion.minutes
import kotlin.time.Duration.Companion.seconds
import kotlin.time.ExperimentalTime
import kotlin.time.Instant
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.drop
import kotlinx.coroutines.launch

/**
 * Central handler for scheduling/canceling end-of-timer alarms based on timer state.
 */
class AlarmHandler(
    private val alarmManager: AlarmManager,
    configuration: ConfigurationStore,
    private val timerAlarmPendingIntent: PendingIntent,
    private val hydrationAlarmPendingIntent: PendingIntent,
    private val timerState: StateFlow<TimerState>,
    private val coroutineScope: CoroutineScope = CoroutineScope(Dispatchers.IO + SupervisorJob())
) {

    private val isHydrationModeOn = configuration.getAsStateFlow(HYDRATION_MODE, coroutineScope)

    init {
        coroutineScope.launch {
            isHydrationModeOn.collect {
                when (val state = timerState.value) {
                    is Running -> if (it) scheduleHydrationAlarms(state.session)
                    else -> {}
                }
            }
        }
        coroutineScope.launch {
            timerState.drop(1).collect { processState(it) }
        }
    }

    private fun processState(event: TimerState) {
        when (event) {
            is Running -> coroutineScope.launch {
                alarmManager.cancel(timerAlarmPendingIntent)
                alarmManager.cancel(hydrationAlarmPendingIntent)
                scheduleTimerAlarm(event.session.end)
                if (isHydrationModeOn.value) scheduleHydrationAlarms(event.session)
            }

            is Finished, is Stopped -> coroutineScope.launch {
                alarmManager.cancel(timerAlarmPendingIntent)
                alarmManager.cancel(hydrationAlarmPendingIntent)
            }
        }
    }

    private fun scheduleHydrationAlarms(session: SessionData) = when (session.type) {
        SessionData.Type.WORK_SESSION -> when {
            session.duration < 45.minutes -> scheduleHydrationAlarm(session.start + session.duration / 2)
            else -> {
                scheduleHydrationAlarm(session.start + 15.minutes)
                scheduleHydrationAlarm(session.end - 15.minutes)
            }
        }

        SessionData.Type.SHORT_BREAK, SessionData.Type.LONG_BREAK -> scheduleHydrationAlarm(session.start + 15.seconds)
    }

    private fun scheduleTimerAlarm(triggerTime: Instant) {
        Log.d("AlarmHandler", "Scheduling timer alarm for $triggerTime")
        scheduleAlarm(triggerTime, timerAlarmPendingIntent)
    }

    private fun scheduleHydrationAlarm(triggerTime: Instant) {
        if (triggerTime > now()) {
            Log.d("AlarmHandler", "Scheduling hydration alarm for $triggerTime")
            scheduleAlarm(triggerTime, hydrationAlarmPendingIntent)
        }
    }

    private fun scheduleAlarm(triggerTime: Instant, intent: PendingIntent) = runCatching {
        when {
            Build.VERSION.SDK_INT < 31 || alarmManager.canScheduleExactAlarms() -> {
                alarmManager.setExactAndAllowWhileIdle(AlarmManager.RTC_WAKEUP, triggerTime.toEpochMilliseconds(), intent)
            }

            else -> {
                Log.w("AlarmHandler", "Exact alarms not allowed; scheduling inexact alarm")
                alarmManager.setAndAllowWhileIdle(AlarmManager.RTC_WAKEUP, triggerTime.toEpochMilliseconds(), intent)
            }
        }
    }.onFailure {
        Log.e("AlarmHandler", "SecurityException when scheduling exact alarm; falling back to inexact")
        alarmManager.setAndAllowWhileIdle(AlarmManager.RTC_WAKEUP, triggerTime.toEpochMilliseconds(), intent)
    }

    companion object {

        private const val HYDRATION_ALARM_ID = 245363
        private const val TIMER_ALARM_ID = 249725

        enum class AlarmType {
            HYDRATION_ALARM, TIMER_ALARM
        }

        fun createAlarmPendingIntent(context: Context, alarmType: AlarmType): PendingIntent =
            PendingIntent.getBroadcast(
                context,
                when (alarmType) {
                    AlarmType.HYDRATION_ALARM -> HYDRATION_ALARM_ID
                    AlarmType.TIMER_ALARM -> TIMER_ALARM_ID
                },
                Intent(alarmType.name, Uri.EMPTY, context, AlarmReceiver::class.java),
                PendingIntent.FLAG_UPDATE_CURRENT or PendingIntent.FLAG_IMMUTABLE
            )
    }
}
