package com.mcsnowflake.worktimer

import android.app.AlarmManager
import android.app.NotificationManager
import android.content.Context
import android.media.AudioManager
import androidx.datastore.core.DataStore
import androidx.datastore.preferences.core.Preferences
import com.mcsnowflake.worktimer.alarms.AlarmHandler
import com.mcsnowflake.worktimer.alarms.AlarmHandler.Companion.AlarmType.HYDRATION_ALARM
import com.mcsnowflake.worktimer.alarms.AlarmHandler.Companion.AlarmType.TIMER_ALARM
import com.mcsnowflake.worktimer.configuration.ConfigurationStore
import com.mcsnowflake.worktimer.configuration.StateStore
import com.mcsnowflake.worktimer.configuration.StateStore.Companion.TIMER_STATE
import com.mcsnowflake.worktimer.notifications.DndModeHandler
import com.mcsnowflake.worktimer.notifications.NotificationBuilder
import com.mcsnowflake.worktimer.notifications.NotificationHandler
import com.mcsnowflake.worktimer.state.TimerEvent
import com.mcsnowflake.worktimer.state.TimerState
import com.mcsnowflake.worktimer.state.TimerStateMachine
import com.mcsnowflake.worktimer.ui.components.appbar.AppBarViewModel
import com.mcsnowflake.worktimer.ui.components.banners.BannersViewModel
import com.mcsnowflake.worktimer.ui.components.screenLock.KeepScreenOnViewModel
import com.mcsnowflake.worktimer.ui.dialogs.DndPermissionDialogViewState
import com.mcsnowflake.worktimer.ui.dialogs.LateAlarmsDialogViewState
import com.mcsnowflake.worktimer.ui.views.dashboard.DashBoardViewModel
import com.mcsnowflake.worktimer.ui.views.settings.SettingsViewModel
import com.mcsnowflake.worktimer.ui.views.dashboard.SetupViewModel
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.runBlocking
import kotlinx.serialization.json.Json
import org.koin.android.ext.koin.androidContext
import org.koin.core.module.dsl.singleOf
import org.koin.core.module.dsl.viewModel
import org.koin.core.module.dsl.viewModelOf
import org.koin.dsl.module

fun getModule(dataStore: DataStore<Preferences>) = module {

    factory { CoroutineScope(Dispatchers.IO + SupervisorJob()) }

    single { dataStore }
    singleOf(::ConfigurationStore)
    singleOf(::StateStore)

    // System services
    single { androidContext().getSystemService(Context.NOTIFICATION_SERVICE) as NotificationManager }
    single { androidContext().getSystemService(Context.AUDIO_SERVICE) as AudioManager }
    single { androidContext().getSystemService(Context.ALARM_SERVICE) as AlarmManager }

    single { NotificationBuilder(androidContext(), get()) }

    // state handling
    single { MutableSharedFlow<TimerEvent>(extraBufferCapacity = 10) }
    single {
        val loadedState: TimerState = runBlocking {
            get<StateStore>()[TIMER_STATE]
                ?.decodeToString()
                ?.let { Json.decodeFromString<TimerState>(it) }
                ?: TimerState.Stopped
        }
        TimerStateMachine(get(), get(), loadedState, get<MutableSharedFlow<TimerEvent>>())
    }
    single { get<TimerStateMachine>().state }

    // App handlers (custom)
    single(createdAtStart = true) {
        AlarmHandler(
            get(),
            get(),
            AlarmHandler.createAlarmPendingIntent(androidContext(), TIMER_ALARM),
            AlarmHandler.createAlarmPendingIntent(androidContext(), HYDRATION_ALARM),
            get()
        )
    }
    single(createdAtStart = true) { DndModeHandler(get(), get(), get()) }
    single(createdAtStart = true) { NotificationHandler(get(), get(), get(), get(), get()) }

    // ViewModels
    viewModel { DashBoardViewModel(get(), get<TimerStateMachine>().state) }
    viewModel { SettingsViewModel(get()) }
    viewModel { SetupViewModel(get()) }
    viewModel { AppBarViewModel() }
    viewModel { KeepScreenOnViewModel(get(), get()) }
    viewModelOf(::BannersViewModel)

    singleOf(::DndPermissionDialogViewState)
    singleOf(::LateAlarmsDialogViewState)
}
