package com.liskovsoft.youtubeapi.videoinfo.models;

import com.liskovsoft.sharedutils.helpers.Helpers;
import com.liskovsoft.sharedutils.querystringparser.UrlQueryString;
import com.liskovsoft.sharedutils.querystringparser.UrlQueryStringFactory;
import com.liskovsoft.googlecommon.common.converters.jsonpath.JsonPath;
import com.liskovsoft.googlecommon.common.models.V2.TextItem;

/**
 * YouTube subs not supported by ExoPlayer?<br/>
 * https://js-jrod.medium.com/the-first-complete-guide-to-youtube-captions-f886e06f7d9d<br/>
 * https://en.wikipedia.org/wiki/Timed_Text_Markup_Language<br/>
 */
public class CaptionTrack {
    public enum CaptionFormat {
        TTML("ttml", "stpp", "application/ttml+xml"),
        VTT("vtt", "wvtt", "text/vtt");

        public final String name;
        public final String codecs;
        public final String mimeType;

        CaptionFormat(String name, String codecs, String mimeType) {
            this.name = name;
            this.codecs = codecs;
            this.mimeType = mimeType;
        }
    }
    
    private static final String TYPE_ASR = "asr";
    public static CaptionFormat sFormat = CaptionFormat.TTML;

    /**
     * Example: "https://www.youtube.com/api/timedtext?caps=&key=yt…&sparams=caps%2Cv%2Cxorp%2Cexpire&lang=en&name=en"
     */
    @JsonPath("$.baseUrl")
    private String mBaseUrl;
    /**
     * Example: true
     */
    @JsonPath("$.isTranslatable")
    private boolean mIsTranslatable;
    /**
     * Example: "en"
     */
    @JsonPath("$.languageCode")
    private String mLanguageCode;
    /**
     * Example: ".en.nP7-2PuUl7o"
     */
    @JsonPath("$.vssId")
    private String mVssId;
    @JsonPath("$.name")
    private TextItem mName;
    /**
     * E.g. asr (Automatic Speech Recognition)
     */
    @JsonPath("$.kind")
    private String mType;
    private UrlQueryString mBaseUrlQuery;

    public String getBaseUrl() {
        UrlQueryString baseUrlQuery = getBaseUrlQuery();

        if (baseUrlQuery == null) {
            return null;
        }

        baseUrlQuery.set("fmt", sFormat.name);
        
        return baseUrlQuery.toString();
    }

    public boolean isTranslatable() {
        return mIsTranslatable;
    }

    public String getLanguageCode() {
        return mLanguageCode;
    }

    public String getVssId() {
        return mVssId;
    }

    public String getName() {
        String name = mName != null ? Helpers.toString(mName.getText()) : null;
        return isAutogenerated() && name != null ? name + TranslatedCaptionTrack.TRANSLATE_MARKER : name;
    }

    public String getType() {
        return mType;
    }

    public String getMimeType() {
        return sFormat.mimeType;
    }

    public String getCodecs() {
        return sFormat.codecs;
    }

    public boolean isAutogenerated() {
        return TYPE_ASR.equals(mType);
    }

    private UrlQueryString getBaseUrlQuery() {
        if (mBaseUrl == null) {
            return null;
        }

        if (mBaseUrl.startsWith("/")) { // Fix relative urls (mweb client)
            mBaseUrl = "https://www.youtube.com" + mBaseUrl;
        }

        if (mBaseUrlQuery == null) {
            mBaseUrlQuery = UrlQueryStringFactory.parse(mBaseUrl);
        }

        return mBaseUrlQuery;
    }
}
