"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getOrCreateWebSocketClient = getOrCreateWebSocketClient;
var _reactNative = require("react-native");
var _isURL = _interopRequireDefault(require("validator/es/lib/isURL"));
var _NativeWebSocketClient = _interopRequireWildcard(require("./NativeWebSocketClient"));
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
// Copyright (c) 2015-present Mattermost, Inc. All Rights Reserved.
// See LICENSE.txt for license information.

const Emitter = new _reactNative.NativeEventEmitter(_NativeWebSocketClient.default);
const CLIENTS = {};
const CREATING_CLIENT = {};

/**
 * Configurable WebSocket client
 */
class WebSocketClient {
  constructor(url) {
    this.url = url;
    this.readyState = _NativeWebSocketClient.WebSocketReadyState.CLOSED;
    this.onReadyStateSubscription = Emitter.addListener(_NativeWebSocketClient.WebSocketEvents.READY_STATE_EVENT, event => {
      if (event.url === this.url) {
        this.readyState = event.message;
      }
    });
  }
  open = () => _NativeWebSocketClient.default.connectFor(this.url);
  close = () => {
    this.readyState = _NativeWebSocketClient.WebSocketReadyState.CLOSED;
    return _NativeWebSocketClient.default.disconnectFor(this.url);
  };
  send = data => _NativeWebSocketClient.default.sendDataFor(this.url, data);
  onOpen = callback => {
    if (this.onWebSocketOpenSubscription) {
      this.onWebSocketOpenSubscription.remove();
    }
    this.onWebSocketOpenSubscription = Emitter.addListener(_NativeWebSocketClient.WebSocketEvents.OPEN_EVENT, event => {
      if (event.url === this.url && callback) {
        callback(event);
      }
    });
  };
  onClose = callback => {
    if (this.onWebSocketCloseSubscription) {
      this.onWebSocketCloseSubscription.remove();
    }
    this.onWebSocketCloseSubscription = Emitter.addListener(_NativeWebSocketClient.WebSocketEvents.CLOSE_EVENT, event => {
      if (event.url === this.url && callback) {
        callback(event);
      }
    });
  };
  onError = callback => {
    if (this.onWebSocketErrorSubscription) {
      this.onWebSocketErrorSubscription.remove();
    }
    this.onWebSocketErrorSubscription = Emitter.addListener(_NativeWebSocketClient.WebSocketEvents.ERROR_EVENT, event => {
      if (event.url === this.url && callback) {
        callback(event);
      }
    });
  };
  onMessage = callback => {
    if (this.onWebSocketMessageSubscription) {
      this.onWebSocketMessageSubscription.remove();
    }
    this.onWebSocketMessageSubscription = Emitter.addListener(_NativeWebSocketClient.WebSocketEvents.MESSAGE_EVENT, event => {
      if (event.url === this.url && callback) {
        callback(event);
      }
    });
  };
  onClientError = callback => {
    if (this.onWebSocketClientErrorSubscription) {
      this.onWebSocketClientErrorSubscription.remove();
    }
    this.onWebSocketClientErrorSubscription = Emitter.addListener(_NativeWebSocketClient.WebSocketEvents.ERROR_EVENT, event => {
      if (event.url === this.url) {
        callback(event);
      }
    });
  };
  invalidate = () => {
    this.onReadyStateSubscription.remove();
    this.onWebSocketOpenSubscription?.remove();
    this.onWebSocketCloseSubscription?.remove();
    this.onWebSocketErrorSubscription?.remove();
    this.onWebSocketMessageSubscription?.remove();
    this.onWebSocketClientErrorSubscription?.remove();
    delete CLIENTS[this.url];
    return _NativeWebSocketClient.default.invalidateClientFor(this.url);
  };
}
async function getOrCreateWebSocketClient(url, config = {}, clientErrorEventHandler) {
  if (!isValidWebSocketURL(url)) {
    throw new Error(`"${url}" is not a valid WebSocket URL`);
  }
  let created = false;
  let client = CLIENTS[url];
  if (!client) {
    if (CREATING_CLIENT[url]) {
      throw new Error(`Already creating a client for url "${url}"`);
    }
    CREATING_CLIENT[url] = true;
    created = true;
    client = new WebSocketClient(url);
    if (clientErrorEventHandler) {
      client.onClientError(clientErrorEventHandler);
    }
    await _NativeWebSocketClient.default.ensureClientFor(url, config);
    CLIENTS[url] = client;
    delete CREATING_CLIENT[url];
  }
  return {
    client,
    created
  };
}
const isValidWebSocketURL = url => {
  return (0, _isURL.default)(url, {
    protocols: ["ws", "wss"],
    require_protocol: true,
    require_valid_protocol: true,
    require_host: true
  });
};
//# sourceMappingURL=index.js.map