Object.defineProperty(exports, "__esModule", { value: true });
exports.createSet = exports.determineDebugIdFromBundleSource = exports.stringToUUID = exports.createDebugIdSnippet = void 0;
const crypto = require("crypto");
/**
 * Returns minified Debug ID code snippet.
 */
function createDebugIdSnippet(debugId) {
    return `var _sentryDebugIds,_sentryDebugIdIdentifier;void 0===_sentryDebugIds&&(_sentryDebugIds={});try{var stack=(new Error).stack;stack&&(_sentryDebugIds[stack]="${debugId}",_sentryDebugIdIdentifier="sentry-dbid-${debugId}")}catch(e){}`;
}
exports.createDebugIdSnippet = createDebugIdSnippet;
/**
 * Deterministically hashes a string and turns the hash into a uuid.
 *
 * https://github.com/getsentry/sentry-javascript-bundler-plugins/blob/58271f1af2ade6b3e64d393d70376ae53bc5bd2f/packages/bundler-plugin-core/src/utils.ts#L174
 */
function stringToUUID(str) {
    const md5sum = crypto.createHash('md5');
    md5sum.update(str);
    const md5Hash = md5sum.digest('hex');
    // Position 16 is fixed to either 8, 9, a, or b in the uuid v4 spec (10xx in binary)
    // RFC 4122 section 4.4
    const v4variant = ['8', '9', 'a', 'b'][md5Hash.substring(16, 17).charCodeAt(0) % 4];
    return `${md5Hash.substring(0, 8)}-${md5Hash.substring(8, 12)}-4${md5Hash.substring(13, 16)}-${v4variant}${md5Hash.substring(17, 20)}-${md5Hash.substring(20)}`.toLowerCase();
}
exports.stringToUUID = stringToUUID;
/**
 * Looks for a particular string pattern (`sdbid-[debug ID]`) in the bundle
 * source and extracts the bundle's debug ID from it.
 *
 * The string pattern is injected via the debug ID injection snipped.
 *
 * https://github.com/getsentry/sentry-javascript-bundler-plugins/blob/40f918458ed449d8b3eabaf64d13c08218213f65/packages/bundler-plugin-core/src/debug-id-upload.ts#L293-L294
 */
function determineDebugIdFromBundleSource(code) {
    const match = code.match(/sentry-dbid-([0-9a-fA-F]{8}\b-[0-9a-fA-F]{4}\b-[0-9a-fA-F]{4}\b-[0-9a-fA-F]{4}\b-[0-9a-fA-F]{12})/);
    return match ? match[1] : undefined;
}
exports.determineDebugIdFromBundleSource = determineDebugIdFromBundleSource;
/**
 * CountingSet was added in Metro 0.72.0 before that NodeJS Set was used.
 *
 * https://github.com/facebook/metro/blob/fc29a1177f883144674cf85a813b58567f69d545/packages/metro/src/lib/CountingSet.js
 */
function resolveSetCreator() {
    try {
        // eslint-disable-next-line @typescript-eslint/no-var-requires, import/no-extraneous-dependencies
        const { default: MetroSet } = require('metro/src/lib/CountingSet');
        return () => new MetroSet();
    }
    catch (e) {
        if (e instanceof Error && 'code' in e && e.code === 'MODULE_NOT_FOUND') {
            return () => new Set();
        }
        else {
            throw e;
        }
    }
}
exports.createSet = resolveSetCreator();
//# sourceMappingURL=utils.js.map